/*
 * @(#)Platform.java   1.0 06/08/25
 */// ************************************************************************
// * Copyright (c) 2006 by the Association for Computing Machinery        *
// *                                                                      *
// * The Java Task Force seeks to impose few restrictions on the use of   *
// * these packages so that users have as much freedom as possible to     *
// * use this software in constructive ways and can make the benefits of  *
// * that work available to others.  In view of the legal complexities    *
// * of software development, however, it is essential for the ACM to     *
// * maintain its copyright to guard against attempts by others to        *
// * claim ownership rights.  The full text of the JTF Software License   *
// * is available at the following URL:                                   *
// *                                                                      *
// *          http://www.acm.org/jtf/jtf-software-license.pdf             *
// *                                                                      *
// ************************************************************************
/**
 * Class extracted from ACM JTF acm.util package (http://jtf.acm.org/). 
 * Modified slightly to support FANG
 */
package fang2.util;

import java.awt.*;
import java.io.*;
import java.lang.reflect.*;
import java.util.*;

/* Class: Platform */
/**
 * This class contains methods to support platform-specific code.
 */
public class Platform {

    /* Constant: UNKNOWN */
    /** Indicates that the type of system cannot be determined. */
    public static final int UNKNOWN = 0;

    /* Constant: MAC */
    /** Indicates that the system is some variety of Apple Macintosh. */
    public static final int MAC = 1;

    /* Constant: UNIX */
    /** Indicates that the system is some variety of Unix or Linux. */
    public static final int UNIX = 2;

    /* Constant: WINDOWS */
    /** Indicates that the system is some variety of Microsoft Windows. */
    public static final int WINDOWS = 3;

    /* Static method: getPlatform() */
    /**
     * Returns an enumeration constant specifying the type of platform
     * on which this applet is running, which is one of the supported
     * types defined at the beginning of this class.
     *
     * @return A constant specifying the platform type
     */
    public static int getPlatform() {
        if (platform != -1) {
            return platform;
        }
        String name = System.getProperty("os.name", "").toLowerCase();
        if (name.startsWith("mac")) {
            return platform = MAC;
        }
        if (name.startsWith("windows")) {
            return platform = WINDOWS;
        }
        if (name.startsWith("microsoft")) {
            return platform = WINDOWS;
        }
        if (name.startsWith("ms")) {
            return platform = WINDOWS;
        }
        if (name.startsWith("unix")) {
            return platform = UNIX;
        }
        if (name.startsWith("linux")) {
            return platform = UNIX;
        }
        return platform = UNKNOWN;
    }

    /* Static method: isMac */
    /**
     * Checks whether the platform is a Macintosh.
     *
\     * @return <code>true</code> if the platform is a Macintosh, <code>false</code> otherwise
     */
    public static boolean isMac() {
        return getPlatform() == MAC;
    }

    /* Static method: isWindows() */
    /**
     * Checks whether the platform is a Windows machine.
     *
\     * @return <code>true</code> if the platform is a Windows machine, <code>false</code> otherwise
     */
    public static boolean isWindows() {
        return getPlatform() == WINDOWS;
    }

    /* Static method: isUnix() */
    /**
     * Checks whether the platform is Unix.
     *
 \    * @return <code>true</code> if the platform is Unix, <code>false</code> otherwise
     */
    public static boolean isUnix() {
        return getPlatform() == UNIX;
    }

    /* Static method: setFileTypeAndCreator(filename, type, creator) */
    /**
     * Sets the Macintosh file type and creator.  This method is ignored on non-Mac
     * platforms.
     *
     * @param filename The name of the file
     * @param type A four-character string indicating the file type
     * @param creator A four-character string indicating the file type
     */
    public static void setFileTypeAndCreator(String filename, String type, String creator) {
        if (!isMac()) {
            return;
        }
        try {
            setFileTypeAndCreator(new File(filename), type, creator);
        } catch (Exception ex) {
            /* Empty */
        }
    }

    /* Static method: setFileTypeAndCreator(file, type, creator) */
    /**
     * Sets the Macintosh file type and creator.  This method is ignored on non-Mac
     * platforms.
     *
     * @param file The <code>File</code> object corresponding to the file
     * @param type A four-character string indicating the file type
     * @param creator A four-character string indicating the file type
     */
    @SuppressWarnings("unchecked")
    public static void setFileTypeAndCreator(File file, String type, String creator) {
        if (!isMac()) {
            return;
        }
        try {
            Class mrjOSTypeClass = Class.forName("com.apple.mrj.MRJOSType");
            Class mrjFileUtilsClass = Class.forName("com.apple.mrj.MRJFileUtils");
            Class[] sig1 = {Class.forName("java.lang.String")};
            Constructor constructor = mrjOSTypeClass.getConstructor(sig1);
            Class[] sig2 = {Class.forName("java.io.File"), mrjOSTypeClass, mrjOSTypeClass};
            Method fn = mrjFileUtilsClass.getMethod("setFileTypeAndCreator", sig2);
            Object[] args1 = {(type + "    ").substring(0, 4)};
            Object osType = constructor.newInstance(args1);
            Object[] args2 = {(creator + "    ").substring(0, 4)};
            Object creatorType = constructor.newInstance(args2);
            Object[] args3 = {file, osType, creatorType};
            fn.invoke(null, args3);
        } catch (Exception ex) {
            /* Empty */
        } finally {
        }
    }

    /* Static method: compareVersion(version) */
    /**
     * This method compares the Java version given in the system properties
     * with the specified version and returns -1, 0, or +1 depending on whether
     * the system version is earlier than, equal to, or later than the specified
     * one.  Thus, to test whether the current version of the JDK was at least
     * 1.2.1, for example, you could write
     *
     * <p><table cellspacing=0 cellpadding=0><tr><td width=36>&nbsp;</td>
     * <td><table cellspacing=0 cellpadding=0><tr><td><pre><font face=courier size=-1><b>
     *     if (Platform.compareVersion("1.2.1") &gt;= 0) . . .
     * </b></font></pre></td></tr></table></td></tr></table>
     *
     * @param version A string consisting of integers separated by periods
     * @return -1, 0, or +1 depending on whether the system version is earlier than,
     *         equal to, or later than the specified one
     */
    public static int compareVersion(String version) {
        return compareVersion(System.getProperty("java.version"), version);
    }

    /* Static method: compareVersion(v1, v2) */
    /**
     * This method compares the version strings <code>v1</code> and <code>v2</code>
     * and returns -1, 0, or +1 depending on whether <code>v1</code> is earlier
     * than, equal to, or later than <code>v2</code>.
     *
     * @param v1 A string consisting of integers separated by periods
     * @param v2 A second version string in the same format
     * @return -1, 0, or +1 depending on whether <code>v1</code> is earlier than,
     *         equal to, or later than <code>v2</code>
     */
    public static int compareVersion(String v1, String v2) {
        StringTokenizer t1 = new StringTokenizer(v1, ".");
        StringTokenizer t2 = new StringTokenizer(v2, ".");
        while (t1.hasMoreTokens() && t2.hasMoreTokens()) {
            int n1 = Integer.parseInt(t1.nextToken());
            int n2 = Integer.parseInt(t2.nextToken());
            if (n1 != n2) {
                return (n1 < n2) ? -1 : +1;
            }
        }
        if (t1.hasMoreTokens()) {
            return +1;
        }
        if (t2.hasMoreTokens()) {
            return -1;
        }
        return 0;
    }

    /* Static method: isSwingAvailable() */
    /**
     * Checks whether Swing is available.  Unfortunately, some browsers seem to lie
     * about the JDK version and return a 1.2 number without actually having Swing.
     * This implementation tests the version first, but then confirms the result
     * by looking for the <code>JComponent</code> class.  Checking the version first
     * means that no <nobr><code>SecurityException</code>s</nobr> will be logged in
     * Windows machines, which always log <nobr><code>SecurityException</code>s</nobr>,
     * even if the exception is caught.
     *
     * @return <code>true</code> if Swing is available, <code>false</code> otherwise
     */
    public static boolean isSwingAvailable() {
        if (!swingChecked) {
            swingChecked = true;
            isSwingAvailable = false;
            if (compareVersion("1.2") >= 0) {
                try {
                    isSwingAvailable = Class.forName("javax.swing.JComponent") != null;
                } catch (Exception ex) {
                    /* Empty */
                }
            }
        }
        return isSwingAvailable;
    }

    /* Static method: isSunAudioAvailable() */
    /**
     * Checks whether the <code>sun.audio</code> package is available.
     *
     * @return <code>true</code> if the <code>sun.audio</code> package is available,
     *         <code>false</code> otherwise
     */
    public static boolean isSunAudioAvailable() {
        if (!sunAudioChecked) {
            sunAudioChecked = true;
            try {
                isSunAudioAvailable = Class.forName("sun.audio.AudioPlayer") != null;
            } catch (Exception ex) {
                isSunAudioAvailable = false;
            }
        }
        return isSunAudioAvailable;
    }

    /* Static method: isJMFAvailable() */
    /**
     * Checks whether the Java Media Framework is available.
     *
     * @return <code>true</code> if the JMF package is available, <code>false</code> otherwise
     */
    public static boolean isJMFAvailable() {
        if (!jmfChecked) {
            jmfChecked = true;
            try {
                isJMFAvailable = Class.forName("javax.media.Player") != null;
            } catch (Exception ex) {
                isJMFAvailable = false;
            }
        }
        return isJMFAvailable;
    }

    /* Static method: areCollectionsAvailable() */
    /**
     * Checks whether the JDK 1.2 collection classes are available.  Some browsers
     * return a version of the JDK that does not actually match what is supported.
     * This method actually checks whether the collection classes are there by
     * looking for the <code>ArrayList</code> class.
     *
     * @return <code>true</code> if collections are available, <code>false</code> otherwise
     */
    public static boolean areCollectionsAvailable() {
        if (!collectionsChecked) {
            collectionsChecked = true;
            try {
                areCollectionsAvailable = Class.forName("java.util.ArrayList") != null;
            } catch (Exception ex) {
                areCollectionsAvailable = false;
            }
        }
        return areCollectionsAvailable;
    }

    /* Static method: areStandardFontFamiliesAvailable() */
    /**
     * Checks whether the JDK 1.2 standard font families (<code>Serif</code>,
     * <code>SansSerif</code>, and <code>Monospaced</code>) are available.
     *
     * @return <code>true</code> if the standard fonts are available, <code>false</code> otherwise
     */
    public static boolean areStandardFontFamiliesAvailable() {
        if (!fontsChecked) {
            fontsChecked = true;
            try {
                Class<?> toolkitClass = Class.forName("java.awt.Toolkit");
                Method getFontList = toolkitClass.getMethod("getFontList", new Class[0]);
                String[] fonts = (String[]) getFontList.invoke(Toolkit.getDefaultToolkit(), new Object[0]);
                int standardFontCount = 0;
                for (int i = 0; i < fonts.length; i++) {
                    if (fonts[i].equals("Serif") || fonts[i].equals("SansSerif") || fonts[i].equals("Monospaced")) {
                        standardFontCount++;
                    }
                }
                areStandardFontFamiliesAvailable = (standardFontCount == 3);
            } catch (Exception ex) {
                areStandardFontFamiliesAvailable = false;
            }
        }
        return areStandardFontFamiliesAvailable;
    }

    /* Private static variables */
    private static int platform = -1;
    private static boolean areStandardFontFamiliesAvailable;
    private static boolean fontsChecked;
    private static boolean isSwingAvailable;
    private static boolean swingChecked;
    private static boolean areCollectionsAvailable;
    private static boolean collectionsChecked;
    private static boolean isSunAudioAvailable;
    private static boolean sunAudioChecked;
    private static boolean isJMFAvailable;
    private static boolean jmfChecked;
}
