/*
 * This class contains code extracted from the ACM JTF acm.program
 * package. It is designed to permit a running Java program to determine
 * the actual class with which Java was invoked.
 *
 * After determining the Game class, fangMain instantiates it and then
 * calls the runAsApplication method on it.
 */

package fang2.util;

import fang2.core.GameLoop;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StreamTokenizer;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * InitializeApplication is a static class which provides the fangMain
 * method.
 *
 * @author  ACM Java Task Force
 * @author  blad
 */
public class InitializeApplication {
  /** name in dictionary for unnamed arguments */
  public static final String ARGS = "UNAMED_ARGS";

  /** holds the dictionary for the command-line arguments */
  private static Map<String, ArrayList<String>> argDictionary = 
	  new HashMap<String, ArrayList<String>>();

  /**
   * Every application must either contain a "Main-Class" entry in its
   * manifest file or include a main method that looks like this, where
   * <code>MyClass</code> is the name of the program class: <code>
   * <p>&nbsp;&nbsp;&nbsp;public static void main(String[] args)<br>
   * &nbsp;&nbsp;&nbsp;{<br>
   * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;new MyClass().start();<br>
   * &nbsp;&nbsp;&nbsp;}<br>
   * </code>
   *
   * <p>If the program needs the command line arguments, the <code>
   * args</code> array can be passed to the <code>start</code> method
   * and then retrieved using the <code>getArgumentArray</code> method.
   *
   * @param  args  An array of string arguments
   */
  public static void fangMain(String[] args) {
	argDictionary.putAll(parameterDictionary(args));
    String className = null;
    // if a "game=..." parameter was supplied, use that for the name of
    // the class to instantiate
    if (argDictionary.containsKey("game")) {
      className = argDictionary.get("game").get(0);
    }

    Class<?> mainClass = null;

    GameLoop program = null;

    if (className == null) {
      try {
        className = System.getProperty("java.main");
      } catch (Exception ex) {
        /* Empty */
      }
    }

    // if className is still null, get the full command-line used for
    // Java and find the class name there
    if (className == null) {
      String commandLine = getCommandLine();
      className = readMainClassFromCommandLine(commandLine);
    }

    if (className != null) {
      if (className.endsWith(".class")) {
        className = className.substring(0, className.length() - 6);
      }
      className = className.replace('/', '.');
      try {
        mainClass = Class.forName(className);
      } catch (ClassNotFoundException ex) {
        /* Empty */
      }
    }

    // System.out.println("mainClass = " + mainClass);

    if (mainClass != null) {
      try {
        Object obj = mainClass.newInstance();

        if (obj instanceof GameLoop) {
          program = (GameLoop) obj;
        } else {
          className = null;
          if (argDictionary.containsKey("game")) {
            className = argDictionary.get("game").get(0);
          }

          if (className == null) {
            throw new ErrorException(
              "Main class does not specify a game");
          }
          program = (GameLoop) Class.forName(className).newInstance();
        }
      } catch (IllegalAccessException ex) {
        /* Empty */
      } catch (InstantiationException ex) {
        /* Empty */
      } catch (ClassNotFoundException ex) {
        /* Empty */
      }
    }

    if (program == null) {
      throw new ErrorException("Cannot determine the game class.");
    }
    program.runAsApplication();
  }

  /**
   * List all names used for named arguments passed in to this program
   * when it was run
   *
   * @return  {@code ArrayList} of command-line argument names
   */
  public static ArrayList<String> getArgumentNames() {
    ArrayList<String> names = new ArrayList<String>();
    for (String key : argDictionary.keySet()) {
      if (!key.equals(ARGS)) {
        names.add(key);
      }
    }
    return names;
  }

  /**
   * Get the named argument's value(s)
   *
   * @param   name  name of arguments to return
   *
   * @return  {@code ArrayList} of command-line arguments assigned to
   *          the named value
   */
  public static ArrayList<String> getNamedArgs(String name) {
    return argDictionary.get(name);
  }

  /**
   * Get the collected unnamed arguments
   *
   * @return  {@code ArrayList} of command-line arguments
   */
  public static ArrayList<String> getUnnamedArgs() {
    return argDictionary.get(ARGS);
  }

  /**
   * Attempts to return the command line for Unix systems and MacOS X.
   * This code is adapted from a more general command-line scanner
   * written by Erik Forslin.
   *
   * @return  The shell command line that invoked this process, or
   *          <code>null</code> if no command line is available
   */
  private static String getShellCommandLine() {
    try {
      String option = (Platform.isMac()) ? "command" : "args";
      // bcl - Added the ww (unlimited width) specification here;
      // problems in emacs subshells otherwise
      String[] argv = { "bash", "-c", "ps ww -p $PPID -o " + option };
      Process p = Runtime.getRuntime().exec(argv);
      p.waitFor();
      if (p.getErrorStream().read() != -1) {
        return null;
      }
      BufferedReader rd = new BufferedReader(new InputStreamReader(
            p.getInputStream()));
      rd.readLine();/* Ignore header */
      return rd.readLine();
    } catch (Exception ex) {
      return null;
    }
  }

  /**
   * Attempts to read the name of the main class from the manifest of
   * the specified JAR file.
   *
   * @param   jarName  The name of the JAR file
   *
   * @return  The name of the main class, or <code>null</code>
   */
  private static String readMainClassFromManifest(String jarName) {
    try {
      ZipFile jarFile = new ZipFile(jarName);
      ZipEntry entry = jarFile.getEntry("META-INF/MANIFEST.MF");
      if (entry == null) {
        return null;
      }
      BufferedReader rd = new BufferedReader(new InputStreamReader(
            jarFile.getInputStream(entry)));
      for (String line = rd.readLine(); line != null;
          line = rd.readLine()) {
        if (line.startsWith("Main-Class:")) {
          return line.substring("Main-Class:".length()).trim();
        }
      }
      return null;
    } catch (IOException ex) {
      return null;
    }
  }

  /* Protected static method: getCommandLine() */
  /**
   * Returns the command line that invoked this program, or <code>
   * null</code> if no command line is available.
   *
   * @return  The command line that invoked this program
   */
  protected static String getCommandLine() {
    switch (Platform.getPlatform()) {
    case Platform.UNIX:
    case Platform.MAC:
      return getShellCommandLine();

    case Platform.WINDOWS:
      return DOSCommandLine.getCommandLine();

    default:
      return null;
    }
  }

  /**
   * Creates a dictionary of parameters. All parameters of the form
   *
   * <p><i>name</i><code>=</code><i>value</i><br />
   * will be stored in the dictionary such that the entry "name" has the
   * value "value". The name is converted to lowercase.
   *
   * <p>All other arguments are collected under the name <code>
   * ARGS</code>. A
   *
   * @param   args  The array of strings passed to the application
   *
   * @return  A <code>HashMap</code> containing the parameter bindings
   */
  protected static Map<String, ArrayList<String>> parameterDictionary(
    String[] args) {
    if (args == null) {
      return null;
    }

    Map<String, ArrayList<String>> ht =
      new HashMap<String, ArrayList<String>>();
    ArrayList<String> newArgs = new ArrayList<String>();
    for (int i = 0; i < args.length; i++) {
      String arg = args[i];
      int equals = arg.indexOf('=');
      if (equals > 0) {
        String name = arg.substring(0, equals).toLowerCase();
        String value = arg.substring(equals + 1);
        if (!ht.containsKey(name)) {
          ht.put(name, new ArrayList<String>());
        }
        ht.get(name).add(value);
      } else {
        newArgs.add(arg);
      }
    }

    ht.put(ARGS, newArgs);
    return ht;
  }

  /**
   * Attempts to read the name of the main class from the specified
   * command line. This strategy is a heuristic and will probably fail
   * in many cases, but it will probably work in enough contexts to be
   * useful. As noted in the documentation for the <code>main</code>
   * method, programs can always avoid the need for this method by
   * supplying their own version of <code>main</code>.
   *
   * @param   line  The command line
   *
   * @return  The name of the main class, or <code>null</code>
   */
  protected static String readMainClassFromCommandLine(String line) {
    if (line == null) {
      return null;
    }

    boolean jarFlag = false;
    try {
      StreamTokenizer tokenizer = new StreamTokenizer(new StringReader(
            line));
      tokenizer.resetSyntax();
      tokenizer.wordChars(33, 255);
      tokenizer.quoteChar('"');
      tokenizer.quoteChar('\'');
      tokenizer.whitespaceChars(' ', ' ');
      tokenizer.whitespaceChars('\t', '\t');
      boolean cmdRead = false;
      while (true) {
        int tt = tokenizer.nextToken();
        String token = tokenizer.sval;

        switch (tt) {
        case StreamTokenizer.TT_EOF:
          return null;

        case StreamTokenizer.TT_WORD:
        case '"':
        case '\'':
          break;

        default:
          return null;
        }

        if (cmdRead) {
          if (token.startsWith("-")) {
            if (token.equals("-jar")) {
              jarFlag = true;
            } else if (token.equals("-cp") ||
                token.equals("-classpath")) {
              tokenizer.nextToken();
            }
          } else {
            if (jarFlag) {
              return readMainClassFromManifest(token);
            }
            return token;
          }
        } else {
          cmdRead = true;
        }
      }
    } catch (IOException ex) {
      /* Empty */
    }
    return null;
  }

  public static void setParameter(String key, String value)
  {
	if(!argDictionary.containsKey(key))
		argDictionary.put(key, new ArrayList<String>());
	argDictionary.get(key).add(value);
  }
}
