package fang2.ui;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URL;
import java.util.LinkedList;
import java.util.regex.Pattern;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.html.StyleSheet;

import fang2.core.*;

/**
 * Displays runtime errors in a meaningful way.
 * 
 * @author Jam Jenkins
 */
public class ErrorConsole
  extends JDialog
  implements ActionListener {
  /**
   * used for serialization versioning
   */
  private static final long serialVersionUID = 1L;

  /** the stylesheet used to display the html */
  private static final URL STYLE_SHEET =
      ErrorConsole.class.getResource("resources/stylesheet.css");

  /** where the html is displayed */
  private JTextPane message;

  /** default size of the window */
  private static final Dimension DEFAULT_SIZE = new Dimension(600, 600);

  /** close/next button */
  private JButton closeButton;

  /** only one error console is needed, this is the only one constructed */
  private static final ErrorConsole single = new ErrorConsole();

  /** list of all of errors that have occurred */
  private final LinkedList<String> errors = new LinkedList<String>();
  
  /**
   * makes the error window, but does not set it visible
   */
  private ErrorConsole() {
    super();
    setTitle("Runtime Errors");
    makeComponents();
    makeLayout();
    setSize(DEFAULT_SIZE);
  }
  
  public static void clear()
  {
	  if(single!=null && single.errors!=null) single.errors.clear();
  }
  
  /**
   * advances to the next error and makes this gui invisible when there
   * are no more errors
   * 
   * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
   */
  public void actionPerformed(ActionEvent e) {
    if (errors.size() == 0) {
      setVisible(false);
    } else if (errors.size() == 1) {
      errors.removeFirst();
      setVisible(false);
    } else if (errors.size() > 1) {
      errors.removeFirst();
      message.setText(errors.getFirst());
      message.setCaretPosition(0);
      if (errors.size() == 1) {
        closeButton.setText("Close Error Console");
      }
    }
  }

  /** make the message pane and set its contents */
  private void makeComponents() {
    message = new JTextPane();
    closeButton = new FunButton("Close Error Console", DEFAULT_SIZE);
    closeButton.addActionListener(this);
    HTMLEditorKit kit = new HTMLEditorKit();
    StyleSheet style = new StyleSheet();
    style.importStyleSheet(STYLE_SHEET);
    kit.setStyleSheet(style);
    message.setEditorKit(kit);
    message.setEditable(false);
  }

  /** place the message pane in the window */
  private void makeLayout() {
    Container container = getContentPane();
    container.setLayout(new BorderLayout());
    container.add(new JScrollPane(message), BorderLayout.CENTER);
    container.add(closeButton, BorderLayout.SOUTH);
  }
  
  /**
   * A URL for the source could be in any of a variety of places.
   * This method checks the common ones until it can hopefully find it.
   * @param fileName the source to look for
   * @return a URL pointing toward the source, or null if none can be found
   */
  private static URL getSourceURL(String fileName)
  {
	  String[] prefixes =
		{ "", "../", "../src/", "src/" };
		for (String prefix : prefixes)
		{
			URL url;
			try
			{
				// try to use the applet's codebase
				if (Game.getCurrentGame().getCodeBase() != null)
				{
					//System.out.println("Trying from codebase: "+ "/" + prefix + fileName);
					url = new URL(Game.getCurrentGame().getCodeBase()
							.toString()
							+ "/" + prefix + fileName);
					if (url != null && url.openStream() != null)
						return url;

				}
			} catch (Exception e)
			{
			}
			// try to use the game engine folder
			try
			{
				String thisClassName = ErrorConsole.class.getCanonicalName();
				int numPackages = thisClassName.split(Pattern.quote(".")).length - 1;
				String dirPrefix = "";
				for (int i = 0; i < numPackages; i++)
				{
					dirPrefix += "../";
				}
				//System.out.println("trying from game engine folder: "+dirPrefix + prefix
				//		+ fileName);
				url = ErrorConsole.class.getResource(dirPrefix + prefix
						+ fileName);
				if (url != null && url.openStream() != null)
					return url;
			} catch (Exception e)
			{
			}
			// try to use the game folder
			try
			{
				String thisClassName = Game.getCurrentGame().getClass()
						.getCanonicalName();
				int numPackages = thisClassName.split(Pattern.quote(".")).length - 1;
				String dirPrefix = "";
				for (int i = 0; i < numPackages; i++)
				{
					dirPrefix += "../";
				}
				//System.out.println("trying from game folder: "+dirPrefix + prefix
				//		+ fileName);
				url = Game.getCurrentGame().getClass().getResource(dirPrefix + prefix
						+ fileName);
				if (url != null && url.openStream() != null)
					return url;
			} catch (Exception e)
			{
			}
			//try relative URL
			try
			{
				url=new URL(prefix+fileName);
				if (url != null && url.openStream() != null)
					return url;
			}
			catch(Exception e)
			{
				
			}
			//try relative File
			try
			{
				url=new java.io.File(prefix+fileName).toURL();
				if (url != null && url.openStream() != null)
					return url;
			}
			catch(Exception e)
			{
				
			}
		}
		//System.out.println("Could not find it");
		return null;
  }

  /**
   * gets the line of the file where the error is. This reads from the
   * file until a semicolon is found.
   * 
   * @param fileName the name of the file to read from
   * @param lineNumber the line to return the contents of
   * @return the statement starting at the given line
   */
  public static String getLine(String fileName, int lineNumber) {
    try {
      URL url = getSourceURL(fileName);
      InputStreamReader fromURL =
          new InputStreamReader(url.openStream());
      BufferedReader reader = new BufferedReader(fromURL);
      String line = reader.readLine();
      int currentLineNumber = 1;
      while (currentLineNumber != lineNumber) {
        line = reader.readLine();
        currentLineNumber++;
      }
      while (line.indexOf(";") < 0) {
        line += "\n" + reader.readLine();
      }
      return line;
    } catch (IOException e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * replaces the symbols less than, greater than, quotes, new lines,
   * spaces, and tabs with the corresponding html to display these
   * properly.
   * 
   * @param text the text to convert, usually Java code
   * @return the html for displaying the text properly
   */
  public static String fixHTML(String text) {
    return text.replaceAll("<", "&lt;").replaceAll(">", "&gt;")
        .replaceAll("\"", "\\\"").replace("\n", "<br>").replace(" ",
            "&nbsp;").replace("\t", "&nbsp;&nbsp;&nbsp;");
  }

  /**
   * makes the text monospaced in html
   * 
   * @param text the text to make monospaced, typically file names and
   *          code
   * @return text surrounded with a tag to make it monospaced
   */
  public static String fixedWidth(String text) {
    return "<span style=\"font-family: monospace; font-weight: bold;color: rgb(255, 255, 0);\">" +
           text + "</span>";
  }

  /**
   * indents the given text 40 pixels
   * 
   * @param text the string to indent
   * @return text surrounded with a tag to indent it 40 pixels
   */
  public static String indent(String text) {
    return "<div style=\"margin-left: 40px;\">" + text + "</div>";
  }

  /**
   * makes the text large
   * 
   * @param text the heading
   * @return text surrounded with a tag for making it large
   */
  public static String heading(String text) {
    return "<h1 style=\"color: rgb(255, 255, 255);\">" + text +
           "</h1><br>";
  }

  /**
   * makes the text slightly smaller than the heading
   * 
   * @param text the subheading
   * @return text surrounded with a tag for making it large
   */
  public static String subHeading(String text) {
    return "<h2 style=\"color: rgb(255, 255, 255);\">" + text + "</h2>";
  }

  /**
   * gets the text for displaying the error's location
   * 
   * @param e the exception that generated the error
   * @return detailed information about where the error occurred
   */
  public static String getLocationSection(Throwable e) {
    String message =
        subHeading("Error Location") +
            "This error was generated by line " +
            getErrorLineNumber(e) + " of the file<br>" +
            indent(fixedWidth(getErrorFile(e))) + "<br>" +
            "This line is <br>" + fixedWidth(fixHTML(getErrorLine(e))) +
            "<br>";
    if (e != null) {
      message += "<br>Exception Stack Trace:<br><br>";
      StringWriter writer = new StringWriter();
      e.printStackTrace(new PrintWriter(writer));
      message += fixedWidth("<pre>" + writer.toString() + "</pre>");
    }
    return message;
  }

  /**
   * gets the text of the line where the error occurred
   * 
   * @return the line of the error ending in a semicolon
   */
  public static String getErrorLine(Throwable t) {
    return getLine(getErrorFile(t), getErrorLineNumber(t));
  }

  /**
   * gets the line number where the error occurred
   * 
   * @return the line number
   */
  public static int getErrorLineNumber(Throwable t) {
    return getErrorElement(t).getLineNumber();
  }

  /**
   * gets the name of the method where the error occurred
   * 
   * @return the method name
   */
  public static String getErrorMethod(Throwable t) {
    return getErrorElement(t).getMethodName();
  }

  /**
   * gets the name of the source file where the error occurred
   * 
   * @return the file name of the code with the error in it
   */
  public static String getErrorFile(Throwable t) {
    StackTraceElement element = getErrorElement(t);
    String fileName = element.getClassName();
    fileName = fileName.replace('.', '/');
    fileName = fileName + ".java";
    return fileName;
  }

  /**
   * iterates through the execution stack to find the first element
   * which is outside of the FANG Engine
   * 
   * @return the stack trace element of the code with the error
   */
  public static StackTraceElement getErrorElement(Throwable t) {
    StackTraceElement[] all = t.getStackTrace();
    int i;
    for (i = 0; i < all.length; i++) {
      String packageName = all[i].getClassName();
      if (!packageName.startsWith("fang2") &&
          !packageName.startsWith("java")) {
        break;
      }
    }
    if (i == all.length) {
      return all[i - 1];
    }
    return all[i];
  }

  /**
   * Set the content of the window displaying the error screen. If more
   * than one error has occurred, adds the error to the queue.
   * 
   * @param diagnosis the error diagnosis
   * @param fix suggested fix
   * @param e the error/exception which happened (used for setting the
   *          HTML header)
   */
  public static void addError(String diagnosis, String fix, Throwable e) {
    String content =
        heading(e.getClass().getCanonicalName()) +
            subHeading("Diagnosis") + diagnosis + "<br>" +
            subHeading("Suggested Fix") + fix + getLocationSection(e);
    single.errors.add(content);
    if (single.errors.size() == 1) {
      single.message.setText(content);
      single.message.setCaretPosition(0);
      single.setVisible(true);
    } else {
      single.closeButton.setText("Next Error Message");
    }
  }

  /**
   * call this method for errors which should not occur. If they do
   * occur, the FANG Engine developers need to know about it.
   * 
   * @param e the unexpected exception
   */
  public static void addUnknownError(Throwable e) {
    addError("Strange, this error does not come up often.",
        "Please make a jar of this game.  Email "
            + "bug@fangengine.org the jar file along"
            + " with a description of how to recreate" + " the error.",
        e);
  }

  /**
   * this catches uncaught exceptions when the game runs as an
   * application. The primary uncaught exceptions are initializer errors
   * in games run as applications. Unfortunately, nothing can detect
   * initializer errors in applets very well.
   */
  public static void registerExceptionHandler() {
    Thread.currentThread().setUncaughtExceptionHandler(
        new TopExceptionHandler());
  }

  /** this class forwards uncaught exceptions to addUnknownError */
  private static class TopExceptionHandler
    implements Thread.UncaughtExceptionHandler {
    /** forwards uncaught exceptions to addUnknownError */
    public void uncaughtException(Thread arg0, Throwable arg1) {
      addUnknownError(arg1);
    }
  }

  public static boolean hasErrors() {
	  return single!=null && !single.errors.isEmpty();
  }

}
