package fang2.ui;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import javax.swing.JComponent;

import fang2.core.AnimationCanvas;
import fang2.sprites.ButtonSprite;

public class ButtonCanvas extends JComponent
            implements MouseListener
{
    private Point2D.Double press;
    private AnimationCanvas canvas;

    public static final double HEIGHT = 0.1;
    /**
     * used for serialization versioning
     */
    private static final long serialVersionUID = 1L;

    private ButtonSprite[] button;

    /** constructs an empty canvas with the default size*/
    public ButtonCanvas(AnimationCanvas canvas)
    {
        this.canvas = canvas;
        setSize(new Dimension(1, 1));
        setFocusable(true);
        //setOpaque(true);
        //setBackground(AnimationCanvas.DEFAULT_BACKGROUND);
        //setIgnoreRepaint(true);
        makeButtons();
        addMouseListener(this);
    }

    private void makeButtons()
    {
        button = new ButtonSprite[4];
        double aspect = 1.0 / button.length / 0.09;
        button[0] = new ButtonSprite("Start", aspect);
        button[1] = new ButtonSprite("Sound", aspect);
        button[2] = new ButtonSprite("Help", aspect);
        button[3] = new ButtonSprite("Quit", aspect);
        for (int i = 0; i < button.length; i++)
        {
            button[i].setScale(0.24);
            button[i].setLocation((i + 0.5) / button.length, 0.05);
        }
    }

    /** redraws the area which contains the Sprites */
    public void paintImmediately()
    {
        paintImmediately(0, 0, getSize().width, getSize().height);
    }

    private double getScalingFactor()
    {
        return canvas.getWidth();
    }

    @Override
    protected void paintComponent(Graphics brush)
    {
        Graphics2D copy = (Graphics2D) brush;
        copy.addRenderingHints(new RenderingHints(
                                   RenderingHints.KEY_ANTIALIASING,
                                   RenderingHints.VALUE_ANTIALIAS_ON));
        double scalingFactor;
        scalingFactor = getScalingFactor(); //Math.min(clip.x, clip.y) * 10;
        copy.transform(AffineTransform.getScaleInstance(scalingFactor,
                       scalingFactor));
        for (ButtonSprite b : button) {
          b.paint(copy);
        }
    }

    /**
     * gets the preferred size as the current size
     * @return the current size
     */
    @Override
    public Dimension getPreferredSize()
    {
        return new Dimension(
                   (int)getScalingFactor(),
                   (int)(HEIGHT*getScalingFactor()));
    }

    public void mouseClicked(MouseEvent e)
    {}

    public void mouseEntered(MouseEvent e)
    {}

    public void mouseExited(MouseEvent e)
    {}

    /**
     * Given a pixel-point (within the {@code ButtonCanvas}), convert it to a point in internal coordinates.
     * Used by mouse* event handlers to translate mouse locations.
     * 
     * @param pixelLocation the pixel location to translate.
     * 
     * @return Point2D representing the location internally
     */
    private Point2D.Double scaledPoint(Point pixelLocation) {
      return new Point2D.Double(pixelLocation.x / getScalingFactor(), pixelLocation.y / getScalingFactor());
    }
    
    public void mousePressed(MouseEvent e)
    {
        Point location = e.getPoint();
        press = scaledPoint(location);
        for (ButtonSprite b: button) {
          if (b.intersects(press)) {
            b.lower();
          }
        }
        repaint();
    }

    public void mouseReleased(MouseEvent e)
    {
        for (ButtonSprite b: button) {
          b.raise();
        }
        Point location = e.getPoint();
        Point2D.Double release = scaledPoint(location);
        for (ButtonSprite b: button) {
          if (b.intersects(release) && b.intersects(press))
          {
              b.executePressAction();
          }
        }
        repaint();
    }

    public void addActionListeners(ActionListener ... listen)
    {
        for (int i = 0; i < listen.length; i++)
        {
            button[i].addActionListener(listen[i]);
        }
    }

    public ButtonSprite getPause()
    {
        // TODO Auto-generated method stub
        return button[0];
    }

    public ButtonSprite getMute()
    {
        // TODO Auto-generated method stub
        return button[1];
    }

    public ButtonSprite getQuit()
    {
        // TODO Auto-generated method stub
        return button[3];
    }

    public ButtonSprite getHelp()
    {
        // TODO Auto-generated method stub
        return button[2];
    }
}