package fang2.transformers;

import fang2.attributes.Location2D;
import fang2.attributes.Vector2D;
import fang2.core.TransformerAdapter;

/**
 * Constant linear velocity {@link TransformerAdapter}. The velocity
 * is stored as a vector and is used, during advance, to calculate the
 * displacement represented by the transform.
 */
public class VelocityTransformer
  extends TransformerAdapter
  implements VelocityProvider {
  /** the velocity vector in screens/second */
  private Vector2D velocity;

  private double maximumMagnitude;
  private double minimumMagnitude;

  private double nominalMagnitude;
  private boolean resetToNominal;

  /**
   * Construct a new {@link VelocityTransformer} with the given
   * velocity.
   *
   * @param  velocity
   */
  public VelocityTransformer(Vector2D velocity) {
    this.velocity = velocity;
    this.maximumMagnitude = Double.MAX_VALUE;
    this.minimumMagnitude = 0.0;
    this.nominalMagnitude = 0.0;
    this.resetToNominal = false;
  }

  /**
   * Construct a new {@link VelocityTransformer} with the given
   * velocity.
   *
   * @param  degrees    facing of the velocity
   * @param  magnitude  distance of the velocity (in screens)
   */
  public VelocityTransformer(double degrees, double magnitude) {
    this(new Vector2D(degrees, magnitude));
  }

  /**
   * Construct a new {@link VelocityTransformer} with a 0, 0 velocity.
   */
  public VelocityTransformer() {
    this(0.0, 0.0);
  }

  /**
   * @return  the maximumMagnitude
   */
  public double getMaximumMagnitude() {
    return maximumMagnitude;
  }

  /**
   * @return  the minimumMagnitude
   */
  public double getMinimumMagnitude() {
    return minimumMagnitude;
  }

  /**
   * @return  the nominalMagnitude
   */
  public double getNominalMagnitude() {
    return nominalMagnitude;
  }

  /**
   * @param  maximumMagnitude  the maximumMagnitude to set
   */
  public void setMaximumMagnitude(double maximumMagnitude) {
    this.maximumMagnitude = maximumMagnitude;
  }

  /**
   * @param  minimumMagnitude  the minimumMagnitude to set
   */
  public void setMinimumMagnitude(double minimumMagnitude) {
    this.minimumMagnitude = minimumMagnitude;
  }

  /**
   * @param  nominalMagnitude  the nominalMagnitude to set
   */
  public void setNominalMagnitude(double nominalMagnitude) {
    this.nominalMagnitude = nominalMagnitude;
  }

  /**
   * @return  the velocity
   */
  public Vector2D getVelocity() {
    return velocity;
  }

  /**
   * Do "after update" clean-up. Make sure that the velocity is damped
   * out if appropriate.
   *
   * @param  dT  time since last advance in seconds
   */
  @Override
  public void nonMaskableAdvance(double dT) {
    if (resetToNominal()) {
      velocity.setMagnitude(nominalMagnitude);
    }
  }

  /**
   * Is the transformer resetting to nominal value each frame?
   *
   * @return  the resetToNominal
   */
  public boolean resetToNominal() {
    return resetToNominal;
  }

  /**
   * Should the velocity be reset to nominal after update?
   *
   * @param  resetToNominal  the resetToNominal to set
   */
  public void setResetToNominal(boolean resetToNominal) {
    this.resetToNominal = resetToNominal;
  }

  /**
   * Set the velocity to the given vector. Make copy to that shared
   * references don't leak out of this object.
   *
   * @param  velocity  the velocity to set
   */
  public void setVelocity(Vector2D velocity) {
    this.velocity = new Vector2D(velocity);
    boundMagnitude();
  }

  /**
   * Set the velocity using the given {@link Location2D}. The x, y
   * components of the location are taken to be the x, y components of
   * the velocity.
   *
   * @param  componentVelocity  the new velocity to set
   */
  public void setVelocity(Location2D componentVelocity) {
    this.velocity.setXYChange(componentVelocity.getX(),
      componentVelocity.getY());
    boundMagnitude();
  }

  /**
   * Advance one frame. The instantaneous velocity is the current
   * velocity value; times the elapsed time it is the displacement.
   *
   * @see  TransformerAdapter#advance(double)
   */
  @Override
  public void advance(double dT) {
    setLocation(new Location2D(velocity.multiply(dT)));
  }

  /**
   * Make sure the magnitude of the velocity is bounded by the lower and
   * upper bounds. This is only for keeping minimum and maximum speeds;
   * a more complex transformer would be necessary to limit the
   * directions which the transformer could go.
   */
  private void boundMagnitude() {
    double magnitude = velocity.getMagnitude();
    if (magnitude > maximumMagnitude) {
      velocity.setMagnitude(maximumMagnitude);
    }
    if (magnitude < minimumMagnitude) {
      velocity.setMagnitude(minimumMagnitude);
    }
  }
}
