package fang2.transformers;

import fang2.core.Sprite;
import fang2.core.TransformerAdapter;

/**
 * A "spinner" {@link TransformerAdapter}. This transformer turns the sprite
 * according to the angular velocity set in the transformer. Treats it
 * as constant velocity.
 */
public class SpinTransformer
  extends TransformerAdapter {
  /** angular velocity applied by this transformer; radians/second */
  private double angularVelocityRadians;

  /** the range of permitted angular velocities; radians/second */
  private double maximumAVRadians;
  private double minimumAVRadians;

  /** nominal (default) velocity; radians/second */
  private double nominalAVRadians;

  /**
   * does the velocity reset to nominal at end of {@link
   * #updateSprite(Sprite)}?
   */
  private boolean resetToNominal;

  /**
   * Construct a new spinner transformer with the given angular
   * velocity.
   *
   * @param  angularVelocityDegrees  turning speed in radians/second
   */
  public SpinTransformer(double angularVelocityDegrees) {
    this.angularVelocityRadians = Math.toRadians(angularVelocityDegrees);
    this.minimumAVRadians = -Double.MAX_VALUE;
    this.maximumAVRadians = Double.MAX_VALUE;
    this.nominalAVRadians = 0.0;
    boundAngularVelocity();
  }

  /**
   * Update the transformer one frame time. This means set the delta
   * rotation to the part of a rotation done it he given time.
   *
   * @param  dT  time since last advance
   */
  @Override
  public void advance(double dT) {
    setRotation(dT * angularVelocityRadians);
  }

  /**
   * Do "after update" clean-up. Make sure that the velocity is damped
   * out if appropriate.
   * @param dT time since last advance in seconds
   */
  @Override
  public void nonMaskableAdvance(double dT) {
    if (resetToNominal()) {
      setRotation(nominalAVRadians);
    }
  }

  /**
   * Get velocity (spin) in radians/second
   *
   * @return  the angular velocity in radians/second
   */
  public double getAngularVelocity() {
    return angularVelocityRadians;
  }

  /**
   * Get angular velocity (spin) in degrees/second
   *
   * @return  angular velocity in degrees/second
   */
  public double getAngularVelocityDegrees() {
    return Math.toDegrees(getAngularVelocity());
  }

  /**
   * Get velocity in radians/second
   *
   * @return  the angular velocity in radians/second
   */
  public double getAngularVelocityRadians() {
    return getAngularVelocity();
  }

  /**
   * Get angular velocity (spin) in revolutions/second
   *
   * @return  angular velocity in revolutions/second
   */
  public double getAngularVelocityRevolutions() {
    return getAngularVelocity() / (2 * Math.PI);
  }

  /**
   * Get maximum spin value.
   *
   * @return  the maximumAVRadians
   */
  public double getMaximumAngularVelocity() {
    return maximumAVRadians;
  }

  /**
   * Get maximum spin value in degrees/second
   *
   * @return  the maximumAVDegrees
   */
  public double getMaximumAngularVelocityDegrees() {
    return Math.toDegrees(getMaximumAngularVelocity());
  }

  /**
   * Get maximum spin value in radians/second
   *
   * @return  the maximumAVRadians
   */
  public double getMaximumAngularVelocityRadians() {
    return getMaximumAngularVelocity();
  }

  /**
   * Get maximum spin value in revolutions/second
   *
   * @return  the maximumAVRevolutions
   */
  public double getMaximumAngularVelocityRevolutions() {
    return getMaximumAngularVelocity() / (2 * Math.PI);
  }

  /**
   * Get minimum spin value.
   *
   * @return  the minimumAVRadians
   */
  public double getMinimumAngularVelocity() {
    return minimumAVRadians;
  }

  /**
   * Get minimum spin value in degrees/second
   *
   * @return  the minimumAVDegrees
   */
  public double getMinimumAngularVelocityDegrees() {
    return Math.toDegrees(getMinimumAngularVelocity());
  }

  /**
   * Get minimum spin value in radians/second
   *
   * @return  the minimumAVRadians
   */
  public double getMinimumAngularVelocityRadians() {
    return getMinimumAngularVelocity();
  }

  /**
   * Get minimum spin value in revolutions/second
   *
   * @return  the minimumAVRevolutions
   */
  public double getMinimumAngularVelocityRevolutions() {
    return getMinimumAngularVelocity() / (2 * Math.PI);
  }

  /**
   * Get the "default" or nominal spin value. Used when resetToNominal
   * is in effect. After each update the value is reset.
   *
   * @return  the nominalAVRadians
   */
  public double getNominalAngularVelocity() {
    return nominalAVRadians;
  }

  /**
   * Get the "default" or nominal spin value. Used when resetToNominal
   * is in effect. After each update the value is reset. In
   * degrees/second.
   *
   * @return  the nominalAVDegrees
   */
  public double getNominalAngularVelocityDegrees() {
    return Math.toDegrees(getNominalAngularVelocity());
  }

  /**
   * Get the "default" or nominal spin value. Used when resetToNominal
   * is in effect. After each update the value is reset. In
   * radians/second.
   *
   * @return  the nominalAVRadians
   */
  public double getNominalAngularVelocityRadians() {
    return getNominalAngularVelocity();
  }

  /**
   * Get the "default" or nominal spin value. Used when resetToNominal
   * is in effect. After each update the value is reset. In
   * revolutions/second.
   *
   * @return  the nominalAVRevolutions
   */
  public double getNominalAngularVelocityRevolutions() {
    return getNominalAngularVelocity() / (2 * Math.PI);
  }

  /**
   * Is the transformer resetting to nominal value each frame?
   *
   * @return  the resetToNominal
   */
  public boolean resetToNominal() {
    return resetToNominal;
  }

  /**
   * Set the angular velocity.
   *
   * @param  angularVelocityRadians  the angular velocity to set;
   *                                 radians/second
   */
  public void setAngularVelocity(double angularVelocityRadians) {
    this.angularVelocityRadians = angularVelocityRadians;
    boundAngularVelocity();
  }

  /**
   * Set the angular velocity.
   *
   * @param  angularVelocityDegrees  the angular velocity to set;
   *                                 degrees/second
   */
  public void setAngularVelocityDegrees(double angularVelocityDegrees) {
    setAngularVelocity(Math.toRadians(angularVelocityDegrees));
  }

  /**
   * Set the angular velocity.
   *
   * @param  angularVelocityRadians  the angular velocity to set;
   *                                 radians/second
   */
  public void setAngularVelocityRadians(double angularVelocityRadians) {
    setAngularVelocity(angularVelocityRadians);
  }

  /**
   * Set the angular velocity.
   *
   * @param  angularVelocityRevolutions  the angular velocity to set;
   *                                     revolutions/second
   */
  public void setAngularVelocityRevolutions(
    double angularVelocityRevolutions) {
    setAngularVelocity(angularVelocityRevolutions * 2 * Math.PI);
  }

  /**
   * Set maximum angular velocity permitted; radians/second
   *
   * @param  maximumAVRadians  the maximumAVRadians to set
   */
  public void setMaximumAngularVelocity(double maximumAVRadians) {
    this.maximumAVRadians = maximumAVRadians;
    boundAngularVelocity();
  }

  /**
   * Set maximum angular velocity permitted; degrees/second
   *
   * @param  maximumAVDegrees  the maximumAVDegrees to set
   */
  public void setMaximumAngularVelocityDegrees(
    double maximumAVDegrees) {
    setMaximumAngularVelocity(Math.toRadians(maximumAVDegrees));
  }

  /**
   * Set maximum angular velocity permitted; radians/second
   *
   * @param  maximumAVRadians  the maximumAVRadians to set
   */
  public void setMaximumAngularVelocityRadians(
    double maximumAVRadians) {
    setMaximumAngularVelocity(maximumAVRadians);
  }

  /**
   * Set maximum angular velocity permitted; revolutions/second
   *
   * @param  maximumAVRevolutions  the maximumAVRevolutions to set
   */
  public void setMaximumAngularVelocityRevolutions(
    double maximumAVRevolutions) {
    setMaximumAngularVelocity(maximumAVRevolutions * 2 * Math.PI);
  }

  /**
   * Set minimum angular velocity permitted; radians/second
   *
   * @param  minimumAVRadians  the minimumAVRadians to set
   */
  public void setMinimumAngularVelocity(double minimumAVRadians) {
    this.minimumAVRadians = minimumAVRadians;
    boundAngularVelocity();
  }

  /**
   * Set minimum angular velocity permitted; degrees/second
   *
   * @param  minimumAVDegrees  the minimumAVDegrees to set
   */
  public void setMinimumAngularVelocityDegrees(
    double minimumAVDegrees) {
    setMinimumAngularVelocity(Math.toRadians(minimumAVDegrees));
  }

  /**
   * Set minimum angular velocity permitted; radians/second
   *
   * @param  minimumAVRadians  the minimumAVRadians to set
   */
  public void setMinimumAngularVelocityRadians(
    double minimumAVRadians) {
    setMinimumAngularVelocity(minimumAVRadians);
  }

  /**
   * Set minimum angular velocity permitted; revolutions/second
   *
   * @param  minimumAVRevolutions  the minimumAVRevolutions to set
   */
  public void setMinimumAngularVelocityRevolutions(
    double minimumAVRevolutions) {
    setMinimumAngularVelocity(minimumAVRevolutions * 2 * Math.PI);
  }

  /**
   * Set the nominal value for rotation velocity; radians/second
   *
   * @param  nominalAVRadians  the nominalAVRadians to set
   */
  public void setNominalAngularVelocity(double nominalAVRadians) {
    this.nominalAVRadians = nominalAVRadians;
  }

  /**
   * Set the nominal value for rotation velocity; degrees/second
   *
   * @param  nominalAVDegrees  the nominalAVDegrees to set
   */
  public void setNominalAngularVelocityDegrees(
    double nominalAVDegrees) {
    setNominalAngularVelocity(Math.toRadians(nominalAVDegrees));
  }

  /**
   * Set the nominal value for rotation velocity; radians/second
   *
   * @param  nominalAVRadians  the nominalAVRadians to set
   */
  public void setNominalAngularVelocityRadians(
    double nominalAVRadians) {
    setNominalAngularVelocity(nominalAVRadians);
  }

  /**
   * Set the nominal value for rotation velocity; revolutions/second
   *
   * @param  nominalAVRevolutions  the nominalAVRevolutions to set
   */
  public void setNominalAngularVelocityRevolutions(
    double nominalAVRevolutions) {
    setNominalAngularVelocity(nominalAVRevolutions * 2 * Math.PI);
  }

  /**
   * Should the velocity be reset to nominal after update?
   *
   * @param  resetToNominal  the resetToNominal to set
   */
  public void setResetToNominal(boolean resetToNominal) {
    this.resetToNominal = resetToNominal;
  }

  /**
   * Make sure the angular velocity is within the prescribed limits.
   * Must be checked after setting velocity, minimum, or maximum.
   */
  private void boundAngularVelocity() {
    if (this.angularVelocityRadians < minimumAVRadians) {
      this.angularVelocityRadians = minimumAVRadians;
    }
    if (this.angularVelocityRadians > maximumAVRadians) {
      this.angularVelocityRadians = maximumAVRadians;
    }
  }
}
