package fang2.transformers;

import fang2.core.TransformerAdapter;

/**
 * A "spinner" {@link TransformerAdapter}. This transformer turns the sprite
 * according to a changing angular velocity. Acceleration is constant,
 * held in the angular acceleration field. The field is available in all
 * angle types supported by fang2.
 */
public class SpinAccelerationTransformer
  extends SpinTransformer {
  /** angular velocity applied by this transformer; radians/second */
  private double angularAccelerationRadians;

  public SpinAccelerationTransformer(double angularVelocityDegrees, double angularAccelerationDegrees) {
    super(angularVelocityDegrees);
    this.angularAccelerationRadians = Math.toRadians(angularAccelerationDegrees);
  }
  
  /**
   * Update the transformer one frame time. This means set the delta
   * rotation to the part of a rotation done it he given time.
   *
   * @param  dT  time since last advance
   */
  @Override
  public void advance(double dT) {
    setAngularVelocity(getAngularVelocity() +
      (dT * angularAccelerationRadians));
  }

  /**
   * Get velocity (spin) in radians/second
   *
   * @return  the angular velocity in radians/second
   */
  public double getAngularAccleration() {
    return angularAccelerationRadians;
  }

  /**
   * Get angular velocity (spin) in degrees/second
   *
   * @return  angular velocity in degrees/second
   */
  public double getAngularAcclerationDegrees() {
    return Math.toDegrees(getAngularAccleration());
  }

  /**
   * Get velocity in radians/second
   *
   * @return  the angular velocity in radians/second
   */
  public double getAngularAcclerationRadians() {
    return getAngularAccleration();
  }

  /**
   * Get angular velocity (spin) in revolutions/second
   *
   * @return  angular velocity in revolutions/second
   */
  public double getAngularAcclerationRevolutions() {
    return getAngularAccleration() / (2 * Math.PI);
  }

  /**
   * Set the angular velocity.
   *
   * @param  angularAcclerationRadians  the angular velocity to set;
   *                                    radians/second
   */
  public void setAngularAccleration(double angularAcclerationRadians) {
    this.angularAccelerationRadians = angularAcclerationRadians;
  }

  /**
   * Set the angular velocity.
   *
   * @param  angularAcclerationDegrees  the angular velocity to set;
   *                                    degrees/second
   */
  public void setAngularAcclerationDegrees(
    double angularAcclerationDegrees) {
    setAngularAccleration(angularAcclerationDegrees);
  }

  /**
   * Set the angular velocity.
   *
   * @param  angularAcclerationRadians  the angular velocity to set;
   *                                    radians/second
   */
  public void setAngularAcclerationRadians(
    double angularAcclerationRadians) {
    setAngularAccleration(angularAcclerationRadians);
  }

  /**
   * Set the angular velocity.
   *
   * @param  angularAcclerationRevolutions  the angular velocity to set;
   *                                        revolutions/second
   */
  public void setAngularAcclerationRevolutions(
    double angularAcclerationRevolutions) {
    setAngularAccleration(angularAcclerationRevolutions);
  }
}
