package fang2.transformers;

import fang2.core.Game;
import fang2.core.Sprite;
import fang2.core.TransformerAdapter;

/**
 * Decorate a {@link TransformerAdapter} so that it runs forward or backward
 * for any frame only when appropriate keys are pressed.
 *
 * <p>When a key in the {@link #plusKeys} collection is pressed, then
 * the controlled {@link TransformerAdapter} is called with a positive time
 * step; when a key in the {@link #minusKeys} collection is pressed,
 * then the controlled {@link TransformerAdapter} is called with an inverted,
 * negative time step.
 *
 * <p>This is an example of the <i>decorator pattern</i>: this is a
 * {@link TransformerAdapter} which includes, as part of itself, a {@link
 * TransformerAdapter} which is decorated or filtered by this transformer.
 */
public class KeyboardTransformer
  extends TransformerAdapter {
  private final StringBuffer minusKeys;

  /** which player's keyboard activity being watched? */
  private int playerNumber;

  /** character collections for forward and backward runs */
  private final StringBuffer plusKeys;

  /** the decorated transformer; must work forward and back */
  private final TransformerAdapter transformer;

  /**
   * Create a new {@link KeyboardTransformer} controlling the given
   * transformer and using the default (local) player's number.
   *
   * @param  transformer  the {@link TransformerAdapter} controlled by the
   *                      plus/minus keys.
   */
  public KeyboardTransformer(TransformerAdapter transformer) {
    this(transformer, 0);
  }

  /**
   * Create a new {@link KeyboardTransformer} controlling the given
   * {@link TransformerAdapter} using the given player's keyboard activity.
   *
   * @param  transformer   the transformer driven by the plus/minus keys
   *                       under this keyboard
   * @param  playerNubmer  which player's keyboard is used to drive the
   *                       transformer
   */
  public KeyboardTransformer(TransformerAdapter transformer,
    int playerNubmer) {
    this.transformer = transformer;
    plusKeys = new StringBuffer();
    minusKeys = new StringBuffer();
  }

  /**
   * @param  minusKey  the minusKey to set
   */
  public void addMinusKey(int minusKey) {
    this.minusKeys.append((char) minusKey);
  }

  /**
   * @param  plusKey  the plusKey to set
   */
  public void addPlusKey(int plusKey) {
    this.plusKeys.append((char) plusKey);
  }

  /**
   * Advance one time step.
   *
   * <p>Check the key pressed status. If a {@link #plusKeys} has been
   * pressed, then call the advance method of transform with dT. If a
   * {@link #minusKeys} has been pressed, then call the advance method
   * of the transform with -dT (negative time!).
   *
   * @param  dT  the time elapsed since the last call to advance
   */
  @Override
  public void advance(double dT) {
    Game g = Game.getCurrentGame();
    if (g.keyPressed(playerNumber)) {
      String keyPressed = Character.toString(g.getKeyPressed(
            playerNumber));

      if (plusKeys.indexOf(keyPressed) >= 0) {
        transformer.advance(dT);
      } else if (minusKeys.indexOf(keyPressed) >= 0) {
        transformer.advance(-dT);
      }
    }
  }

  /**
   * Get the collection of minus keys.
   *
   * @return  a string with all the characters used for the minus keys,
   *          for calling the transform in a backward direction
   */
  public String getMinusKeys() {
    return minusKeys.toString();
  }

  /**
   * Get the player number for whom this transformer is installed.
   *
   * @return  player number for this object.
   */
  public int getPlayerNumber() {
    return playerNumber;
  }

  /**
   * Get the collection of plus keys.
   *
   * @return  a string with all the characters used for the plus keys,
   *          for calling the transform in a forward direction
   */
  public String getPlusKeys() {
    return plusKeys.toString();
  }

  /**
   * Make sure the non-maskable advance method of any wrapped
   * (decorated) transformer is called. Required for damping and similar
   * "after update delta" effects.
   *
   * @param  dT  time since last advance
   */
  @Override
  public void nonMaskableAdvance(double dT) {
    transformer.nonMaskableAdvance(dT);
  }

  /**
   * Remove the given character from the minusKeys set.
   *
   * @param  minusKey  the character to no longer use
   */
  public void removeMinusKey(int minusKey) {
    int match = this.minusKeys.indexOf(Character.toString(
          (char) minusKey));
    if (match >= 0) {
      minusKeys.deleteCharAt(match);
    }
  }

  /**
   * Remove the given character from the plusKeys set.
   *
   * @param  plusKey  the character to no longer use
   */
  public void removePlusKey(int plusKey) {
    int match = this.plusKeys.indexOf(Character.toString(
          (char) plusKey));
    if (match >= 0) {
      plusKeys.deleteCharAt(match);
    }
  }

  /**
   * Extension point for {@link TransformerAdapter}
   *
   * @param  sprite  the sprite to move
   */
  @Override
  public void updateSprite(Sprite sprite) {
    transformer.updateSprite(sprite);
  }
}
