package fang2.transformers;

import fang2.core.Sprite;
import fang2.core.Transformer;

public abstract class InterpolatorTransformer
  implements Transformer {
  /** current time in the cycle */
  private double currTime;

  /** should it cycle */
  private boolean cycling;

  /** the direction to move */
  private double direction = 1.0;

  /** is the interpolator enabled? */
  private boolean enabled;

  /** should it oscillate */
  private boolean oscillating;

  /** time to go from start to end */
  private double transitionTime;

  /**
   * Make one that has the given cycle and enabled status. Cycling and
   * oscillating set to false.
   *
   * @param  transitionTime  time to transition
   * @param  enabled         is it turned on?
   */
  public InterpolatorTransformer(double transitionTime,
    boolean enabled) {
    this(transitionTime, enabled, false, false);
  }

  /**
   * Make a new {@link InterpolatorTransformer}.
   *
   * @param  transitionTime  how long does transition take from start to
   *                         finish? in seconds
   * @param  enabled         is this running?
   * @param  cycling         does it recycle at the top (go from 1.0 to
   *                         0.0)?
   * @param  oscillating     does it oscillate (change directions at top
   *                         and bottom of the range)
   */
  public InterpolatorTransformer(double transitionTime,
    boolean enabled, boolean cycling, boolean oscillating) {
    this.transitionTime = transitionTime;
    this.currTime = 0.0;
    this.enabled = enabled;
    this.cycling = cycling;
    this.oscillating = oscillating;
  }

  /**
   * Advance one frame: if the transformer is enabled then update the
   * current time (multiply direction times the dT for the change). If
   * time is past one end or the other, check if we reset or change
   * direction. Then, finally, call abstract interpolate method with the
   * interpolation value (linear interpolation means it is just a
   * percentage of time between 0 and transitionTime).
   *
   * @param  dT  the time since the last call to advance
   */
  public void advance(double dT) {
    if (enabled) {
      currTime += direction * dT;

      if (currTime < 0.0) {
        currTime = 0.0;
        if (oscillating) {
          direction = 1.0;
        } else {
          direction = 0.0;
        }
      } else if (currTime > transitionTime) {
        currTime = transitionTime;
        if (oscillating) {
          direction = -1.0;
        } else if (cycling) {
          currTime = 0.0;
        } else {
          direction = 0.0;
        }
      }
      interpolate(interpolateValue());
    }
  }

  /**
   * @return  the transitionTime
   */
  public double getTransitionTime() {
    return transitionTime;
  }

  /**
   * @return  the cycling
   */
  public boolean isCycling() {
    return cycling;
  }

  /**
   * @return  the enabled
   */
  public boolean isEnabled() {
    return enabled;
  }

  /**
   * @return  the oscillating
   */
  public boolean isOscillating() {
    return oscillating;
  }

  public void nonMaskableAdvance(double dT) {
    // nothing to do
  }

  /**
   * @param  cycling  the cycling to set
   */
  public void setCycling(boolean cycling) {
    this.cycling = cycling;
  }

  /**
   * @param  enabled  the enabled to set
   */
  public void setEnabled(boolean enabled) {
    this.enabled = enabled;
  }

  /**
   * @param  oscillating  the oscillating to set
   */
  public void setOscillating(boolean oscillating) {
    this.oscillating = oscillating;
  }

  /**
   * @param  transitionTime  the transitionTime to set
   */
  public void setTransitionTime(double transitionTime) {
    this.transitionTime = transitionTime;
  }

  /**
   * Must be provided by subclasses: actually applies the transformer to
   * the sprite.
   *
   * @param  sprite  the sprite to transform
   */
  public abstract void updateSprite(Sprite sprite);

  /**
   * Determine the interpolation value, what percentage of the end value
   * we should use.
   *
   * @return  percentage of end value to use
   */
  private double interpolateValue() {
    return currTime / transitionTime;
  }

  /**
   * Must be provided by subclasses: updates the values according to %
   * of interpolation done; called from {@link #advance(double)}.
   *
   * @param  interpolated  0.0-1.0 the amount of end value to use
   */
  protected abstract void interpolate(double interpolated);
}
