package fang2.transformers;

import java.awt.Color;

import fang2.core.Sprite;

public class ColorTransformer
  extends InterpolatorTransformer {
  /** the current color */
  private Color currColor;
  private final float[] currColorComponents;

  /** final color */
  private Color endColor;
  private final float[] endColorComponents;

  /** initial color */
  private Color startColor;
  private final float[] startColorComponents;

  /**
   * Make a {@link ColorTransformer} with the given cycle
   * and colors.
   *
   * @param  transitionTime  transition cycle time
   * @param  enabled         does it start enabled?
   * @param  startColor      starting color
   * @param  endColor        ending color
   */
  public ColorTransformer(double transitionTime,
    boolean enabled, Color startColor, Color endColor) {
    super(transitionTime, enabled);
    this.startColorComponents = new float[4];
    this.currColorComponents = new float[4];
    this.endColorComponents = new float[4];
    setStartColor(startColor);
    setEndColor(endColor);
  }

  /**
   * Make a {@link ColorTransformer} with the given cycle
   * and colors.
   *
   * @param  transitionTime  transition cycle time
   * @param  startColor      starting color
   * @param  endColor        ending color
   */
  public ColorTransformer(double transitionTime,
    Color startColor, Color endColor) {
    this(transitionTime, true, startColor, endColor);
  }

  /**
   * @return  the endColor
   */
  public Color getEndColor() {
    return endColor;
  }

  /**
   * @return  the startColor
   */
  public Color getStartColor() {
    return startColor;
  }

  /**
   * @param  endColor  the endColor to set
   */
  public void setEndColor(Color endColor) {
    this.endColor = endColor;
    this.endColor.getComponents(this.endColorComponents);
  }

  /**
   * @param  startColor  the startColor to set
   */
  public void setStartColor(Color startColor) {
    this.startColor = startColor;
    this.startColor.getComponents(this.startColorComponents);
  }

  /**
   * Apply the color to the sprite.
   *
   * @param  sprite  the sprite to update
   */
  @Override
  public void updateSprite(Sprite sprite) {
    sprite.setColor(currColor);
  }

  /**
   * Update the current color based on interpolated percentage.
   *
   * @param  fractionEndColor  the percentage of endColor to use
   *                           (remainder comes from startColor)
   */
  @Override
  protected void interpolate(double fractionEndColor) {
    for (int channel = 0; channel != 4; ++channel) {
      currColorComponents[channel] = (float) (((1.0 -
              fractionEndColor) * startColorComponents[channel]) +
          (fractionEndColor * endColorComponents[channel]));
    }
    currColor = new Color(currColorComponents[0],
        currColorComponents[1], currColorComponents[2],
        currColorComponents[3]);
  }
}
