package fang2.transformers;

import fang2.attributes.Location2D;
import fang2.attributes.Vector2D;
import fang2.core.Sprite;
import fang2.core.Transformer;

/**
 * Shared root for all Bounce transformers. This class is a <i>decorator</i> of {@link VelocityProvider} objects. It <i>is</i> a velocity provider so that
 * it provides velocity (which moves a sprite it transforms automatically) and it <i>wraps around</i> a velocity provider so that it has something to change
 * when it is told to bounce. Subclasses determine when to bounce. 
 */
public class BounceTransformer
  implements Transformer, VelocityProvider {
  /**
   * the zero-threshold; if a value after damping is at or below this
   * value, treat it as zero
   */
  public static final double threshold = 0.00;
  
  /** how elastic are bouncing collisions? */
  private double elasticity;

  /** the velocity which this bouncer modifies */
   protected final VelocityProvider modified;

  /**
   * Construct a new {@link BounceTransformer}.
   *
   * @param  modified  the initial velocity of the sprite
   */
  public BounceTransformer(VelocityProvider modified) {
    this.modified = modified;
    this.elasticity = 1.0;
  }

  /**
   * Advance one frame.
   * @param dT
   * @see fang2.core.Transformer#advance(double)
   */
  public void advance(double dT) {
    modified.advance(dT);
  }

  /**
   * Get the amount of bounce back
   *
   * @return  the dampingValue
   */
  public double getElacticity() {
    return elasticity;
  }

  /**
   * Get the velocity being used
   * @return the velocity, provided by the underlying {@link VelocityProvider}.
   */
  public Vector2D getVelocity() {
    return modified.getVelocity();
  }

  /**
   * The pre-advance advance. Pass it down to the one we decorate
   * @param dT time since last update
   * @see fang2.core.Transformer#nonMaskableAdvance(double)
   */
  public void nonMaskableAdvance(double dT) {
    modified.nonMaskableAdvance(dT);
  }

  /**
   * Set the damping value (amount of bounce back) to the given value
   *
   * @param  elasticity  the dampingValue to set
   */
  public void setElasticity(double elasticity) {
    this.elasticity = elasticity;
  }

  /**
   * Set the velocity to the given value
   * @param componentVelocity a {@link Location2D} interpreted as a vector for purposes of setting velocity in the wrapped {@link VelocityProvider}.
   */
  public void setVelocity(Location2D componentVelocity) {
    modified.setVelocity(componentVelocity);
  }

  /**
   * Set the velocity to the given value
   * @param velocity the new velocity (vector) to set
   */
  public void setVelocity(Vector2D velocity) {
    modified.setVelocity(velocity);
  }

  /**
   * Update the sprite; passes everything down to the wrapped transformer.
   * @param sprite
   * @see fang2.core.Transformer#updateSprite(fang2.core.Sprite)
   */
  public void updateSprite(Sprite sprite) {
    modified.updateSprite(sprite);
  }

  /**
   * Reflect the x-coordinate in the velocity.
   */
  protected void bounceX() {
    Vector2D currentVelocity = getVelocity();
    double dX = currentVelocity.getX() * elasticity;
    double dY = currentVelocity.getY();
    if (Math.abs(dX) <= threshold) {
      dX = 0.0;
    }

    setVelocity(currentVelocity.setXYChange(-dX, dY));
  }

  /**
   * Reflect the y-coordinate in the velocity.
   */
  protected void bounceY() {
    Vector2D currentVelocity = getVelocity();
    double dX = currentVelocity.getX();
    double dY = currentVelocity.getY() * elasticity;
    if (Math.abs(dY) <= threshold) {
      dY = 0.0;
    }

    setVelocity(currentVelocity.setXYChange(dX, -dY));
  }
}
