package fang2.transformers;

import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;

import fang2.attributes.Box2D;
import fang2.attributes.Location2D;
import fang2.attributes.Vector2D;
import fang2.core.Sprite;

public class BounceSpriteTransformer
  extends BounceTransformer {
  /** the list of explicit targets to be tested */
  private final List<Sprite> targets;

  /** the time elapsed in the last frame */
  private double deltaT;

  private boolean useBoundingBox;

  /**
   * Construct a bounce transformer with a 0 velocity for the given
   * sprite.
   *
   * @param  bouncer  the sprite to bounce
   */
  public BounceSpriteTransformer(Sprite bouncer) {
    this(bouncer, new VelocityTransformer(new Vector2D()));
  }

  /**
   * Construct a {@link BounceSpriteTransformer} wrapped around the
   * given velocity provider and bouncing the given sprite.
   *
   * @param  bouncer   sprite to bounce
   * @param  modified  the velocity provider to work with
   */
  public BounceSpriteTransformer(Sprite bouncer,
    VelocityProvider modified) {
    super(modified);
    this.targets = new ArrayList<Sprite>();
    this.useBoundingBox = true;
  }

  /**
   * Create a new {@link BounceSpriteTransformer} with the given source
   * and target nodes. The controlled field is the sprite which does the
   * hitting. This is for the common case where there is just one
   * target; rather than having to construct and call add, this does it
   * all in one step.
   *
   * @param  bouncer   the controlled {@link Sprite}
   * @param  modified  the velocity modified by bouncing
   * @param  target    the target {@link Sprite}
   */
  public BounceSpriteTransformer(Sprite bouncer,
    VelocityProvider modified, Sprite target) {
    this(bouncer, modified);
    add(target);
  }

  /**
   * Add a target to the list of targets.
   *
   * @param  target  the sprite to add
   */
  public void add(Sprite target) {
    if ((target != null) && (targets.indexOf(target) < 0)) {
      targets.add(target);
    }
  }

  /**
   * Advance one frame; check for "impact".
   */
  @Override
  public void advance(double dT) {
    deltaT = dT; // hold the dT for updateSprite
    super.advance(dT);
  }

  /**
   * Checks for "impact" during the last advance.
   *
   * @param  bouncer  the sprite potentially changing direction
   */
  @Override
  public void updateSprite(Sprite bouncer) {
    for (Sprite t : targets) {
      if ((t != bouncer) && bouncer.intersects(t)) {
        bounce(bouncer, t,
          bouncer.getLocation().add(getVelocity().multiply(-deltaT)),
          deltaT);
      }
    }
    super.updateSprite(bouncer);
  }

  /**
   * Clear the list of target sprites.
   */
  public void clear() {
    targets.clear();
  }

  /**
   * Access the whole list of targets
   *
   * @return  the list of targets
   */
  public List<Sprite> getTargets() {
    return targets;
  }

  /**
   * Get the useBoundingBox flag. Using bounding boxes or rounder
   * collision volumes?
   *
   * @return  true if using bounding boxes, false otherwise.
   */
  public boolean getUseBoundingBox() {
    return this.useBoundingBox;
  }

  /**
   * Remove a target from the list of targets
   *
   * @param  target  the target to remove
   */
  public void remove(Sprite target) {
    targets.remove(target);
  }

  /**
   * set the useBoundingBox flag. Should collisions use just bounding
   * boxes or should they be more precise?
   *
   * @param  useBoundingBox  true or false
   */
  public void setUseBoundingBox(boolean useBoundingBox) {
    this.useBoundingBox = true;// useBoundingBox;
  }

  /**
   * Bounce one sprite off of the other. If {@link #useBoundingBox} is
   * true, then the collision is done just using the bounding boxes of
   * the two sprites. Otherwise a calculation is made about the
   * collision point and vector between the centers and then a more
   * complex vector calculation is made.
   *
   * @param  bouncer  the sprite that we are allowed to move and change
   *                  the velocity of. Our sprite.
   * @param  target   the other sprite. The one we are not allowed to
   *                  move (in this call).
   */
  private void bounce(Sprite bouncer, Sprite target,
    Location2D previous, double dT) {
    if (useBoundingBox) {
      double dX = bouncer.getX() - target.getX();
      double dY = bouncer.getY() - target.getY();
      Box2D targetBounds = target.getBounds2D();
      Location2D testLocation =
        (targetBounds.contains(bouncer.getLocation()))
        ? previous : bouncer.getLocation();
      int outcode = targetBounds.outcode(testLocation);
      if (((outcode & (Rectangle2D.OUT_LEFT | Rectangle2D.OUT_RIGHT)) !=
            0) &&
          (Math.signum(dX) != Math.signum(getVelocity().getX()))) {
        bounceX();
        if ((outcode & Rectangle2D.OUT_LEFT) != 0) {
          if (bouncer.getMaxX() > target.getMinX()) {
            bouncer.translateX(target.getMinX() - bouncer.getMaxX());
          }
        }
        if ((outcode & Rectangle2D.OUT_RIGHT) != 0) {
          if (bouncer.getMinX() < target.getMaxX()) {
            bouncer.translateX(target.getMaxX() - bouncer.getMinX());
          }
        }
      }
      if (((outcode & (Rectangle2D.OUT_BOTTOM | Rectangle2D.OUT_TOP)) !=
            0) &&
          (Math.signum(dY) != Math.signum(getVelocity().getY()))) {
        bounceY();
        if ((outcode & Rectangle2D.OUT_TOP) != 0) {
          if (bouncer.getMaxY() > target.getMinY()) {
            bouncer.translateY(target.getMinY() - bouncer.getMaxY());
          }
        }
        if ((outcode & Rectangle2D.OUT_BOTTOM) != 0) {
          if (bouncer.getMinY() < target.getMaxY()) {
            bouncer.translateY(target.getMaxY() - bouncer.getMinY());
          }
        }
      }
    } else {// center-to-center collision (angled bounce like a ball)
/*  This is currently broken; I am not sure what I am doing wrong but I
 * have writing that needs to get done.   Location2D lvelocity = new
 * Location2D(getVelocity());   Location2D backout =
 * lvelocity.multiply(-(1.0 / 32.0)); bouncer.translate(backout);
 *
 * while ((backout.getMagintude() < 1.0) && bouncer.intersects(target)) {
 *    bouncer.translate(backout); backout = backout.multiply(2.0);   }
 *
 * Location2D centerVector = new Location2D(target.getX() -
 * bouncer.getX(), target.getY() - bouncer.getY()).normalize();
 * LineSprite centers = new LineSprite(bouncer.getX(), bouncer.getY(),
 * target.getX(), target.getY());
 * centers.setColor(Game.getColor("red"));
 * Game.getCurrentGame().addSprite(centers);   // Break velocity into
 * normal and tangent components   Location2D normalVelocity =
 * Location2D.componentVector(lvelocity,       centerVector); Location2D
 * impactPoint =
 * bouncer.getLocation().add(centerVector.multiply((bouncer.getWidth() /
 * 2) / (target.getX() - bouncer.getX()) *
 * Location2D.distance(bouncer.getX(), bouncer.getY(), target.getX(),
 * target.getY())));   LineSprite normal = new
 * LineSprite(impactPoint.getX(), impactPoint.getY(), impactPoint.getX()
 * + normalVelocity.getX(), impactPoint.getY() + normalVelocity.getY());
 *  normal.setColor(Game.getColor("blue"));
 * Game.getCurrentGame().addSprite(normal);      Location2D
 * tangentVelocity = lvelocity.subtract(normalVelocity);
 *
 * LineSprite tangent = new LineSprite(impactPoint.getX(),
 * impactPoint.getY(), impactPoint.getX() + tangentVelocity.getX(),
 * impactPoint.getY() + tangentVelocity.getY());
 * tangent.setColor(Game.getColor("green"));
 * Game.getCurrentGame().addSprite(tangent);      Location2D
 * resultantVelocity = tangentVelocity.subtract(normalVelocity);
 * LineSprite resultant = new LineSprite(impactPoint.getX(),
 * impactPoint.getY(), impactPoint.getX() + resultantVelocity.getX(),
 * impactPoint.getY() + resultantVelocity.getY());
 * resultant.setColor(Game.getColor("yellow"));
 * Game.getCurrentGame().addSprite(resultant);
 * setVelocity(tangentVelocity.subtract(normalVelocity));
 */
    }
  }
}
