package fang2.transformers;

import fang2.attributes.Vector2D;
import fang2.core.Sprite;

public class BounceInsideRectangleTransformer
  extends BounceTransformer {
  /** the rectangle on the screen (in parent screen coordinates) */
  private final double maxX;
  private final double maxY;
  private final double minX;
  private final double minY;

  /**
   * Construct a new {@link BounceInsideRectangleTransformer}. Bounce
   * the sprite inside of the screen. Initial velocity is (0.0 degrees,
   * 0.0 screens)
   *
   * <p><b>Precondition</b> The sprite must be <i>inside</i> the given
   * rectangle when this transformer is added. If it is not, behavior
   * will undoubtedly not be what the programmer wants.
   */
  public BounceInsideRectangleTransformer() {
    this(new VelocityTransformer(new Vector2D()));
  }

  /**
   * Construct a new {@link BounceInsideRectangleTransformer}. Bounce
   * the sprite inside of the given rectangle.
   *
   * <p><b>Precondition</b> The sprite must be <i>inside</i> the given
   * rectangle when this transformer is added. If it is not, behavior
   * will undoubtedly not be what the programmer wants.
   *
   * @param  minX      the minimum x-coordinate of the sprite
   * @param  minY      the minimum y-coordinate of the sprite
   * @param  maxX      the maximum x-coordinate of the sprite
   * @param  maxY      the maximum y-coordinate of the sprite
   * @param  modified  the initial velocity of the sprite
   */
  public BounceInsideRectangleTransformer(double minX, double minY,
    double maxX, double maxY, VelocityProvider modified) {
    super(modified);
    this.minX = minX;
    this.minY = minY;
    this.maxX = maxX;
    this.maxY = maxY;
  }

  /**
   * Construct a new {@link BounceInsideRectangleTransformer}. Bounce
   * the sprite inside of the screen.
   *
   * <p><b>Precondition</b> The sprite must be <i>inside</i> the given
   * rectangle when this transformer is added. If it is not, behavior
   * will undoubtedly not be what the programmer wants.
   *
   * @param  modified  the initial velocity of the sprite
   */
  public BounceInsideRectangleTransformer(VelocityProvider modified) {
    this(0.0, 0.0, 1.0, 1.0, modified);
  }

  /**
   * Bounce if necessary. In any case, call the super advance to keep
   * the velocity and offset calculations correct
   */
  @Override
  public void advance(double dT) {
    super.advance(dT);
  }

  /**
   * Update the location of the sprite. Bump it into the rectangle if
   * necessary (keep from getting caught on an edge.
   */
  @Override
  public void updateSprite(Sprite bouncer) {
    super.updateSprite(bouncer);
    if ((bouncer.getMinX() < minX) || (maxX < bouncer.getMaxX())) {
      bounceX();
    }
    if ((bouncer.getMinY() < minY) || (maxY < bouncer.getMaxY())) {
      bounceY();
    }
    
    if (bouncer.getMinX() < minX) {
      bouncer.translateX(minX - bouncer.getMinX());
    }
    if (maxX < bouncer.getMaxX()) {
      bouncer.translateX(maxX - bouncer.getMaxX());
    }
    if (bouncer.getMinY() < minY) {
      bouncer.translateY(minY - bouncer.getMinY());
    }
    if (maxY < bouncer.getMaxY()) {
      bouncer.translateY(maxY - bouncer.getMaxY());
    }
  }
}
