package fang2.transformers;

import fang2.attributes.Vector2D;
import fang2.core.Game;
import fang2.core.Sprite;
import fang2.core.Transformer;
import fang2.sprites.RectangleSprite;

import java.util.ArrayList;
import java.util.List;

/**
 * Bounce controlled sprite off of <i>all</i> sprites of a given class.
 * Thus if a ball should bounce off of all {@link RectangleSprite}
 * objects, use this {@link Transformer}. The transformer will then
 * automatically gather up all of the matching sprites and test them for
 * collision.
 */
public class BounceClassTransformer
  extends BounceSpriteTransformer {
  /** the list of explicit targets to be tested */
  private final List<Sprite> spriteTargets;

  /**
   * the list of classes of sprites with which this transformer
   * interacts
   */
  private final List<Class<? extends Sprite>> targetClasses;

  /**
   * Construct a new {@link BounceClassTransformer} to bounce bouncer
   * off of all objects of a type or types to be determined (see {@link
   * #add(Class)}. this method creates its own velocity transformer
   * which is reversed when bouncing off of an object. All objects of
   * the given type are automatically gathered by this class.
   *
   * @param  bouncer  the controlled sprite, the thing being tested for
   *                  bouncing
   */
  public BounceClassTransformer(Sprite bouncer) {
    this(bouncer, new VelocityTransformer(new Vector2D()));
  }

  /**
   * Construct a new {@link BounceClassTransformer} to bounce bouncer
   * off of all objects of targetClass type, using a newly created
   * velocity. Velocity is reversed when bouncing off of an object. All
   * objects of the given type are automatically gathered by this class.
   *
   * @param  bouncer      the controlled sprite, the thing being tested
   *                      for bouncing
   * @param  targetClass  the class with which to initialize the list of
   *                      target classes.
   */
  public BounceClassTransformer(Sprite bouncer,
    Class<? extends Sprite> targetClass) {
    this(bouncer, new VelocityTransformer(new Vector2D()),
      targetClass);
  }

  /**
   * Construct a new {@link BounceClassTransformer} to bounce bouncer
   * off of all objects of some type (or types), using velocity as the
   * velocity. Velocity is reversed when bouncing off of an object. All
   * objects of the given type are automatically gathered by this class.
   *
   * @param  bouncer   the controlled sprite, the thing being tested for
   *                   bouncing
   * @param  velocity  a {@link VelocityProvider}, something that has a
   *                   velocity we can change (to do the bounce)
   */
  public BounceClassTransformer(Sprite bouncer,
    VelocityProvider velocity) {
    this(bouncer, velocity, null);
  }

  /**
   * Construct a new {@link BounceClassTransformer} to bounce bouncer
   * off of all objects of targetClass type, using velocity as the
   * velocity. Velocity is reversed when bouncing off of an object. All
   * objects of the given type are automatically gathered by this class.
   *
   * @param  bouncer      the controlled sprite, the thing being tested
   *                      for bouncing
   * @param  velocity     a {@link VelocityProvider}, something that has
   *                      a velocity we can change (to do the bounce)
   * @param  targetClass  the class with which to initialize the list of
   *                      target classes.
   */
  public BounceClassTransformer(Sprite bouncer,
    VelocityProvider velocity, Class<? extends Sprite> targetClass) {
    super(bouncer, velocity);
    targetClasses = new ArrayList<Class<? extends Sprite>>();
    spriteTargets = new ArrayList<Sprite>();
    if (targetClass != null) {
      targetClasses.add(targetClass);
    }
  }

  /**
   * Add a target to the list of targets.
   *
   * @param  target  the sprite to add
   */
  @Override
  public void add(Sprite target) {
    if ((target != null) && (spriteTargets.indexOf(target) < 0)) {
      spriteTargets.add(target);
    }
  }

  /**
   * Add a class to the list of classes with which the bouncer
   * interacts.
   *
   * @param  targetClass  the class to add
   */
  public void add(Class<? extends Sprite> targetClass) {
    if (targetClasses.indexOf(targetClass) < 0) {
      targetClasses.add(targetClass);
    }
  }

  /**
   * Advance to the next frame. Collect up all the sprites of the given
   * type(s) and calls {@link
   * BounceSpriteTransformer#advance(double)}.
   *
   * @param  dT  time since last advance
   */
  @Override
  public void advance(double dT) {
    collectUpSprites();
    super.advance(dT);
  }

  /**
   * Remove a target from the list of targets
   *
   * @param  target  the target to remove
   */
  @Override
  public void remove(Sprite target) {
    spriteTargets.remove(target);
  }

  
  /**
   * Remove the given class from the list of all classes with which this
   * transformer will collide the bouncer.
   *
   * @param  targetClass  the class to remove
   */
  public void remove(Class<? extends Sprite> targetClass) {
    targetClasses.remove(targetClass);
  }

  /**
   * Collect up all of the sprites in the current game which are of the
   * given type or types.
   */
  private void collectUpSprites() {
    List<Sprite> targets = getTargets();
    targets.clear();
    targets.addAll(spriteTargets);

    if (!targetClasses.isEmpty()) {
      Game game = Game.getCurrentGame();
      for (Sprite sprite : game.getCanvas().getAllSprites()) {
        for (Class<? extends Sprite> currClass : targetClasses) {
          if (currClass.isInstance(sprite)) {
            targets.add(sprite);
            break;
          }
        }
      }
    }
  }
}
