package fang2.transformers;

import fang2.attributes.Location2D;
import fang2.attributes.Vector2D;
import fang2.core.Sprite;
import fang2.core.TransformerAdapter;

/** 
 * Transformer representing constant acceleration. 
 */
public class AccelerationTransformer
  extends TransformerAdapter 
  implements VelocityProvider {
  /** the acceleration vector screens/sec^2 */
  private Vector2D acceleration;

  /** the actual velocity provider; this provides acceleration for the velocity (through setVelocity) */
  private final VelocityProvider velocityNG;

  /**
   * Construct a new {@link AccelerationTransformer} with the given
   * initial velocity and initial acceleration
   * @param  acceleration  initial acceleration
   * @param  velocityNG      initial velocity
   */
  public AccelerationTransformer(Vector2D acceleration,
    VelocityProvider velocityNG) {
    if (velocityNG != null) {
      this.velocityNG = velocityNG;
    } else {
      this.velocityNG = new VelocityTransformer(new Vector2D());
    }
    this.acceleration = acceleration;
  }

  /**
   * Update the velocity according to the acceleration. Acceleration is
   * a vector so .times multiplies it. Scale and add.
   *
   * @param  dT  time since last call to advance
   */
  @Override
  public void advance(double dT) {
    velocityNG.setVelocity(velocityNG.getVelocity().add(acceleration.multiply(dT)));
    velocityNG.advance(dT);
  }

  /**
   * Get the current acceleration
   *
   * @return  the acceleration screens/second
   */
  public Vector2D getAcceleration() {
    return acceleration;
  }

  
  /**
   * Get the velocity vector. 
   * @return the {@link Vector2D} of the velocity of this transformer
   * @see fang2.transformers.VelocityTransformer#getVelocity()
   */
  public Vector2D getVelocity() {
    return velocityNG.getVelocity();
  }


  /**
   * @param dT
   * @see fang2.transformers.VelocityTransformer#nonMaskableAdvance(double)
   */
  @Override
  public void nonMaskableAdvance(double dT) {
    velocityNG.nonMaskableAdvance(dT);
  }


  /**
   * Set the acceleration to the screens/second value given
   *
   * @param  acceleration  the acceleration to set
   */
  public void setAcceleration(Vector2D acceleration) {
    this.acceleration = acceleration;
  }


  /**
   * @param componentVelocity
   * @see fang2.transformers.VelocityTransformer#setVelocity(fang2.attributes.Location2D)
   */
  public void setVelocity(Location2D componentVelocity) {
    velocityNG.setVelocity(componentVelocity);
  }


  /**
   * @param velocity
   * @see fang2.transformers.VelocityTransformer#setVelocity(fang2.attributes.Vector2D)
   */
  public void setVelocity(Vector2D velocity) {
    velocityNG.setVelocity(velocity);
  }


  /**
   * @param sprite
   * @see fang2.core.TransformerAdapter#updateSprite(fang2.core.Sprite)
   */
  @Override
  public void updateSprite(Sprite sprite) {
    super.updateSprite(sprite);
    velocityNG.updateSprite(sprite);
  }

}
