package fang2.sprites;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import fang2.core.Sprite;

/**
 * This class converts text into a sprite. The text can contain multiple
 * lines by putting in newlines. Also, different true type fonts may be
 * used. During development, the fonts may be loaded automatically from
 * the system, but when publishing the game, all fonts used should be
 * copied into the tipgame.fonts package and should have the extension
 * .ttf In Windows, fonts can typically be found in C:\WINDOWS\Fonts.
 * The files corresponding to the fonts used can be copied into
 * tipgame.fonts to make sure the target computers are able to properly
 * display the font (since not all computers have the same fonts
 * available by default). This class differs from the PrettyStringSprite
 * in that it is much faster and sometimes does not represent the font
 * as cleanly.
 *
 * @author  Jam Jenkins
 */

public class StringSprite
  extends Sprite {
  /** the horizontal spacing between characters */
  private static final double ADVANCE = 0.1;

  /** cache of letter samples for each font used */
  private static final 
    Map<Font, Map<Character, PrettyStringSprite>> CACHE =
      new HashMap<Font, Map<Character, PrettyStringSprite>>();

  /** the vertical margin between lines of text */
  private static final double LEADING = 0.2;

  /** the horizontal distance of a space */
  private static final double SPACE_WIDTH = 0.2;

  /** how many character spaces a tab represents */
  private static final double TAB_WIDTH = 2.5;

  /**
   * rotation, in radians, of the string on screen in cannonical
   * position
   */
  private double baselineAngle = 0;

  /**
   * deep refers to whether this method is being called by a user of the
   * class or by this class internally. Because StringSprites need to
   * maintain consistent heights when changing text, scale must be
   * handled differently than in the normal way. This is because the
   * scale can actually change with a change in the text displayed.
   */
  private int deep = 0;

  /** how this text is rendered */
  private Font font = new Font(null, Font.PLAIN, 12);

  /** base height of a line of text */
  private double height;

  /**
   * for left, center, and right justification which is -1, 0, and 1
   * respectively
   */
  private final Point2D.Double justify = new Point2D.Double(0, 0);

  /**
   * true indicates the text should retain its original width and
   * height, false means the text should be expanded to fit a square
   */
  private boolean keepAspect = true;

  /** sample letters in the current font used */
  private Map<Character, PrettyStringSprite> letterRepository;

  /** max width of any character in this font */
  private double maxCharacterWidth;

  /**
   * whether to make all characters take the same number of space or be
   * variable width
   */
  private boolean monospaced = false;

  /** the current sequence of characters */
  private String text = "";

  /** makes a line at the base line of the text */
  private boolean underlined = false;

  /**
   * Construct a new StringSprite with a height of 1 screen, an empty
   * string, no initial rotation, which keeps its aspect ratio.
   */
  public StringSprite() {
    this("", 0.0, true);
  }

  /**
   * Construct a new StringSprite with the given text, a height of one
   * screen, no initial rotation which keeps its aspect ratio
   *
   * @param  text  initially displayed text
   */
  public StringSprite(String text) {
    this(text, 0.0, true);
  }

  /**
   * Construct a new StringSprite with the given text, a height scale of
   * 1.0 screen, 0.0 radian rotation, and the given value for keeping
   * its aspect ratio.
   *
   * @param  text        initial text
   * @param  keepAspect  true indicates to leave the aspect ratio of the
   *                     text unchanged while false means to shrink or
   *                     expand the text to fit into a square
   */
  public StringSprite(String text, boolean keepAspect) {
    this(text, 0.0, keepAspect);
  }

  /**
   * Construct a new StringSprite with the given text, scale, base
   * rotation, and aspect.
   *
   * @param  text           initial text
   * @param  baselineAngle  an initial offset for the rotation which
   *                        does not affect its original orientation
   * @param  keepAspect     true indicates to leave the aspect ratio of
   *                        the text unchanged while false means to
   *                        shrink or expand the text to fit into a
   *                        square
   */
  public StringSprite(String text, double baselineAngle,
    boolean keepAspect) {
    super();
    loadSampleLetters();
    setText(text);
    setScale(1.0);
    this.baselineAngle = baselineAngle;
    setKeepAspect(keepAspect);
  }

  /**
   * makes the position represent the bottom most position of the string
   */
  public void bottomJustify() {
    justify.y = 1;
  }

  /**
   * makes the position represent the center position of the string.
   * StringSprites are center justified by default.
   */
  public void centerJustify() {
    justify.x = 0;
    justify.y = 0;
  }

  /**
   * gets whether the dimensions should or should not be manipulated to
   * fill a square
   *
   * @return  true if the normal dimensions remain unchanged, false
   *          means the text is being expanded to fit a square
   */
  public boolean getAspect() {
    return keepAspect;
  }

  /**
   * gets the style of the current text
   *
   * @return  the font being used to generate the text
   */
  public Font getFont() {
    return font;
  }

  /**
   * gets the family name of the font
   *
   * @return  the family name of the font
   */
  public String getFontFamilyName() {
    return font.getFamily();
  }

  /**
   * Get the height of the StringSprite in screens.
   *
   * @return  height, in screens, of the current value of the
   *          StringSprite
   */
  @Override
  public double getHeight() {
    ++deep;
    double h = getScale() * getUnscaledHeight();
    --deep;
    return h;
  }

  /**
   * gets the full height that this StringSprite could have with any
   * given text
   *
   * @return  the maximum height the StringSprite will take for any
   *          given sequence of characters
   */
  public double getLineHeight() {
    ++deep;
    double h = height * getScale();
    --deep;
    return h;
  }

  /**
   * gets the width of a given line of text in this sprite. Invalid
   * lines return width 0.
   *
   * @param   index  the line number with 0 being the first line
   *
   * @return  the width, in screens, of the given line of the string.
   */
  public double getLineWidth(int index)
    throws IndexOutOfBoundsException {
    ++deep;
    double w = getScale() * getUnscaledLineWidth(index);
    --deep;
    return w;
  }

  /**
   * gets the rotation of this sprite. The rotation returned is minus
   * the original base rotation
   *
   * @see  fang2.core.Sprite#getRotation()
   */
  @Override
  public double getRotation() {
    ++deep;
    double adjusted = super.getRotation() - baselineAngle;
    while (adjusted < 0) {
      adjusted += Math.PI * 2;
    }
    double r = super.getRotation();
    --deep;
    return r;
  }

  /**
   * Get the height scale of the StringSprite; note that this refers to
   * the height, in screens, of a single line of text.
   *
   * @return  current height scale in screens
   */
  @Override
  public double getScale() {
    if (deep > 0) {
      return super.getScale();
    }
    return getMaxDimension();
  }

  /**
   * gets the rotated bounds of the shape. Note: unlike in other
   * Sprites, this does not give the outline of the individual shapes
   * used to make the text.
   *
   * @return  the smallest rectangle fitting around the text
   */
  @Override
  public Shape getShape() {
    ++deep;
    Rectangle2D.Double bounds;
    if (keepAspect) {
      bounds = new Rectangle2D.Double(0, 0, getUnscaledWidth(),
          getUnscaledHeight());
    } else {
      bounds = new Rectangle2D.Double(0, 0,
          Math.max(getUnscaledWidth(), getUnscaledHeight()),
          Math.max(getUnscaledWidth(), getUnscaledHeight()));
    }
    if (justify.x == 0) {
      bounds.x = -bounds.getCenterX();
    } else if (justify.x == 1) {
      bounds.x = -bounds.width;
    }
    if (justify.y == 0) {
      bounds.y = -bounds.getCenterY();
    } else if (justify.y == 1) {
      bounds.y = -bounds.height;
    }
    GeneralPath path = new GeneralPath(bounds);
    path.transform(transform);
    --deep;
    return path;
  }

  /**
   * Current string contents of this StringSprite.
   *
   * @return  current string displayed by this StringSprite.
   */
  public String getText() {
    return text;
  }

  /**
   * Get the width of the StringSprite in screens.
   *
   * @return  width, in screens, of the StringSprite with its current
   *          text
   */
  @Override
  public double getWidth() {
    ++deep;
    double w = getScale() * getUnscaledWidth();
    --deep;
    return w;
  }

  /**
   * Is the font displayed using the bold variant?
   *
   * @return  true if font is bold, false otherwise
   */
  public boolean isBold() {
    return font.isBold();
  }

  /**
   * Is the font displayed an italic variant?
   *
   * @return  true if font is italic; false otherwise
   */
  public boolean isItalic() {
    return font.isItalic();
  }

  /**
   * Is the font displayed an italic variant?
   *
   * @return  tru if font is italic; false otherwise
   */
  @Deprecated
  public boolean isItalicized() {
    return isItalic();
  }

  /**
   * Is the font being displayed with an underline?
   *
   * @return  true if underline is used; false otherwise
   */
  public boolean isUnderlined() {
    return underlined;
  }

  /**
   * Move the reference point for the string (its location) to the left
   * edge of the StringSprite.
   */
  public void leftJustify() {
    justify.x = -1;
  }

  /**
   * paints the text by transforming the brush and placing the
   * characters from the letterRepository
   *
   * @param  brush  the drawing instrument
   */
  @Override
  public void paint(Graphics2D brush) {
    AffineTransform original = brush.getTransform();
    brush.transform(transform);
    if (!keepAspect) {
      brush.transform(AffineTransform.getScaleInstance(
          getMaxDimension() / getWidth(),
          getMaxDimension() / getHeight()));
    }
    brush.setColor(color);
    brush.translate(0,
      (-getUnscaledHeight() / 2.0 * (justify.y + 1)) + (height / 2.0));
    draw(brush);
    brush.setTransform(original);
    if (outline != null) {
      brush.transform(transform);
      brush.setColor(outlineColor);
      brush.fill(outline);
      brush.setTransform(original);
    }
  }

  /**
   * Move the reference point for the string (its location) to the right
   * edge of the StringSprite.
   */
  public void rightJustify() {
    justify.x = 1;
  }

  /**
   * scales the StringSprite by a given factor relative to it's current
   * scale.
   *
   * @param  s  the scaling factor
   */
  @Override
  public void scale(double s) {
    if (deep > 0) {
      super.scale(s);
    } else {
      setMaxDimension(s * getMaxDimension());
    }
  }

  /**
   * sets the thickness of the lettering
   *
   * @param  bold  true indicates thick lettering, false indicates
   *               normal thickness
   */
  public void setBold(boolean bold) {
    if (font.isBold() == bold) {
      return;
    }
    if (bold) {
      font = font.deriveFont(font.getStyle() & Font.BOLD);
    } else {
      font = font.deriveFont(font.getStyle() & ~Font.BOLD);
    }
    loadSampleLetters();
  }

  /**
   * sets the color of this sprite
   *
   * @param  color  the shade of the sprite
   */
  @Override
  public void setColor(Color color) {
    this.color = color;
    for (PrettyStringSprite sprite : letterRepository.values()) {
      sprite.setColor(color);
    }
  }

  /**
   * sets the style of the current text
   *
   * @param  font  the font being used to generate the text
   */
  public void setFont(Font font) {
    this.font = font;
    loadSampleLetters();
  }

  /**
   * sets the family name of the font
   *
   * @param  familyName  the family name of the font
   */
  public void setFontFamilyName(String familyName) {
    if ((familyName == font.getFamily()) ||
        ((familyName != null) && (font.getFamily() != null) &&
          familyName.equals(font.getFamily()))) {
      return;
    }
    font = PrettyStringSprite.getFont(font.getStyle(), familyName);
    font = font.deriveFont(font.getSize());
    loadSampleLetters();
  }

  /**
   * Set the height of the entire sprite. Use setLineHeight to set the
   * height of each line in the sprite. For example, calling
   * setHeight(0.2) on a sprite with 2 lines is the same as calling
   * setLineHeight(0.1) on the same sprite with 2 lines.
   *
   * @param  lineHeight  the height, in screens, of the entire sprite
   */
  public void setHeight(double lineHeight) {
    if (getText().length() == 0) {
      setText(" ");
    }

    ++deep;
    setScale(lineHeight / getText().split("\n").length);
    // setText(originalText);
    --deep;
  }

  /**
   * sets the slant of the text
   *
   * @param  italics  true indicates slant, false is for no slant
   */
  public void setItalic(boolean italics) {
    if (font.isItalic() == italics) {
      return;
    }
    if (italics) {
      font = font.deriveFont(font.getStyle() & Font.ITALIC);
    } else {
      font = font.deriveFont(font.getStyle() & ~Font.ITALIC);
    }
    loadSampleLetters();
  }

  /**
   * sets the slant of the text
   *
   * @param  italics  true indicates slant, false is for no slant
   */
  @Deprecated
  public void setItalicized(boolean italics) {
    setItalic(italics);
  }

  /**
   * sets whether the dimensions should or should not be manipulated to
   * fill a square
   *
   * @param  keepAspect  true indicates the normal dimensions should
   *                     remain unchanged, false means the text should
   *                     be expanded to fit a square
   */
  public void setKeepAspect(boolean keepAspect) {
    this.keepAspect = keepAspect;
  }

  /**
   * Set the height of a single line of text.
   *
   * @param  height  height, in screens, of a single line of text
   */
  public void setLineHeight(double height) {
    ++deep;
    setScale(height);
    --deep;
  }

  /**
   * sets whether this text should be represented with fixed or variable
   * width text.
   *
   * @param  monospaced  true indicates fixed width text, false means
   *                     variable width text
   */
  public void setMonospaced(boolean monospaced) {
    this.monospaced = monospaced;
  }

  /**
   * Set the scale of the height of a line of text to the given value,
   * in screens.
   *
   * @param  scale  the new line height, in screens
   */
  @Override
  public void setScale(double scale) {
    if (deep > 0) {
      super.setScale(scale);
    } else {
      setMaxDimension(scale);
    }
  }

  /**
   * Set the displayed text of the sprite to the given value.
   *
   * @param  text  new text to display
   */
  public void setText(String text) {
    setText(text, keepAspect, underlined);
  }

  /**
   * Update displayed text with the given value and the given aspect
   * ratio protection.
   *
   * @param  text    new text to display
   * @param  aspect  true indicates original width and height should be
   *                 retained, false indicates the text should be
   *                 resized to fit within a square
   */
  public void setText(String text, boolean aspect) {
    setText(text, aspect, underlined);
  }

  /**
   * Update text with the given aspect ratio protection and underline
   * values.
   *
   * @param  text       new text to display
   * @param  aspect     false means the text should be expanded to fit a
   *                    square, true indicates the text should retain
   *                    its original shape
   * @param  underline  true indicates there should be a line at the
   *                    baseline, false indicates no line
   */
  public void setText(String text, boolean aspect, boolean underline) {
    this.underlined = underline;
    this.keepAspect = aspect;
    this.text = text;
  }

  /**
   * sets whether there should or should not be a line at the baseline
   * of the text
   *
   * @param  underline  true indicates there should be a line, false
   *                    indicates no line should be there
   */
  public void setUnderlined(boolean underline) {
    this.underlined = underline;
  }

  /**
   * Set the width of this sprite such that the width of the widest line
   * of text is the given screen width. Width is set based on the
   * current contents of the StringSprite; resetting the text may change
   * the width of the sprite.
   *
   * @param  width  width, in screens, that the currently longest line
   *                should have
   */
  public void setWidth(double width) {
    String originalText = getText();
    if (getText().length() == 0) {
      setText(" ");
    }

    ++deep;
    scale(width / getWidth());
    setText(originalText);
    --deep;
  }

  @Override
  public void showOutline() {
    shape = new GeneralPath(getShape());
    super.showOutline();
  }

  /**
   * makes the position represent the top most position of the string
   */
  public void topJustify() {
    justify.y = -1;
  }

  /**
   * {@inheritDoc fang2.core.Sprite#toString()} Augmented with the
   * current text value.
   *
   * @return  string representing current value of sprite
   */
  @Override
  public String toString() {
    return super.toString() + " text = \"" + getText() + "\"";
  }

  /**
   * Internal draw routine.
   *
   * @param  brush  the graphics context in which to draw the glyphs
   */
  private void draw(Graphics2D brush) {
    int lineNumber = 0;
    double lateralMovement = -getUnscaledLineWidth(lineNumber) / 2 *
      (justify.x + 1);
    brush.translate(lateralMovement, 0);
    for (char letter : text.toCharArray()) {
      if (letter == '\n') {
        brush.translate(-lateralMovement, height + (LEADING * height));
        lineNumber++;
        try
        {        
	   lateralMovement = -getUnscaledLineWidth(lineNumber) / 2 *
      	      (justify.x + 1);
        }
        catch(IndexOutOfBoundsException e)
        {
           
        }
        brush.translate(lateralMovement, 0);
        continue;
      }
      if (letter == '\t') {
        double spaces = TAB_WIDTH;
        if (monospaced) {
          spaces = Math.round(spaces);
        }
        brush.transform(AffineTransform.getTranslateInstance(
            (maxCharacterWidth + (maxCharacterWidth * ADVANCE)) *
            spaces, 0));
        lateralMovement += (maxCharacterWidth +
            (maxCharacterWidth * ADVANCE)) * spaces;
        continue;
      }
      if (letter == ' ') {
        double fractionalWidth = SPACE_WIDTH;
        if (monospaced) {
          fractionalWidth = 1;
        }
        brush.transform(AffineTransform.getTranslateInstance(
            (maxCharacterWidth * ADVANCE) +
            (maxCharacterWidth * fractionalWidth), 0));
        lateralMovement += (maxCharacterWidth * ADVANCE) +
          (maxCharacterWidth * fractionalWidth);
        continue;
      }
      if (monospaced) {
        lateralMovement += maxCharacterWidth;
      } else {
        lateralMovement += getSprite(letter).getWidth();
      }
      lateralMovement += maxCharacterWidth * ADVANCE;
      if (monospaced) {
        brush.translate(maxCharacterWidth / 2, 0);
      } else {
        brush.translate(getSprite(letter).getWidth() / 2, 0);
      }
      if (underlined) {
        double uWidth = maxCharacterWidth * ADVANCE;
        if (monospaced) {
          uWidth += maxCharacterWidth;
        } else {
          uWidth += getSprite(letter).getWidth();
        }
        Rectangle2D.Double line = new Rectangle2D.Double(-uWidth / 2,
            5 / 12.0, uWidth, 1 / 12.0);
        brush.fill(line);
      }
      getSprite(letter).setColor(color);
      getSprite(letter).paint(brush);
      if (monospaced) {
        brush.translate(maxCharacterWidth / 2, 0);
      } else {
        brush.translate(getSprite(letter).getWidth() / 2, 0);
      }
      brush.translate(maxCharacterWidth * ADVANCE, 0);
    }
  }

  /**
   * Get the size of the sprite as if the scale were 1.0. .
   *
   * @return  current largest dimension of the StringSprite
   */
  private double getMaxDimension() {
    ++deep;
    double max = Math.max(getWidth(), getHeight());
    --deep;
    return max;
  }

  /**
   * gets the PrettyStringSprite corresponding to the character given
   *
   * @param   letter  the letter to translate
   *
   * @return  the corresponding PrettyStringSprite
   */
  private PrettyStringSprite getSprite(char letter) {
    if (!letterRepository.containsKey(letter)) {
      PrettyStringSprite letterSprite = new PrettyStringSprite("" +
          letter, true);
      letterSprite.setFont(font.getFamily());
      letterSprite.setBold(font.isBold());
      letterSprite.setItalicized(font.isItalic());
      letterSprite.setHeight(1);
      letterSprite.setColor(color);
      height = letterSprite.getHeight();
      letterRepository.put(letter, letterSprite);
    }
    return letterRepository.get(letter);
  }

  /**
   * Get the height in screens of the sprite as if the scale were 1.0
   *
   * @return  the height, in screens, of the current StringSprite
   *          contents
   */
  private double getUnscaledHeight() {
    return (height * (1 + LEADING) * text.split("\n").length) -
      (LEADING * height);
  }

  /**
   * gets the untransformed width of a given line of text in this
   * sprite. Invalid lines return width 0.
   *
   * @param   index  the line number with 0 being the first line
   *
   * @return  the untransformed horizontal span of the given line in
   *          pixels
   */
  private double getUnscaledLineWidth(int index)
    throws IndexOutOfBoundsException {
    String[] lines = text.split("\n");
    if ((index < 0) || (index >= lines.length)) {
      throw new IndexOutOfBoundsException();
    }
    String line = text.split("\n")[index];

    double lineWidth = -ADVANCE * maxCharacterWidth;
    for (char letter : line.toCharArray()) {
      if (letter == '\t') {
        if (monospaced) {
          lineWidth += Math.round(TAB_WIDTH) *
            (maxCharacterWidth + (ADVANCE * maxCharacterWidth));
        } else {
          lineWidth += TAB_WIDTH *
            (maxCharacterWidth + (ADVANCE * maxCharacterWidth));
        }
      } else if (letter == ' ') {
        if (monospaced) {
          lineWidth += maxCharacterWidth +
            (ADVANCE * maxCharacterWidth);
        } else {
          lineWidth += (SPACE_WIDTH * maxCharacterWidth) +
            (ADVANCE * maxCharacterWidth);
        }
      } else {
        if (monospaced) {
          lineWidth += maxCharacterWidth +
            (ADVANCE * maxCharacterWidth);
        } else {
          lineWidth += getSprite(letter).getWidth() +
            (ADVANCE * maxCharacterWidth);
        }
      }
    }
    return lineWidth;
  }

  /**
   * gets the width of this Sprite in pixels before adding the transform
   *
   * @return  the untransformed horizontal span in pixels
   */
  private double getUnscaledWidth() {
    double lineWidth = getUnscaledLineWidth(0);
    for (int i = 1; i < text.split("\n").length; i++) {
      lineWidth = Math.max(lineWidth, getUnscaledLineWidth(i));
    }
    return lineWidth;
  }

  /**
   * loads all numbers and letters into the letterRepository
   */
  private void loadSampleLetters() {
    synchronized (CACHE) {
      if (CACHE.containsKey(font)) {
        letterRepository = CACHE.get(font);
      } else {
        letterRepository = Collections.synchronizedMap(
            new HashMap<Character, PrettyStringSprite>());
        for (int i = 0; i < 26; i++) {
          getSprite((char) ('A' + i));
          getSprite((char) ('a' + i));
          if (i < 10) {
            getSprite((char) ('0' + i));
          }
        }
        CACHE.put(font, letterRepository);
      }
      // we need to go through an array to avoid a concurrent
      // modification exception in multiplayer games run all at once
      for (PrettyStringSprite sprite :
        letterRepository.values().toArray(new PrettyStringSprite[0])) {
        maxCharacterWidth = Math.max(maxCharacterWidth,
            sprite.getWidth());
        height = sprite.getHeight();
      }
    }
  }

  /**
   * sets the actual scale while keeping the location constant. This is
   * necessary because of the justification manipulations.
   *
   * @param  dimension  the size in screens
   */
  private void setMaxDimension(double dimension) {
    ++deep;
    scale(dimension / getMaxDimension());
    --deep;
  }
}
