package fang2.sprites;

import fang2.attributes.*;
import fang2.core.Sprite;

import java.awt.geom.*;

/**
 * A sprite representing a filled, closed polygon. Can be built with a
 * list of points or a number of sides. The number of sides constructor
 * will build a regular polygon (inscribed within a unit circle) with
 * the given number of edges.
 *
 * @author  Jam Jenkins
 */
public class PolygonSprite
  extends Sprite {
  /**
   * Construct a new, filled {@code PolygonSprite}. The parameter is a
   * sequence of pairs of (x, y) coordinates. Thus the total number of
   * entries must be even. Order matters for determining what part of
   * the screen is the interior of the polygon. The shape will be closed
   * automatically by connecting the last point back to the first. No
   * fewer than 3 points must be provided. To make a square, you could
   * send in the following points: 0, 0, 1, 0, 1, 1, 1, 0 which
   * represent the four corners of a square: (0, 0) (1, 0) (1, 1) and
   * (1, 0). The order in which the points are given is the order in
   * which the lines will be drawn.
   *
   * @param  points  the x,y pairs for the vertexes of the polygon. The
   *                 order of the points matters and determines the
   *                 order of the lines drawn to connect the edges of
   *                 the polygon.
   */
  public PolygonSprite(double... points) {
    GeneralPath path = new GeneralPath();
    path.moveTo( (float)points[0],  (float)points[1]);
    for (int i = 2; i < points.length; i += 2) {
      path.lineTo( (float)points[i],  (float)points[i + 1]);
    }
    path.closePath();
    setAbsoluteShape(path);
  }

  /**
   * Construct a new, filled {@code PolygonSprite} with the given
   * vertices.
   *
   * @param  points  The vertices of the polygon
   */
  public PolygonSprite(Point2D.Double... points) {
    GeneralPath path = new GeneralPath();
    path.moveTo( (float)points[0].x,  (float)points[0].x);
    for (int i = 1; i < points.length; i++) {
      path.lineTo( (float)points[i].x,  (float)points[i].y);
    }
    path.closePath();
    setAbsoluteShape(path);
  }

  /**
   * Construct a new, filled {@code PolygonSprite} with the given
   * vertices.
   *
   * @param  points  The vertices of the polygon
   */
  public PolygonSprite(Location2D... points) {
    this((Point2D.Double[]) points);
  }

  /**
   * Construct a new, filled, regular {@code PolygonSprite} with the
   * given number of sides. 
   * @param  sides  number of sides of the regular polygon; sides >= 3
   */
  public PolygonSprite(int sides) {
    if (sides < 3) {
      sides = 3;
    }
    double angle = -Math.PI / 2;
    GeneralPath path = new GeneralPath();
    path.moveTo((float)Math.cos(angle) * 0.5f,
    		(float)Math.sin(angle) * 0.5f);
    for (int i = 1; i < sides; i++) {
      angle = (-Math.PI / 2) + (i * 2 * Math.PI / sides);
      path.lineTo((float)Math.cos(angle) * 0.5f,
    		  (float)Math.sin(angle) * 0.5f);
    }
    path.closePath();
    setAbsoluteShape(path);
  }
}
