package fang2.sprites;

import fang2.attributes.Location2D;
import fang2.core.Sprite;

import java.awt.BasicStroke;
import java.awt.Shape;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;
import java.util.ArrayList;

/**
 * A sprite composed of line and curve segments. This sprite does not
 * include fill; for filled sprites with arbitrary outlines, see {@link
 * PolygonSprite}.
 *
 * @author  Jam Jenkins
 * @author  Brian C. Ladd
 */
public class PolyLineSprite
  extends Sprite {
  /** the default initial thickness of all outline sprites */
  public static final double DEFAULT_LINE_THICKNESS = 0.01;

  /** the initial thickness of all outline sprites; in screen widths */
  private static double INITIAL_LINE_THICKNESS = DEFAULT_LINE_THICKNESS;

  /** the thinnest line permitted for any outline; in screen widths */
  public static final double MINIMUM_LINE_THICKNESS = 0.001;

  /** the path that can also include curves */
  protected GeneralPath lines;

  /**
   * the current thickness of the line drawn for this outline sprite.
   */
  private double lineWidth;

  /** the part that turns the path into a shape */
  protected BasicStroke stroke;

  /**
   * creates a polyline from (0, 0) to (1, 1)
   */
  public PolyLineSprite() {
    this(0, 0, 1, 1);
  }

  /**
   * creates a polyline given a set of 2 or more coordinates. Note:
   * every pair of values represents the next (x, y) coordinate,
   * therefore the number of parameters must be even. The PolyLineSprite
   * does not automatically close shapes. If you want a closed shape,
   * use PolygonSprite instead.
   *
   * @param  points  the set of (x,y) pairs for the desired line
   */
  public PolyLineSprite(double... points) {
    lineWidth = INITIAL_LINE_THICKNESS;
    stroke = new BasicStroke((float) lineWidth);
    ArrayList<Point2D.Double> p = new ArrayList<Point2D.Double>();
    for (int i = 0; (i + 1) < points.length; i += 2) {
      p.add(new Point2D.Double(points[i], points[i + 1]));
    }
    initPath(p.toArray(new Point2D.Double[0]));
  }

  /**
   * Construct a new, outlined, regular {@code PolyLineSprite} with the
   * given number of sides. Uses {@link
   * PolyLineSprite#makePolygonPoints(int)
   * PolyLineSprite.makePolygonPoints} method to generate the corner
   * points. That routine has the built-in check to make sure side count
   * >= 3.
   *
   * @param  sides  number of sides of the regular polygon; sides >= 3
   */
  public PolyLineSprite(int sides) {
    this(makePolygonPoints(sides));
  }

  /**
   * creates a polyline given a set of 2 or more coordinates. The
   * PolyLineSprite does not automatically close shapes. If you want a
   * closed shape, use PolygonSprite instead.
   *
   * @param  points  the set of (x,y) pairs for the desired line
   */
  public PolyLineSprite(Location2D... points) {
    this((Point2D.Double[]) points);
  }

  /**
   * creates a polyline given a set of 2 or more coordinates. The
   * PolyLineSprite does not automatically close shapes. If you want a
   * closed shape, use PolygonSprite instead.
   *
   * @param  points  the set of (x,y) pairs for the desired line
   */
  public PolyLineSprite(Point2D.Double... points) {
    lineWidth = INITIAL_LINE_THICKNESS;
    stroke = new BasicStroke((float) lineWidth);
    initPath(points);
  }

  /**
   * creates a polyline from the outline of any given shape.
   *
   * @param  shape  the closed shape from which an outline should be
   *                drawn.
   */
  public PolyLineSprite(Shape shape) {
    lineWidth = INITIAL_LINE_THICKNESS;
    stroke = new BasicStroke((float) lineWidth);
    lines = new GeneralPath(shape);
    initShape(true);
  }

  /**
   * creates a polyline from the outline of any given sprite.
   *
   * @param  sprite  the sprite from which an outline should be drawn.
   */
  public PolyLineSprite(Sprite sprite) {
    this(sprite.getShape());
  }

  /**
   * Get the default line thickness, the line thickness used by all
   * newly created {@code PolyLineSprite}s from this point forward.
   *
   * @return  the default (constructed, initial) line thickness in
   *          screen widths
   */
  public static double getInitialLineThickness() {
    return INITIAL_LINE_THICKNESS;
  }

  /**
   * Generate point-set for a polygon with the given number of {@code
   * sides}; if {@code sides} is <= 3, generate a triangle.
   *
   * @param  sides  the number of sides in the polygon (sides >= 3)
   */
  public static Location2D[] makePolygonPoints(int sides) {
    if (sides < 3) {
      sides = 3;
    }
    double angle = -Math.PI / 2;
    Location2D[] polygonPoints = new Location2D[sides + 1];
    polygonPoints[0] = new Location2D(Math.cos(angle) * 0.5f,
        Math.sin(angle) * 0.5f);
    polygonPoints[sides] = new Location2D(Math.cos(angle) * 0.5f,
        Math.sin(angle) * 0.5f);
    for (int i = 1; i < sides; i++) {
      angle = (-Math.PI / 2) + (i * 2 * Math.PI / sides);
      polygonPoints[i] = new Location2D(Math.cos(angle) * 0.5f,
          Math.sin(angle) * 0.5f);
    }
    return polygonPoints;
  }

  /**
   * Set the line thickness used for all newly created {@code
   * PolyLineSprite} objects. When a sprite is created, it copies this
   * thickness (in screens) as its line thickness and no longer uses
   * this value.
   *
   * @param  lineThickness  the new inital line thickness in screens
   */
  public static void setInitialLineThickness(double lineThickness) {
    INITIAL_LINE_THICKNESS = Math.max(
        PolyLineSprite.MINIMUM_LINE_THICKNESS, lineThickness);
  }

  /**
   * gets the portion of the screen each dash takes up. The return value
   * of this method changes as the size of the sprite changes.
   *
   * @return  the portion of the screen each dash takes up
   */
  public double getLineDashLength() {
    float[] dashes = stroke.getDashArray();
    if (dashes.length == 0) {
      return 0;
    } else {
      return dashes[0];
    }
  }

  /**
   * gets the on/off pattern of the dashes used in drawing the outline.
   * The pattern is an ordered list of on and off alternating. For
   * example 0.2f, 0.1f, 0.4f, 0.3f means 0.2f of outline, skip 0.1f,
   * 0.4f of outline skip 0.3f then repeat. The pattern sizes change
   * when the size of the sprite changes.
   *
   * @return  the on/off pattern of the dashes used in drawing the
   *          outline
   */
  public float[] getLineDashPattern() {
    float[] dashes = stroke.getDashArray();
    return dashes;
  }

  /**
   * gets the portion of the screen used in drawing the polyline. Note:
   * the line thickness changes as the size of the sprite changes.
   *
   * @return  thickness of the line in screens
   */
  public double getLineThickness() {
    return lineWidth;
  }

  /**
   * determines if line segments are joined with a straight line at the
   * edge
   *
   * @return  true if the segments are joined with a straignt edge,
   *          false otherwise
   */
  public boolean hasBevelJoin() {
    return stroke.getLineJoin() == BasicStroke.JOIN_BEVEL;
  }

  /**
   * determines if the end of the line has extra added or not
   *
   * @return  true if there is no extra space at the end of the line,
   *          false otherwise
   */
  public boolean hasButtCap() {
    return stroke.getEndCap() == BasicStroke.CAP_BUTT;
  }

  /**
   * determines if line segments are extended until they join
   *
   * @return  true if the segments do extend, false otherwise
   */
  public boolean hasMiterJoin() {
    return stroke.getLineJoin() == BasicStroke.JOIN_MITER;
  }

  /**
   * determines if the end of the line is capped with a small circle
   *
   * @return  true if the end of the line has a circle, false otherwise
   */
  public boolean hasRoundCap() {
    return stroke.getEndCap() == BasicStroke.CAP_ROUND;
  }

  /**
   * determines if line segments are joined with a round edge
   *
   * @return  true if the segments are joined with a round edge, false
   *          otherwise
   */
  public boolean hasRoundJoin() {
    return stroke.getLineJoin() == BasicStroke.JOIN_ROUND;
  }

  /**
   * determines if the end of the line is capped with a small square
   *
   * @return  true if the end of the line has a square, false otherwise
   */
  public boolean hasSquareCap() {
    return stroke.getEndCap() == BasicStroke.CAP_SQUARE;
  }

  /**
   * converts the points into a GeneralPath, then initializes the
   * outline shape
   *
   * @param  points  a set of connected points
   */
  protected void initPath(Point2D.Double... points) {
    lines = new GeneralPath();
    lines.moveTo((float) points[0].x, (float) points[0].y);
    for (int i = 1; i < points.length; i++) {
      lines.lineTo((float) points[i].x, (float) points[i].y);
    }
    initShape(true);
  }

  /**
   * uses the lines and stroke to create and set the shape
   *
   * @param  firstTime  true meaning this should set the size and
   *                    location, false meaning that this should only
   *                    set the shape
   */
  protected void initShape(boolean firstTime) {
    stroke = new BasicStroke((float) lineWidth, stroke.getEndCap(),
        stroke.getLineJoin(), stroke.getMiterLimit(),
        stroke.getDashArray(), 0.0f);

    if (firstTime) {
      setAbsoluteShape(stroke.createStrokedShape(lines));
    } else {
      setShape(stroke.createStrokedShape(lines));
    }
  }

  /**
   * sets the length of the dashes used in drawing the polyline. This
   * length changes when the size of the sprite changes.
   *
   * @param  length  the length of each dash as a portion of the screen
   */
  public void setLineDashLength(double length) {
    float[] dashes = { (float) (length), (float) (length) };
    stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(),
        stroke.getLineJoin(), stroke.getMiterLimit(), dashes,
        stroke.getDashPhase());
    setShape(stroke.createStrokedShape(lines));
  }

  /**
   * sets the on/off pattern of the dashes used in drawing the polyline.
   * The pattern is an ordered list of on and off alternating. For
   * example 0.2f, 0.1f, 0.4f, 0.3f means 0.2f of outline, skip 0.1f,
   * 0.4f of outline skip 0.3f then repeat. The pattern sizes change
   * when the size of the sprite changes.
   *
   * @param  dashes  the on/off pattern of the dashes used in drawing
   *                 the outline
   */
  public void setLineDashPattern(float... dashes) {
    stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(),
        stroke.getLineJoin(), stroke.getMiterLimit(), dashes,
        stroke.getDashPhase());
    setShape(stroke.createStrokedShape(lines));
  }

  /**
   * sets the portion of the screen that the polyline thickness should
   * fill.
   *
   * @param  thickness  the width of the polyline as a portion of the
   *                    screen
   */
  public void setLineThickness(double thickness) {
    lineWidth = (float) Math.max(MINIMUM_LINE_THICKNESS, thickness);
    initShape(false);
  }

  /**
   * makes line segments join by drawing a line between the endpoints.
   */
  public void useBevelJoin() {
    stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(),
        BasicStroke.JOIN_BEVEL, stroke.getMiterLimit(),
        stroke.getDashArray(), stroke.getDashPhase());
    setShape(stroke.createStrokedShape(lines));
  }

  /**
   * makes it so that the line does not extend past the end point. The
   * cap is the end of the line and butt means that there is no extra
   * space after the end point (unlike with square and round caps).
   */
  public void useButtCap() {
    stroke = new BasicStroke(stroke.getLineWidth(),
        BasicStroke.CAP_BUTT, stroke.getLineJoin(),
        stroke.getMiterLimit(), stroke.getDashArray(),
        stroke.getDashPhase());
    setShape(stroke.createStrokedShape(lines));
  }

  /**
   * makes line segments join by extending the segment edges until they
   * meet.
   */
  public void useMiterJoin() {
    stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(),
        BasicStroke.JOIN_MITER, stroke.getMiterLimit(),
        stroke.getDashArray(), stroke.getDashPhase());
    setShape(stroke.createStrokedShape(lines));
  }

  /**
   * makes it so that the line extends past the end point by adding a
   * small circle. The cap is the end of the line and round means that a
   * circle the diameter of the line should be added as extra space at
   * the end of the line.
   */
  public void useRoundCap() {
    stroke = new BasicStroke(stroke.getLineWidth(),
        BasicStroke.CAP_ROUND, stroke.getLineJoin(),
        stroke.getMiterLimit(), stroke.getDashArray(),
        stroke.getDashPhase());
    setShape(stroke.createStrokedShape(lines));
  }

  /**
   * makes line segments join with rounded edges.
   */
  public void useRoundJoin() {
    stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(),
        BasicStroke.JOIN_ROUND, stroke.getMiterLimit(),
        stroke.getDashArray(), stroke.getDashPhase());
    setShape(stroke.createStrokedShape(lines));
  }

  /**
   * makes it so that the line extends past the end point by adding a
   * small square. The cap is the end of the line and square means that
   * a square the width of the line should be added as extra space at
   * the end of the line.
   */
  public void useSquareCap() {
    stroke = new BasicStroke(stroke.getLineWidth(),
        BasicStroke.CAP_SQUARE, stroke.getLineJoin(),
        stroke.getMiterLimit(), stroke.getDashArray(),
        stroke.getDashPhase());
    setShape(stroke.createStrokedShape(lines));
  }
}
