package fang2.sprites;

import java.awt.geom.Arc2D;

import fang2.core.Sprite;


/**
 * This class represents a wedge of pie as a Sprite.
 * 
 * This arc is defined as part of a full oval by a 
 * start and end angle around the oval. The angles 
 * are specified such that 45 degrees falls halfway
 * up between the width and height ratio of the oval.
 * The arc sweeps around counter-clockwise from the 
 * start angle to the end angle.  For example, given
 * 0 and 180 as the start and end angles, defines an 
 * arc over the top half of the oval; but given 180 
 * and 0 as the start and end angles, defines an arc 
 * over the bottom half of the oval.
 * 
 * The arc is enclosed by drawing a straight line from 
 * the start of the arc segment to the center of the 
 * defining oval and from there to end of the arc segment
 * (like a slice of pie).
 * 
 * @author Robert Duvall
 */
public class PieSprite extends Sprite
{
    /**
     * Makes a Sprite that looks like a circular wedge of pie,
     * i.e., its width and height ratio is 1:1,
     * that goes from the given start angle to the
     * given end angle.
     * 
     * @param start beginning angle in degrees of the arc
     * @param end ending angle in degrees of the arc
     */
    public PieSprite (double start, double end)
    {
        this(1, 1, start, end);
    }


    /**
     * Makes a Sprite that looks like a wedge of pie 
     * that goes from the given start angle to the
     * given end angle, on an oval proportioned 
     * according to the given width and height 
     * ratio.
     * 
     * The magnitude of the width and height only 
     * matter with respect to each other. For example, 
     * a Sprite constructed with a width 2 and height 
     * 1 is the same as making the width 20 and the 
     * height 10 because they both have the same 
     * height/width aspect ratio of 2.
     * 
     * @param width width scale of the rectangle
     * @param height height scale of the rectangle
     * @param start beginning angle in degrees of the arc
     * @param end ending angle in degrees of the arc
     */
    public PieSprite (double width, double height, double start, double end)
    {
        super(new Arc2D.Double(0, 0, width, height,
                               start, (end > start) ? end - start : (360 + end) - start,
                               Arc2D.PIE));
    }
}