package fang2.sprites;

import fang2.attributes.Location2D;
import fang2.core.Sprite;

import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;
import java.awt.geom.PathIterator;
import static java.awt.geom.PathIterator.SEG_CLOSE;
import static java.awt.geom.PathIterator.SEG_LINETO;
import static java.awt.geom.PathIterator.SEG_MOVETO;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

public class OutlineSprite
  extends Sprite {
  /**
   * default flattening value applied to shaped when they are converted
   * into outlines
   */
  private static final double DEFAULT_FLATNESS = 0.001;

  /** the initial thickness of all outline sprites; in screen widths */
  private static double INITIAL_LINE_THICKNESS =
    PolyLineSprite.DEFAULT_LINE_THICKNESS;

  /**
   * Get the default line thickness, the line thickness used by all
   * newly created OutlineSprites from this point forward.
   *
   * @return  the default (constructed, initial) line thickness in
   *          screen widths
   */
  public static double getInitialLineThickness() {
    return INITIAL_LINE_THICKNESS;
  }

  /**
   * Set the default, starting line thickness used by all newly created
   * OutlineSprites; when an OutlineSprite is constructed, it copies its
   * line thickness from this value.
   *
   * @param  lineThickness  the new initial line thickness value
   */
  public static void setInitialLineThickness(double lineThickness) {
    INITIAL_LINE_THICKNESS = Math.max(
        PolyLineSprite.MINIMUM_LINE_THICKNESS, lineThickness);
  }

  /**
   * the current thickness of the line drawn for this outline sprite.
   */
  private double lineThickness;

  /** the shape which is outlined by this sprite */
  private Shape startingShape;

  /**
   * Create a new {@link OutlineSprite} connecting the given points.
   *
   * @param  points  even number of coordinates; paired into (x, y)
   *                 values to designate points through which theoutlne
   *                 will go
   */
  public OutlineSprite(double... points) {
    lineThickness = INITIAL_LINE_THICKNESS;
    startingShape = getPath(points);
    initShape();
  }

  /**
   * Create a new {@link OutlineSprite} connecting the given points.
   *
   * @param  points  arbitrary number of {@link Location2D} points
   *                 through which an outline will be drawn
   */
  public OutlineSprite(Location2D... points) {
    lineThickness = INITIAL_LINE_THICKNESS;
    startingShape = getPath(points);
    initShape();
  }

  /**
   * Create a new {@link OutlineSprite} connecting the given points.
   *
   * @param  points  arbitrary number of {@link java.awt.geom.Point2D.Double} points
   *                 through which an outline will be drawn
   */
  public OutlineSprite(Point2D.Double... points) {
    lineThickness = INITIAL_LINE_THICKNESS;
    startingShape = getPath(points);
    initShape();
  }

  /**
   * Create a new {@link OutlineSprite} outlining the given shape.
   *
   * @param  shape  the {@link Shape} to outline
   */
  public OutlineSprite(Shape shape) {
    lineThickness = INITIAL_LINE_THICKNESS;
    startingShape = new Area(shape);
    initShape();
  }

  /**
   * Create a new {@link OutlineSprite} outlining the shape of the given
   * sprite.
   *
   * @param  sprite  the {@link Sprite} to outline
   */
  public OutlineSprite(Sprite sprite) {
    lineThickness = INITIAL_LINE_THICKNESS;
    startingShape = sprite.getShape();
    initShape();
  }

  /**
   * Get the line thickness of this {@link OutlineSprite}.
   *
   * @return  the current outline thickness of this sprite
   */
  public double getLineThickness() {
    return lineThickness;
  }

  /**
   * Initialize the shape.
   */
  public void initShape() {
    Shape shape = startingShape;
    Point2D.Double translate = new Point2D.Double();
    double scaleFactor = 1;
    lineThickness /= scaleFactor;
    startingShape = shape;

    Shape outline = getOutline(shape);
    Rectangle2D bounds = outline.getBounds2D();
    setShape(outline);
    setScale(Math.max(bounds.getWidth(), bounds.getHeight()));
    setLocation(bounds.getCenterX(), bounds.getCenterY());
    scale(scaleFactor);
    translate(translate);
    lineThickness *= scaleFactor;
  }

  /**
   * Set the line thickness of this sprite. Update the shape to use the
   * new thickness.
   *
   * @param  thickness  new thickness in screens
   */
  public void setLineThickness(double thickness) {
    lineThickness = Math.max(PolyLineSprite.MINIMUM_LINE_THICKNESS,
        thickness);
    initShape();
  }

  /**
   * Get a line between the two points (x0, y0) and (x1, y1). A "line"
   * is really a thin polygon drawn along the given line segment.
   *
   * @param   x0  x-coordinate of the first point
   * @param   y0  y-coordinate of the first point
   * @param   x1  x-coordinate of the second point
   * @param   y1  y-coordinate of the second point
   *
   * @return  the {@link Area} covering the given line segment
   */
  private Area getLine(double x0, double y0, double x1, double y1) {
    Point2D.Double start = new Point2D.Double(x0, y0);
    Point2D.Double end = new Point2D.Double(x1, y1);
    Point2D.Double midpoint = new Point2D.Double((start.x + end.x) / 2,
        (start.y + end.y) / 2);
    double angle = Math.atan2(end.y - start.y, end.x - start.x);
    double distance = start.distance(end);
    Rectangle2D.Double rectangle = new Rectangle2D.Double(0, 0,
        distance, lineThickness);
    Ellipse2D.Double startButt = new Ellipse2D.Double(-lineThickness /
        2, 0, lineThickness, lineThickness);
    Ellipse2D.Double endButt = new Ellipse2D.Double(distance -
        (lineThickness / 2), 0, lineThickness, lineThickness);
    Area total = new Area(rectangle);
    total.add(new Area(startButt));
    total.add(new Area(endButt));

    AffineTransform transform = new AffineTransform();
    transform.translate(midpoint.x, midpoint.y);
    transform.rotate(angle);
    transform.translate(-distance / 2, -lineThickness / 2);
    total.transform(transform);

    return total;
  }

  /**
   * Get the shape corresponding to the outline of the given shape
   *
   * @param   shape  the {@link Shape} to outline
   *
   * @return  a {@link Shape} representing the outline of the original
   *          shape
   */
  private Shape getOutline(Shape shape) {
    PathIterator path;
    path = shape.getPathIterator(new AffineTransform(),
        DEFAULT_FLATNESS);

    double[] curve = new double[6];
    path.currentSegment(curve);

    Point2D.Double currentPoint = new Point2D.Double();
    currentPoint.x = curve[0];
    currentPoint.y = curve[1];

    Point2D.Double startPoint = new Point2D.Double();
    startPoint.x = curve[0];
    startPoint.y = curve[1];
    path.next();

    Area total = new Area();

    while (!path.isDone()) {
      int curveType = path.currentSegment(curve);

      if (curveType == SEG_CLOSE) {
        curve[0] = startPoint.x;
        curve[1] = startPoint.y;
        curveType = SEG_LINETO;
      } else if (curveType == SEG_MOVETO) {
        startPoint.x = curve[0];
        startPoint.y = curve[1];
      }

      if (curveType == SEG_LINETO) {
        Area line = getLine(currentPoint.x, currentPoint.y, curve[0],
            curve[1]);
        total.add(line);
      }

      currentPoint.x = curve[0];
      currentPoint.y = curve[1];
      path.next();
    }

    return total;
  }

  /**
   * Get the {@link Shape} corresponding to the outline through the
   * given points.
   *
   * @param   points  an even number of coordinates, paired into (x, y)
   *                  pairs to determine the corners of the polygon
   *                  outlined by this sprite
   *
   * @return  {@link Shape} outline going through the given points
   */
  private Shape getPath(double... points) {
    GeneralPath path = new GeneralPath();
    path.moveTo((float) points[0], (float) points[1]);

    for (int i = 2; i < (points.length - 1); i += 2) {
      path.lineTo((float) points[i], (float) points[i + 1]);
    }

    return path;
  }

  /**
   * Get the {@link Shape} corresponding to the outline through the
   * given points.
   *
   * @param   points  an arbitrary number of {@link Point2D.Double}
   *                  points; they determine the corners of the polygon
   *                  outlined by this sprite
   *
   * @return  {@link Shape} outline going through the given points
   */
  private Shape getPath(Point2D.Double... points) {
    GeneralPath path = new GeneralPath();
    path.moveTo((float) points[0].x, (float) points[0].y);

    for (int i = 1; i < points.length; i++) {
      path.lineTo((float) points[i].x, (float) points[i].y);
    }

    return path;
  }
}
