package fang2.sprites;
//start auto-imports
import java.util.*;

import fang2.attributes.*;
import fang2.core.*;

//end auto-imports


/**
 * This is a text input widget.
 * The InputStringField is like a JTextArea for the FANG Engine.
 * The text is always scaled to fit within the given box dimensions.
 * Note: this sprite must be on the screen (but not necessarily visible)
 * in order to receive keyboard input.
 * @author Jam Jenkins
 */
public class InputStringField extends CompositeSprite
{
	/**the background of the field, initially white*/
	private final RectangleSprite box;
	/**the background of the field, initially white*/
	private final RectangleSprite deactivated;
	/**the foreground string*/
	private final InputStringSprite input;
	private final StringSprite label;
	/**how much the string fills the box*/
	private final double used=0.9;
	/**which players can enter text*/
	private final ArrayList<Integer> players;
	/**toggle for whether mouse must be over box*/
	private boolean disregardMouse=true;

	/**creates a box with a given aspect ratio
	 * The only player who can give input is the first player.
	 * @param width the horizontal span
	 * @param height the vertical span
	 */
	public InputStringField(double width, double height)
	{
		this(width, height, 0);
	}

	/**creates a box with a given aspect ratio
	 * This constructor allows the specification of who can type
	 * into this field.  The parameter playerIndex takes the player
	 * numbers of those allowed to change the field.
	 * @param width the horizontal span
	 * @param height the vertical span
	 * @param playerIndex the list of players who are allowed to type
	 */
	public InputStringField(double width, double height, int ... playerIndex)
	{
		players=new ArrayList<Integer>();
		for(int p: playerIndex)
			players.add(p);

		double max=Math.max(width, height);
		box=new RectangleSprite(width/max, height/max);
		box.setColor(Game.getColor("white"));
		deactivated=new RectangleSprite(width/max, height/max);
		deactivated.setColor(Palette.getColor("Gray", 64));
		deactivated.setVisible(false);
		input=new InputStringSprite();
		input.setLineHeight(box.getHeight()*used);
		input.setLocation(-box.getWidth()/2*used,
		                  -box.getHeight()/2*used);
		label=new StringSprite();
		label.setLineHeight(box.getHeight()*used*0.5);
		label.setLocation(-box.getWidth()/2,
		                  -box.getHeight()/2);
		label.leftJustify();
		label.bottomJustify();
		addSprite(label);
		addSprite(box);
		addSprite(deactivated);
		addSprite(input);
		setSize(max);
	}
	
	/**
	 * determines if the mouse must be over the box to capture keyboard typing
	 * By default the input box will accept keyboard input any time (i.e. needsMouse
	 * is false by default). If needsMouse is true, then the mouse must be over
	 * the box in order to capture input.
	 * @param needsMouse whether the mouse does or does not have to be over the box to accept input
	 * false means input is accepted any time, true means input accepted only when the mouse
	 * is over the box.
	 */
	public void requireMouseOver(boolean needsMouse)
	{
		disregardMouse=!needsMouse;
	}
	
	/**sets the text above the box
	 * The text is by default about half the height of the
	 * text inside the box when there is a single line.
	 * @param text
	 */
	public void setLabelText(String text)
	{
		label.setText(text);
	}
	
	/**
	 * gets the text used to label this box
	 * @return the sequence of characters above the box
	 */
	public String getLabelText()
	{
		return label.getText();
	}
	
	/**
	 * makes the text above the box visible
	 */
	public void showLabel()
	{
		setLabelVisible(true);
	}
	
	/**
	 * makes the text above the box invisible
	 */
	public void hideLabel()
	{
		setLabelVisible(false);
	}
	
	/**
	 * makes the text above the box visible or invisible
	 * @param vis true means show the text, false means hide the text
	 */
	public void setLabelVisible(boolean vis)
	{
		label.setVisible(false);
	}
	
	/**
	 * determines if enter has been pressed in the field
	 * @return true if enter has been pressed, false otherwise
	 */
	public boolean hasEnteredText()
	{
		return getText().contains("\n");
	}
	
	/**
	 * gets the text up to the point where enter has been pressed.
	 * This method returns null when hasEnteredText() is false.
	 * @return the text prior to the new line, or null if enter has yet to be pressed
	 */
	public String getEnteredText()
	{
		if(!hasEnteredText()) return null;
		return getText().split("\n")[0];
	}

	/**
	 * gets the text from the input field
	 * @return the sequence of characters in the field
	 */
	public String getText()
	{
		return input.getText();
	}

	/**
	 * sets the text of the input field
	 * @param text the sequence of characters for the field
	 */
	public void setText(String text)
	{
		input.setText(text);
	}

	/**
	 * allows the given players to type into the field
	 * @param playerIndex all of the player numbers who can type in the field
	 */
	public void addPlayerInput(int ... playerIndex)
	{
		for(int p: playerIndex)
			if(!players.contains(p))
				players.add(p);
	}

	/**
	 * keeps the given players from typing into the field
	 * @param playerIndex all of the player numbers who cannot type in the field
	 */
	public void removePlayerInput(int ... playerIndex)
	{
		for(int p: playerIndex)
			if(players.contains(p))
				players.remove(p);
	}
	
	/**
	 * makes it so that no player can enter text
	 */
	public void clearPlayerInput()
	{
		players.clear();
	}

	/**
	 * @param playerIndex the player to see if capable of input
	 * @return true if the player can add to the field, false otherwise
	 */
	public boolean canInput(int playerIndex)
	{
		return players.contains(playerIndex);
	}

	/**
	 * updates the size of the text and capability to input
	 * based upon the players' mouse positions
	 */
	public void update()
	{
		//only allow input when mouse is over box
		input.clearPlayerInput();
		deactivated.setVisible(true);
		for(int index: players)
		{

			Location2D mouse=Game.getCurrentGame().getMouse2D(index);
			boolean active=disregardMouse || intersects(mouse);
			if(active)
			{
				input.addPlayerInput(index);
				deactivated.setVisible(false);
			}
			else
			{
				input.removePlayerInput(index);
			}
		}
		input.update();
		boolean tooBig=input.getWidth()>box.getWidth()*used ||
		               input.getHeight()>box.getHeight()*used;
		boolean smaller=input.getHeight()<box.getHeight()*used;
		//make larger if possible
		while(!tooBig && smaller)
		{
			input.setLineHeight(input.getLineHeight()+0.0025);
			tooBig=input.getWidth()>box.getWidth()*used ||
			       input.getHeight()>box.getHeight()*used;
			smaller=input.getHeight()<box.getHeight()*used;
		}
		//shrink if too big
		while(tooBig)
		{
			input.setLineHeight(input.getLineHeight()-0.0025);
			tooBig=input.getWidth()>box.getWidth()*used ||
			       input.getHeight()>box.getHeight()*used;
		}
	}
}
