package fang2.sprites;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import fang2.attributes.*;
import fang2.core.Sprite;
import fang2.core.Game;
import fang2.core.Transformer;
/**
 * A {@code CompositeSprite} that contains a group of sprites;
 * individual sprites are positioned relative to the origin of the
 * composite (they can be translated, rotate, or scaled relative to the
 * {@code CompositeSprite}s origin and scale). The entire group is then
 * treated as a single sprite so it can be translated, rotated, or
 * scaled as a unit.
 * 
 * When the {@code CompositeSprite} is created, it has an origin just
 * like a {@link fang2.core.Game}: the origin is in the center of the
 * sprite; this is different than the screen (where the center of the
 * screen is at (0.5, 0.5)) but this makes it easier to picture where
 * and how the composite sprite will be positioned and will rotate. As
 * sprites are added to the {@code CompositeSprite}, their position,
 * rotation, and scale are relative to the {@code CompositeSprite}'s
 * origin.
 * 
 * Individual elements in a {@code CompositeSprite} can be referred to
 * individually and moved/rotated/scaled relative to the whole sprite.
 * The {@code CompositeSprite} can also be moved, rotated, scaled
 * relative to the game canvas (assuming it has been added to the
 * canvas). Visibility is handled at both the composite and subsprite
 * levels: If the composite is visible, then the subsprites that are
 * visible are drawn; if the composite is not visible, then none of the
 * subsprites are drawn.
 * 
 * Like the game canvas, the range of coordinates is effectively
 * unbounded though only the portion scaled to fit on the screen is
 * actually seen. Note that the location of the {@code CompositeSprite}
 * (and the center of rotation for the whole sprite) is (0.0, 0.0) in
 * terms of the {@code CompositeSprite}'s coordinate system. This is so
 * that a scale 1.0 sprite with no rotation and a location of (0.5, 0.5)
 * will be centered on the screen (and coordinates on the sprite will be
 * offset from those on the screen by 0.5 in each dimension).
 * 
 * @author Jam Jenkins
 * @author Robert C. Duvall
 * @author Brian C. Ladd
 */
public class CompositeSprite
  extends Sprite {
  protected static Color debugCrosshairsColor =
      new Color(150, 150, 200);

  /** the sprites in this group */
  private final List<Sprite> sprites = new ArrayList<Sprite>();

  /**
   * Add a single sprite to the top of this sprite's canvas.
   * 
   * @param sprite the {@link Sprite} object to add to the top of the
   *          current scene
   */
  public void addSprite(Sprite sprite) {
    sprites.add(sprite);
  }

  /**
   * Add all of the sprites ({@code sprite} can be a variable number of
   * {@link Sprite Sprite} objects) to this sprite's canvas. Real work
   * is forwarded to the single parameter version of {@link #addSprite}.
   * 
   * @param sprite a list of {@link Sprite} objects to add to the top of
   *          the current scene
   */
  public void addSprite(Sprite... sprite) {
    for (Sprite s : sprite) {
      addSprite(s);
    }
  }

  /**
   * determines if the sprite given as a parameter is part of this
   * group.
   * 
   * @param sprite the sprite to check for membership
   * @return true if the sprite is part of the group, false otherwise
   */
  public boolean contains(Sprite sprite) {
    return sprites.contains(sprite);
  }

  /**
   * Get an {@link java.util.List} of {@link fang2.core.Sprite} that make
   * up this composite. The value returned
   * 
   * @return the ArrayList containing all of the sprites in this {@code
   *         CompositeSprite}.
   */
  public List<Sprite> getAllSprites() {
    return sprites;
  }

  /**
   * given a point in the canvas coordinate system, this method
   * calculates where that point would currently be relative to this
   * group
   * 
   * @param outside the position in the canvas coordinate system
   * @return {@link fang2.attributes.Location2D Location2D} corresponding
   *         to the outside point but in the inside coordinate system.
   */
  public Location2D getFrameLocation(Location2D outside) {
    Location2D location = null;
    try {
      location =
          new Location2D(transform.inverseTransform(outside, null));
    } catch (NoninvertibleTransformException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    return location;
  }

  /**
   * Translate a point from the <i>outside</i> coordinate system (the
   * one containing the {@code CompositeSprite}) into the coordinate
   * system <i>inside</i> the {@code CompositeSprite}.
   * 
   * @param outsideX x-coordinate of outside point
   * @param outsideY y-coordinate of outside point
   * @return {@link fang2.attributes.Location2D Location2D} corresponding
   *         to the outside point but in the inside coordinate system.
   */
  public Location2D getFrameLocation(double outsideX, double outsideY) {
    return getFrameLocation(new Location2D(outsideX, outsideY));
  }

  /**
   * given a point relative to the center of this group, this method
   * calculates where that point would currently be in the canvas
   * coordinate system
   * 
   * @param inside the position relative to this group
   * @return the corresponding position in the canvas coordinate system
   */
  public Location2D getRealLocation(Location2D inside) {

    Location2D location = null;
    try {
      location =
          new Location2D(transform.inverseTransform(inside, null));
    } catch (NoninvertibleTransformException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    return location;
  }

  /**
   * Convert a location <i>inside</i> the {@code CompositeSprite}'s
   * coordinate space into the coordinate space <i>containing</i> the
   * {@code CompositeSprite}. Translate an inside coordinate to an
   * outside coordinate.
   * 
   * @param insideX x-coordinate of point to translate
   * @param insideY y-coordinate of point to translate
   * @return {@link Location2D} in the same coordinate space
   *         <i>containing</i> this {@code CompositeSprite}
   *         corresponding to the given point.
   */
  public Location2D getRealLocation(double insideX, double insideY) {
    return getRealLocation(new Location2D(insideX, insideY));
  }

  /**
   * shows the outline for all sprites currently contained in this
   * composite sprite. This is an instantaneous property which means
   * that when new sprites are added to the composite, they do not
   * automatically show their outline.
   */
  @Override
  public void showOutline() {
    for (Sprite sprite : sprites)
      sprite.showOutline();
  }

  /**
   * sets the thickness for all sprites currently contained in this
   * composite sprite. This is an instantaneous property which means
   * that when new sprites are added to the composite, they do not
   * automatically have the same outline thickness.
   */
  @Override
  public void setOutlineThickness(double thickness) {
    for (Sprite sprite : sprites)
      sprite.setOutlineThickness(thickness);
  }

  /**
   * gets the thickness of the first sprite in the composite.
   * 
   * @return the outline thickness of the first sprite
   */
  @Override
  public double getOutlineThickness() {
    if (sprites.size() > 0)
      return sprites.get(0).getOutlineThickness();
    else
      return -1;
  }

  /**
   * gets the outline color of the first sprite in the composite.
   * 
   * @return the outline color of the first sprite
   */
  @Override
  public Color getOutlineColor() {
    if (sprites.size() > 0)
      return sprites.get(0).getOutlineColor();
    else
      return super.getOutlineColor();
  }

  /**
   * sets the outline color for all sprites currently contained in this
   * composite sprite. This is an instantaneous property which means
   * that when new sprites are added to the composite, they do not
   * automatically have the same outline color.
   */
  @Override
  public void setOutlineColor(Color c) {
    for (Sprite sprite : sprites)
      sprite.setOutlineColor(c);
  }

  /**
   * hides the outline for all sprites currently contained in this
   * composite sprite. This is an instantaneous property which means
   * that when new sprites are added to the composite, they do not
   * automatically hide their outline.
   */
  @Override
  public void hideOutline() {
    for (Sprite sprite : sprites)
      sprite.showOutline();
  }

  /**
   * sets the dash length for all sprites currently contained in this
   * composite sprite. This is an instantaneous property which means
   * that when new sprites are added to the composite, they do not
   * automatically have the same dash length.
   * 
   * @param length the dash length as a portion of the screen
   */
  @Override
  public void setDashLength(double length) {
    for (Sprite sprite : sprites)
      sprite.setDashLength(length);
  }

  /**
   * gets the dash length of the first sprite in the composite.
   * 
   * @return the dash length of the first sprite
   */
  @Override
  public double getDashLength() {
    if (sprites.size() > 0)
      return sprites.get(0).getDashLength();
    else
      return 0;
  }

  /**
   * sets the dash pattern for all sprites currently contained in this
   * composite sprite. This is an instantaneous property which means
   * that when new sprites are added to the composite, they do not
   * automatically have the same dash pattern.
   * 
   * @param dashes the on/off dash lengths as portions of the screen
   */
  @Override
  public void setDashPattern(float... dashes) {
    for (Sprite sprite : sprites)
      sprite.setDashPattern(dashes);
  }

  /**
   * gets the dash pattern of the first sprite in the composite.
   * 
   * @return the dash pattern of the first sprite
   */
  @Override
  public float[] getDashPattern() {
    if (sprites.size() > 0)
      return sprites.get(0).getDashPattern();
    else
      return new float[0];
  }

  /**
   * The shape of the composite sprite, scaled and transformed
   * appropriately. Note that the {@link java.awt.Shape Shape} of a
   * {@code CompositeSprite} is the union of the shapes of the
   * subsprites.
   * 
   * @return the shape
   */
  @Override
  public Shape getShape() {
    GeneralPath ourShape = new GeneralPath(); // initially empty
    ourShape.setWindingRule(GeneralPath.WIND_NON_ZERO);
    for (Sprite subsprite : sprites) {
      ourShape.append(subsprite.getShape(), false);
    }
    // ourShape.transform(AffineTransform.getTranslateInstance(-ourShape.getBounds2D().getWidth()/2,
    // -ourShape.getBounds2D().getHeight()/2));
    ourShape.transform(transform);

    return ourShape;
  }

  /**
   * Determines if a point is within the sprite's bounding box. This is
   * the fundamental version of intersects with a point (used by both
   * {@link fang2.attributes.Location2D Location2D} and
   * {@link java.awt.geom.Point2D Point2D} versions of the method) so
   * overriding this one makes it work for all intersects methods.
   * 
   * @param x x-coordinate of the point to check
   * @param y y-coordinate of the point to check
   * @return true if the point is in the shape, false otherwise
   */
  @Override
  public boolean intersects(double x, double y) {
    boolean retval = false;
    if ((getBounds2D().getMinX() <= x) &&
        (x < getBounds2D().getMaxX()) &&
        (getBounds2D().getMinY() <= y) && (y < getBounds2D().getMaxY())) {
      if (!getUseBoundingBox()) {
        Location2D point = getFrameLocation(x, y);
        for (Sprite s : sprites) {
          if (s.intersects(point)) {
            retval = true;
          }
        }
      } else
        retval = true;
    }
    return retval;
  }

  /**
   * determines if another Sprite intersects this Sprite. Intersection
   * is a geometric property of the sprites and does not depend on
   * whether the sprites are on the canvas or visible.
   * 
   * @param other another Sprite
   * @return true if the other Sprite intersects this Sprite, false if
   *         there is no intersection
   */
  @Override
  public boolean intersects(Sprite other) {
    boolean retval = false;
    if (getBounds2D().intersects(other)) {
      if (!getUseBoundingBox()) {
        for (Sprite subsprite : sprites) {
          AffineTransform originalTransform =
              (AffineTransform) subsprite.transform.clone();

          // Transform the location of the subsprite to be in world
          // coordinates
          subsprite.transform = (AffineTransform) transform.clone();
          subsprite.transform.concatenate(originalTransform);
          retval = retval || subsprite.intersects(other);

          // return subsprite to local (CompositeSprite) coordinates
          subsprite.transform = originalTransform;
        }
      } else
        retval = true;
    }
    return retval;
  }
  
  /**
   * All CompositeSprites are supposed to be centered at (0, 0) and span
   * from (-0.5, -0.5) to (0.5, 0.5).  Sometimes you want to get the
   * sprite locations and sizes correct relative to each other, but you
   * don't want to do this in the given coordinate system.  This method
   * takes all of the sprites in whatever coordinate system they are in
   * and shifts and scales them to make them fit exactly in a box centered
   * at (0, 0) and filling as much of the space as possible between
   * (-0.5, -0.5) and (0.5, 0.5).  For example, you could put all of your
   * sprites between (0.0, 0.0) and (1.0, 1.0) (screen coordinates), then
   * call normalizeSprites to shift and scale them the way CompositeSprite
   * needs them.
   * 
   * Note: sprites added after calling this method are unaffected.
   */
  public void normalizeSprites()
  {
	  if(sprites.size()==0) return;
	  Location2D min=sprites.get(0).getLocation();
	  Location2D max=sprites.get(0).getLocation();
	  for(Sprite s: sprites)
	  {
		  Box2D box=s.getBounds2D();
		  min.x=Math.min(min.x, box.getMinX());
		  min.y=Math.min(min.y, box.getMinY());
		  max.x=Math.max(max.x, box.getMaxX());
		  max.y=Math.max(max.y, box.getMaxY());
	  }
	  double sizeInverse=1.0/Math.max(max.x-min.x, max.y-min.y);
	  Location2D negativeCenter=new Location2D(-(min.x+max.x)/2, -(min.y+max.y)/2);
	  for(Sprite s: sprites)
	  {
		  s.translate(negativeCenter);
		  s.setX(s.getX()*sizeInverse);
		  s.setY(s.getY()*sizeInverse);
		  s.scale(sizeInverse);
	  }
  }

  /**
   * draws all added sprites in the proper location, orientation, and
   * size
   * 
   * @param brush the Graphics used to draw the shape
   */
  @Override
  public void paint(Graphics2D brush) {
	  for(Sprite s: sprites)
	  {
	     if(s.isVisible())
	     {
		  Location2D original=s.getLocation();
		  Location2D current=s.getLocation();
		  double scale=getScale();
		  double rotation=getRotation();
		  current.x*=scale;
		  current.y*=scale;
		  AffineTransform t=AffineTransform.getScaleInstance(scale, scale);
		  t.concatenate(AffineTransform.getRotateInstance(rotation));
		  t.transform(original, current); 
		  s.setLocation(current);
		  s.translate(getLocation());
		  s.scale(scale);
		  s.rotate(rotation);
		  s.paint(brush);
		  s.rotate(-rotation);
		  s.scale(1.0/scale);
		  s.setLocation(original);
	     }
	  }
	  /*the code below does not appropriately handle nesting
    for (Sprite s : sprites) {
    	//ImageSprites require special handling since they
    	//are not scaled/rotated on the fly.
    	if(s instanceof ImageSprite)
    	{
    		AffineTransform original=brush.getTransform();
    		try {
				brush.transform(transform.createInverse());
			} catch (NoninvertibleTransformException e) {
				e.printStackTrace();
			}
			Location2D real=this.getRealLocation(s.getLocation());
    		s.translate(real);
    		s.scale(this.getScale());
    		s.rotate(this.getRotation());
    		s.paintInternal(brush);
    		brush.setTransform(original);
    		s.translate(-real.x, -real.y);
    		s.scale(1.0/this.getScale());
    		s.rotate(-this.getRotation());    		
    	}
    	//Everything else is handled normally.
    	else
    	{
    	      s.paintInternal(brush);    		
    	}
    }*/
  }

  /**
   * removes a sprite from this group. If the sprite is not part of this
   * group, the call is ignored and no exception is thrown.
   * 
   * @param s the sprite to remove
   */
  public void removeSprite(Sprite s) {
    sprites.remove(s);
  }

  /**
   * Paint crosshairs at the center of rotation. The crosshairs are
   * painted in the current color with the current line style. They are
   * 
   * @param brush the graphics context to use to draw the crosshairs.
   */
  private void paintCrosshairs(Graphics2D brush) {
    Line2D.Double yAxis = new Line2D.Double(0.0, -0.2, 0.0, 0.2);
    Line2D.Double xAxis = new Line2D.Double(-0.2, 0.0, 0.2, 0.0);
    brush.draw(yAxis);
    brush.draw(xAxis);
  }

  /**
   * Paint the bounding box and the crosshairs.
   * 
   * @param brush the graphics context to use to draw them
   */
  @Override
  protected void drawDebugShapes(Graphics2D brush) {
    super.drawDebugShapes(brush);
    AffineTransform original = brush.getTransform();
    brush.transform(transform);
    setDebugBrush(debugCrosshairsColor, brush);
    paintCrosshairs(brush);
    brush.setTransform(original); // reset transform on the way out
  }
  
  /**
   * Updates all of the sprites
   *
   */
  @Override
  public void update() {
    for(Sprite one: sprites) {
      Game game=Game.getCurrentGame();
      if(game!=null && !game.containsSprite(one))
        one.applyTransformerNG();
        one.update();
    }
  }
  
  /**
   * Return a string representation of the {@code CompositeSprite}. The
   * type, location, and size of the {@code CompositeSprite}. Then recur
   * inward.
   */
  @Override
  public String toString() {
    return getClass().getName() + "[x = " + getX() + ", y = " + getY() +
           "], " + "[w = " + getWidth() + ", h = " + getHeight() +
           "] " + sprites;
  }
  
  public void reportTransformers(Set<Transformer> allTransformers) {
    super.reportTransformers(allTransformers);
    for(Sprite one: sprites) {
      Game game=Game.getCurrentGame();
      if(game!=null && !game.containsSprite(one)) {
        one.reportTransformers(allTransformers);
      }
    }
  }
}
