package fang2.sprites;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.awt.geom.RoundRectangle2D;
import java.awt.geom.Point2D.Double;
import java.util.ArrayList;

import fang2.core.Alarm;
import fang2.core.GameLoop;
import fang2.core.Mouse;
import fang2.core.Sprite;
import fang2.core.AlarmAdapter;

public class ButtonSprite
  extends CompositeSprite {
  /** this gets the shape of the text */
  private final PrettyStringSprite text;
  /** the top oval */
  private Sprite top;
  /** the bottom oval */
  private Sprite bottom;
  /** the mice to which the button responds */
  private ArrayList<Mouse> mouse;
  /** the alarms to call when the button is clicked */
  private ArrayList<Alarm> alarm;
  /** the listeners to alert when the button is clicked */
  private ArrayList<ActionListener> action;
  /** color of the text and bottom oval */
  private Color foreground;
  /** color of the top oval */
  private Color background;

  /** aspect (width/height) of the oval */
  private double aspect;

  /** color of the text and bottom oval */
  private static final Color DEFAULT_FOREGROUND = Color.YELLOW;
  /** color of the top oval */
  private static final Color DEFAULT_BACKGROUND = new Color(92, 0, 0);

  /** maximum height used for the text */
  private static final double MAX_HEIGHT = 0.7;
  /** maximum width used for the text */
  private static final double MAX_WIDTH = 0.8;
  /** padding for default aspect buttons */
  private static final double ASPECT_PADDING = 1.2;

  /**
   * makes a button with an aspect to fit the text. By default, the
   * button does not change aspect when the button's text changes.
   * 
   * @param label text for the button
   */
  public ButtonSprite(String label) {
    text = new PrettyStringSprite(label, true);
    double aspect = text.getWidth() / text.getHeight();
    if (label.trim().length() > 0) {
      aspect = aspect * ASPECT_PADDING;
    } else {
      aspect = 1;
    }
    initialize(label, aspect);
  }

  /**
   * makes a button with a given aspect ratio of width/height. By
   * default, the button does not change aspect when the button's text
   * changes.
   * 
   * @param label text for the button
   * @param aspect the ratio of width/height of the oval
   */
  public ButtonSprite(String label, double aspect) {
    text = new PrettyStringSprite(label, true);
    initialize(label, aspect);
  }

  /**
   * initializes the shapes for the button.
   * 
   * @param label text for the button
   * @param aspect the ratio of width/height of the oval
   */
  private void initialize(String label, double aspect) {
    alarm = new ArrayList<Alarm>();
    mouse = new ArrayList<Mouse>();
    this.aspect = aspect;
    action = new ArrayList<ActionListener>();
    foreground = DEFAULT_FOREGROUND;
    background = DEFAULT_BACKGROUND;
    RoundRectangle2D.Double roundBox =
        new RoundRectangle2D.Double(0, 0, aspect, 1, 1, 1);
    top = new Sprite(roundBox);
    top.setScale(1);
    bottom = new Sprite(roundBox);
    bottom.setScale(1);
    Point2D.Double location = top.getLocation();
    location.y += 0.025;
    location.x += 0.025;
    bottom.setLocation(location);
    setTextSize();
    addSprite(bottom);
    addSprite(top);
    addSprite(text);
    raise();
  }

  /**
   * makes the text fit within the button oval
   */
  private void setTextSize() {
    text.setLineHeight(MAX_WIDTH / aspect);
    if (text.getWidth() > MAX_WIDTH) {
      text.setWidth(MAX_WIDTH);
    }
    if (text.getHeight() > MAX_HEIGHT) {
      text.setLineHeight(MAX_HEIGHT);
    }
  }

  /**
   * sets the color of the button when it is up
   */
  public void raise() {
    top.setColor(background);
    bottom.setColor(foreground);
    text.setColor(foreground);
  }

  /**
   * sets the color of the button when it is down
   */
  public void lower() {
    top.setColor(foreground);
    bottom.setColor(background);
    text.setColor(background);
  }

  /**
   * sets the color of the text on the button
   * 
   * @param textColor the color for the text and the oval underneath
   */
  public void setTextColor(Color textColor) {
    foreground = textColor;
  }

  /**
   * sets the color of the top oval of the button
   * 
   * @param buttonColor the color for the top oval of the button
   */
  public void setButtonColor(Color buttonColor) {
    background = buttonColor;
  }

  /**
   * calls the methods of all Alarms and ActionListeners that have been
   * added to the button
   */
  public void executePressAction() {
    for (Alarm a : alarm) {
      a.act();
    }
    for (ActionListener a : action) {
      a.actionPerformed(new ActionEvent(this, 0, null));
    }
  }

  /**
   * adds an alarm to the button. The alarm will be called when the
   * button is clicked.
   * 
   * @param a the alarm to call when the button is clicked
   */
  public void addAlarm(Alarm... a) {
    for (Alarm one : a) {
      alarm.add(one);
    }
  }

  /**
   * removes the alarm from the button. The alarm removed will no longer
   * be called when the button is clicked.
   * 
   * @param a the alarm to remove
   */
  public void removeAlarm(Alarm... a) {
    for (Alarm one : a) {
      alarm.remove(one);
    }
  }

  /**
   * adds an alarm to the button. The alarm will be called when the
   * button is clicked.
   * 
   * @param a the alarm to call when the button is clicked
   */

  public void addTimedAction(AlarmAdapter... a) {
    for (AlarmAdapter one : a) {
      alarm.add(one);
    }
  }

  /**
   * removes the alarm from the button. The alarm removed will no longer
   * be called when the button is clicked.
   * 
   * @param a the alarm to remove
   */
  public void removeTimedAction(AlarmAdapter... a) {
    for (AlarmAdapter one : a) {
      alarm.remove(one);
    }
  }

  /**
   * adds a listener to the button. The listener will be called when the
   * button is clicked.
   * 
   * @param a the listener to call when the button is clicked
   */
  public void addActionListener(ActionListener... a) {
    for (ActionListener one : a) {
      action.add(one);
    }
  }

  /**
   * removes the listener from the button. The listener removed will no
   * longer be called when the button is clicked.
   * 
   * @param a the listener to remove
   */
  public void removeActionListener(ActionListener... a) {
    for (ActionListener one : a) {
      action.remove(one);
    }
  }

  /**
   * enables all players to click on the button
   * 
   * @param loop the game
   */
  public void enable(GameLoop loop) {
    for (int i = 0; i < loop.getNumberOfPlayers(); i++) {
      mouse.add(loop.getPlayer(i).getMouse());
    }
  }

  /**
   * enables the specified mouse to interact with the button
   * 
   * @param m the mouse to enable interactions with
   */
  public void enable(Mouse... m) {
    for (Mouse one : m) {
      mouse.add(one);
    }
  }

  /**
   * disables clicking on the button
   */
  public void disable() {
    mouse.clear();
  }

  /**
   * enables the specified mouse to interact with the button
   * 
   * @param m the mouse to enable interactions with
   */
  public void disable(Mouse... m) {
    for (Mouse one : m) {
      mouse.remove(one);
    }
  }

  /**
   * gets the text on the button
   * 
   * @return the text on the face of the button
   */
  public String getText() {
    return text.getText();
  }

  /**
   * sets the text on the button. This may reduce or increase the size
   * of the font based upon the length of the string.
   * 
   * @param t
   */
  public void setText(String t) {
    text.setText(t);
    text.setColor(foreground);
    setTextSize();
  }


  /**
   * not used
   */
  public double getRotationAddition() {
    return 0;
  }

  /**
   * not used
   */
  public double getScaleFactor() {
    return 1;
  }

  /**
   * not used
   */
  public Double getTranslation() {
    return new Point2D.Double();
  }
}
