package fang2.media;

/**
 * This class can play only one sound at a time. Once the sound has
 * played to completion, it can be replayed. SequentialSound is the base
 * class for MidiSound and SampledSound, and the Sound class contains a
 * collection of SequentialSound to play sounds simultaneously.
 * 
 * @author Jam Jenkins
 * 
 */
public abstract class SequentialSound {
  double pan = 0.5;
  double volume = 1.0;
  boolean muted;
  boolean paused;

  /**
   * gets the clip position in seconds
   * 
   * @return the seconds that this clip has played
   */
  public abstract double getClipPosition();

  /**
   * sets the clip position in seconds.
   * 
   * @param time the seconds from the start of the sound
   */
  public abstract void setClipPosition(double time);

  /**
   * gets the duration of this sound in seconds
   * 
   * @return the seconds long this sound would play
   */
  public abstract double getClipLength();

  /** starts the sound playing */
  public abstract void play();

  /** stops the sound keeping the current position */
  public abstract void pause();

  /** makes the sound inaudible */
  public abstract void mute();

  /** makes the sound audible */
  public abstract void turnSoundOn();

  /**
   * sets whether the sound should loop or not
   * 
   * @param doLoop true indicates the sound should loop indefinitely,
   *          false means to terminate playback at the end of the clip's
   *          duration
   */
  public abstract void setLooping(boolean doLoop);

  /**
   * sets how many times the sound should play before terminating
   * 
   * @param loops the number of times to play the sound before stopping.
   *          The default is 1.
   */
  public abstract void setLooping(int loops);

  /**
   * gets how many times the sound will play through until it stops. The
   * default is 0 meaning that once the current clip ends, the sound
   * terminates.
   * 
   * @return the number of entire clip durations left to play (zero is
   *         the default)
   */
  public abstract int getLoopsLeft();

  /**
   * determines if the sound clip has been entirely loaded. Since
   * loading is a time consuming operation, a sound may not be ready to
   * play immediately after it is constructed.
   * 
   * @return true if the sound has been loaded completely, false
   *         otherwise
   */
  public abstract boolean isLoaded();

  /**
   * determines if the sound is audible when the clip is playing
   * 
   * @return true indicates that sound will be audible when the clip is
   *         playing, false means the sound is currently muted.
   */
  public abstract boolean isMuted();

  /**
   * determines if the sound is currently responding to the change in
   * time.
   * 
   * @return true if the sound is paused, false otherwise
   */
  public abstract boolean isPaused();

  /**
   * gets a sound that is just like this one in its default initially
   * constructed state
   * 
   * @return a copy of the current sound
   */
  public abstract SequentialSound getDuplicate();

  /**
   * determines if this sound is done playing. The sound is done playing
   * when it either has not started or when it has reached the end of
   * its playing duration.
   * 
   * @return true if sound is done; false otherwise
   */
  public boolean isFinishedPlaying() {
    return !isLooping() && getClipPosition() + 0.01 >= getClipLength();
  }

  /**
   * sets the volume of this clip. This method actually affects the gain
   * in a linear way. Setting the value to 1 means that the sound's
   * volume is unchanged. Setting the value to 2 makes it twice as loud
   * whereas setting it to 0.5 makes the sound 1/2 as loud. A default
   * method is provided, but classes that extend SequentialSound may
   * optionally make this method affect the actual sound.
   * 
   * @param volume the multiplier to affect the volume
   */
  public void setVolume(double volume) {
    this.volume = volume;
  }

  // optional
  /**
   * gets the multiplier that has been added to this sound. By default,
   * all sounds are unchanged and have a mutliplier of 1.
   * 
   * @return the multiplier that affects the amplitude of the sound
   */
  public double getVolume() {
    return volume;
  }

  // optional
  public void setPan(double pan) {
    this.pan = Math.max(0, Math.min(1, pan));
  }

  // optional
  public double getPan() {
    return pan;
  }

  /**
   * starts the clip playing from the beginning
   */
  public void start() {
    setClipPosition(0);
    play();
  }

  /**
   * sets the clip position to the end to stop the clip from playing
   */
  public void stop() {
    setClipPosition(getClipLength());
  }

  /**
   * plays the sound, controlling the panning left or right
   * 
   * @param pan the side on which to play where 0.0 is entirely on the
   *          left and 1.0 is entirely on the right, and 0.5 is evenly
   *          balanced. Any value between 0.0 and 1.0 is valid.
   */
  public void play(double pan) {
    setPan(pan);
    play();
  }

  /**
   * plays the sound repeatedly
   */
  public void loop() {
    setLooping(true);
    play();
  }

  /**
   * plays the sound from its current location in time
   */
  public void resume() {
    play();
  }

  /**
   * tells whether the sound is currently looping
   * 
   * @return true means the sound is looping, false means the sound will
   *         terminate when it gets to the end of the clip
   */
  public boolean isLooping() {
    return getLoopsLeft() > 0;
  }

  /**
   * tells whether the clip is currently audible and not yet at the end
   * 
   * @return true if it is playing; false otherwise
   */
  public boolean isPlaying() {
    return !isFinishedPlaying() && getClipPosition() > 0;
  }
}
