package fang2.media;

import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.lang.reflect.Constructor;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashSet;
import java.util.Timer;
import java.util.TimerTask;

import javax.sound.midi.MidiUnavailableException;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Control;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.sound.sampled.Mixer.Info;
import javax.sound.sampled.spi.AudioFileReader;

/**
 * Plays mp3 and wav files. This class plays sounds as soon as they are
 * available, skipping the beginning part of the sound if it still
 * loading when it is requested to start playing.
 * 
 * @author Jam Jenkins
 * 
 */
public class SampledSound
  extends SequentialSound
  implements RawMediaCache.MediaLoadListener {
  /** size of the buffer for reading and writing sound */
  private static final int BUFFER_SIZE = 44100;
  /** how often to update the sound buffer */
  private static final int TIME_INTERVAL = 100;

  /** an array of zeros for silence */
  /*
   * -- bcl never read locally private static final byte[] SILENCE = new
   * byte[BUFFER_SIZE];
   */

  /** control for muting all sounds */
  private static boolean masterMuted;
  /** control for pausing all sounds */
  private static boolean masterPlaying;
  /** the sounds that are currently playing */
  private static HashSet<SampledSound> activePlayable =
      new HashSet<SampledSound>();
  /** the offline reader/writer for sounds */
  private static Streamer streamer;
  /** the length of the clip in seconds */
  private double clipLength;
  /** control for muting this sound */
  private boolean muted;
  /** control for pausing this sound */
  private boolean playing = false;
  /** whether the sound is done yet or not */
  private boolean finishedPlaying = false;
  /** whether the sound is in memory yet */
  private boolean loaded = false;
  /**
   * part of the sound to skip at the beginning because it needed more
   * loading time
   */
  private long timeMissed = 0;

  /** bytes in the entire clip */
  /*
   * -- bcl never read locally private long totalBytesRead;
   */

  /** how many times this sound should play again */
  private int loopsLeft;
  /** the buffer used for reading/writing sound */
  private final byte[] byteBuffer;
  /** where sound is written */
  private SourceDataLine dataLine;
  /** where uncompressed sound is read */
  private PushbackInputStream inputStream;
  /** where compressed sound is read */
  private InputStream compressedIn;
  /** the sound location */
  private final URL soundFile;
  /** the gain (like volume additive) */
  private float gain = 0.0f;

  private static Class<?> MPEG_AUDIO_FILE_READER;
  private static Constructor<?> DECODED_MPEG_AUDIO_INPUT_STREAM;
  private static Class<?> OGG_AUDIO_FILE_READER;
  private static Constructor<?> DECODED_OGG_AUDIO_INPUT_STREAM;

  static {
    try {
      MPEG_AUDIO_FILE_READER =
          Class
              .forName("javazoom.spi.mpeg.sampled.file.MpegAudioFileReader");
      Class<?> decoderClass =
          Class
              .forName("javazoom.spi.mpeg.sampled.convert.DecodedMpegAudioInputStream");

      Class<?>[] types = new Class[2];
      types[0] = AudioFormat.class;
      types[1] = AudioInputStream.class;
      DECODED_MPEG_AUDIO_INPUT_STREAM =
          decoderClass.getDeclaredConstructor(types);
    } catch (Exception e) {
      // ignore when jars not available
    }
    try {
      OGG_AUDIO_FILE_READER =
          Class
              .forName("javazoom.spi.vorbis.sampled.file.VorbisAudioFileReader");
      Class<?> decoderClass =
          Class
              .forName("javazoom.spi.vorbis.sampled.convert.DecodedVorbisAudioInputStream");

      Class<?>[] types = new Class[2];
      types[0] = AudioFormat.class;
      types[1] = AudioInputStream.class;
      DECODED_OGG_AUDIO_INPUT_STREAM =
          decoderClass.getDeclaredConstructor(types);
    } catch (Exception e) {
      // ignore when jars not available
    }
  }

  /**
   * loads a sound. The sound is loaded serially in a separate thread in
   * the order in which the SinglePlayable is constructed.
   * 
   * @param soundFile the location of the sound
   */
  public SampledSound(URL soundFile) {
    if (streamer == null) {
      streamer = new Streamer();
      java.awt.EventQueue.invokeLater(new Runnable() {
        public void run() {
          Timer timer = new Timer();
          timer.scheduleAtFixedRate(streamer, 0, TIME_INTERVAL);
        }
      });
      // System.out.println("starting timer");
    }
    this.soundFile = soundFile;
    byteBuffer = new byte[BUFFER_SIZE];
    loaded = RawMediaCache.isFullyLoaded(soundFile);
    // bcl -- never used --long start = System.currentTimeMillis();
    if (!loaded) {
      RawMediaCache.load(soundFile, this);
    } else {
      prepareToPlay();
    }
    // System.out.println("Preparing: " + (System.currentTimeMillis() -
    // start));
    muted = false;
    loopsLeft = 0;
  }

  private void prepareToPlay() {
    resetStream();
    loaded = true;
    setPan();
    setVolume();
    dataLine.start();
    long missed = getNumberOfBytesNeeded(timeMissed);
    try {
      if (missed > 0) {
        skip(missed);
        timeMissed = 0;
        // adds a little buffer for smooth playback
        update(2 * TIME_INTERVAL);
        play();
      }

    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  /**
   * This inner class regularly streams data from the source to the
   * speakers as needed. All work is done on the AWT thread.
   * 
   * @author jam
   * 
   */
  static class Streamer
    extends TimerTask {
    long last = System.currentTimeMillis();

    @Override
    public void run() {
      long current = System.currentTimeMillis();
      long duration = current - last;
      last = current;
      for (SampledSound playable : activePlayable
          .toArray(new SampledSound[0])) {
        playable.update(duration);
      }
    }
  }

  /**
   * skips past a number of bytes at the beginning of the sound file.
   * This is required when it takes awhile for the sound file to load
   * and play is requested before the file is loaded.
   * 
   * @param maxBytes to bytes to skip
   * @return the actual bytes skipped
   * @throws IOException should not occur
   */
  private int skip(long maxBytes) throws IOException {
    long bytesRead = inputStream.skip(maxBytes);
    return (int) bytesRead;
  }

  /**
   * 
   * @param maxBytes
   * @return
   * @throws IOException
   */
  private int readThenWrite(int maxBytes) throws IOException {
    /** may need these lines later? */
    // int frameSize=dataLine.getFormat().getFrameSize();
    // int frames;//need to get frames here
    // int framesToRead=Math.min(maxBytes, byteBuffer.length);
    // System.out.println("Bytebuffer: " + byteBuffer.length);
    // System.out.println("Available: " + inputStream.available());
    // There's a problem here that it is not reading sufficiently
    int targetToRead = Math.min(maxBytes, byteBuffer.length);
    // System.out.println("Need: " + targetToRead);
    // bcl -- never used --long readStart = System.currentTimeMillis();
    int bytesRead = inputStream.read(byteBuffer, 0, targetToRead);
    int bytesBeyondFrame =
        bytesRead % dataLine.getFormat().getFrameSize();
    if (bytesBeyondFrame > 0) {
      inputStream.unread(byteBuffer, bytesRead - bytesBeyondFrame,
          bytesBeyondFrame);
    }
    if(bytesRead>=0)
    	bytesRead -= bytesBeyondFrame;
    // System.out.println("Reading took: " + (System.currentTimeMillis()
    // - readStart));
    if (bytesRead > 0) {
      if (!muted) {
        // System.out.println("writing: " + bytesRead);
        // bcl -- never read -- long writeStart =
        // System.currentTimeMillis();
        dataLine.write(byteBuffer, 0, bytesRead);
        // System.out.println("Writing took: " +
        // (System.currentTimeMillis() - writeStart));
      }
    }
    return bytesRead;
  }

  /**
   * gets how long the sound has been playing. When looping, this is
   * just the time since the most recent repeat.
   * 
   * @return the time since the start in seconds
   */
  @Override
  public double getClipPosition() {
    return dataLine.getMicrosecondPosition() / 1000;
  }

  /**
   * given a period of time, this method determines how many bytes must
   * be read. The number of bytes read depends on the format of the
   * audio. It is important to read a discrete number of frames in order
   * to avoid an exception.
   * 
   * @param timePassed the milliseconds that are needed
   * @return how many bytes should be read.
   */
  private long getNumberOfBytesNeeded(long timePassed) {
    AudioFormat format = dataLine.getFormat();
    // System.out.println("Frame rate: "+format.getFrameRate());
    int frames =
        (int) Math.ceil(timePassed / 1000.0 * format.getFrameRate());
    return frames * format.getFrameSize();
  }

  /**
   * this method handles all of the updates necessary when time passes.
   * Ideally, the timePassed should be long enough to provide smooth
   * playing, but short enough to be responsive. This method is called
   * at the beginning of playback to provide a short buffer (twice
   * interval) to insure no gaps between calls.
   * 
   * @param timePassed
   */
  private synchronized void update(long timePassed) {
    // bcl -- never used -- long otherStart =
    // System.currentTimeMillis();
    if (!playing) {
      return;
    }
    // System.out.println("Playing");
    try {
      long bytesNeeded;
      if (!loaded) {
        timeMissed += timePassed;
        bytesNeeded = 0;
      } else {
        bytesNeeded = getNumberOfBytesNeeded(timePassed);
        if (bytesNeeded > BUFFER_SIZE) {
          long bytesMissed = bytesNeeded - BUFFER_SIZE;
          skip(bytesMissed);
          bytesNeeded = BUFFER_SIZE;
          // System.out.println("skipping: " + bytesMissed);
        }
      }
      // System.out.println("before write delay: " +
      // (System.currentTimeMillis() - otherStart));
      // System.out.println("bytesNeeded: "+bytesNeeded);
      // bcl -- never used -- long start = System.currentTimeMillis();
      while (bytesNeeded > 0) {
        int bytesRead = readThenWrite((int) bytesNeeded);
        // System.out.println("bytesRead="+bytesRead);
        if (bytesRead < 0) {
          if (loopsLeft > 0) {
            // bcl -- never used -- totalBytesRead = 0;
            loopsLeft--;
            resetStream();
            bytesRead = readThenWrite((int) bytesNeeded);
            if (bytesRead < 0) {
              playing = false;
              finishedPlaying = true;
              resetStream();
            } else {
              bytesNeeded -= bytesRead;
            }
          } else {
            playing = false;
            finishedPlaying = true;
            resetStream();
            bytesNeeded = 0;
          }
        } else {
          bytesNeeded -= bytesRead;
        }
      }
      // System.out.println("write delay: " +
      // (System.currentTimeMillis() - start));
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  /**
   * This method gets a mixer capable of panning. Unfortunately, mono
   * sounds don't pan in the default mixer, but the Java 1.4 version had
   * this, so we use the 'Java Sound Audio Engine' from Java 1.4.
   * 
   * @return a pannable mixer if one exists, otherwise the default mixer
   *         is used.
   */
  private Mixer.Info getPanMixer() {
    Mixer.Info mix = null;
    Mixer.Info[] mixers = AudioSystem.getMixerInfo();
    for (Info element : mixers) {
      if ("Java Sound Audio Engine".equals(element.getName())) {
        mix = element;
      }
    }
    if (mix == null) {
      mix = AudioSystem.getMixer(null).getMixerInfo();
    }
    return mix;
  }

  /**
   * resets the compressed input stream to the beginning. If the streams
   * are not yet set up, this will do that too.
   * 
   * @throws MidiUnavailableException
   */
  private void resetStream() {
    AudioFormat decodedFormat = null;
    if (compressedIn == null) {
      compressedIn = RawMediaCache.getCachedStream(soundFile);
    } else {
      try {
        compressedIn.reset();
      } catch (IOException e1) {
        // TODO Auto-generated catch block
        e1.printStackTrace();
      }
    }
    AudioInputStream in = null;
    try {
      boolean isMP3 =
          soundFile.toString().toLowerCase().endsWith("mp3");
      boolean isOGG =
          soundFile.toString().toLowerCase().endsWith("ogg");
      if (isMP3 || isOGG) {
        AudioFileReader reader = null;
        try {
          if (isMP3) {
            reader =
                (AudioFileReader) MPEG_AUDIO_FILE_READER.newInstance();
          } else if (isOGG) {
            reader =
                (AudioFileReader) OGG_AUDIO_FILE_READER.newInstance();
          }
        } catch (Exception e) {
          System.err.println("JavaZoom needed for this sound");
          return;
        }
        in = reader.getAudioInputStream(compressedIn);
        decodedFormat =
            new AudioFormat(AudioFormat.Encoding.PCM_SIGNED, // encoding
                44100, // sample rate
                16, // bits per sample
                2, // channels
                4, // frame bytes
                44100, // frame rate
                false); // big endian
        try {
          Object[] args = new Object[] { decodedFormat, in };
          InputStream notBuffered = null;
          if (isMP3) {
            notBuffered =
                (InputStream) DECODED_MPEG_AUDIO_INPUT_STREAM
                    .newInstance(args);
          } else if (isOGG) {
            notBuffered =
                (InputStream) DECODED_OGG_AUDIO_INPUT_STREAM
                    .newInstance(args);
          }
          inputStream = new PushbackInputStream(notBuffered);
        } catch (Exception e) {
          e.printStackTrace();
          System.err.println("JavaZoom needed for this sound");
        }
        // javazoom.spi.mpeg.sampled.convert.DecodedMpegAudioInputStream
        // decoder=
        // new
        // javazoom.spi.mpeg.sampled.convert.DecodedMpegAudioInputStream(
        // decodedFormat, in);
        // inputStream=decoder;
      } else if (soundFile.toString().toLowerCase().endsWith("wav")) {
        InputStream notBuffered =
            AudioSystem.getAudioInputStream(compressedIn);
        inputStream = new PushbackInputStream(notBuffered);
        decodedFormat = ((AudioInputStream) notBuffered).getFormat();
      } else {
        System.err.println("Unsupported sound format: " +
                           soundFile.toString());
        return;
      }
      if (dataLine == null) {
        Mixer.Info mix = getPanMixer();
        dataLine = AudioSystem.getSourceDataLine(decodedFormat, mix);
        /*
         * -- bcl never read int bufferSize =
         * (decodedFormat.getFrameSize() decodedFormat.getFrameSize());
         */
        dataLine.open(decodedFormat, 44100 * 4);
      }
      if (in != null) {
        clipLength =
            in.getFrameLength() /
                (double) in.getFormat().getFrameRate();
      }
    } catch (UnsupportedAudioFileException e) {
      e.printStackTrace();
    } catch (IOException e) {
      e.printStackTrace();
    } catch (LineUnavailableException e) {
      e.printStackTrace();
    }
  }

  /**
   * starts playing the sound. This adds a brief buffer (twice interval)
   * to insure smooth playback.
   */
  @Override
  public void start() {
    playing = true;
    activePlayable.add(this);
    if (loaded) {
      dataLine.start();
      // adds a little buffer for smooth playback
      /*
       * bcl -- removed the group of lines long start =
       * System.currentTimeMillis(); update(2TIME_INTERVAL);
       * //System.out.println("Updating here: " +
       * (System.currentTimeMillis() - start));
       */
    }
  }

  /**
   * sets the panning of the audio. Panning is like balance except that
   * it does not eliminate the side; panning moves the sound more to one
   * side or the other. The value for panning is much like the values
   * for the horizontal positioning in the FANG Engine. 0 is all the way
   * to the left and 1 is all the way to the right. It is also possible
   * to pan somewhere in between by using fractional values.
   * 
   * @param pan the side on which to play. 0 is all the way to the left
   *          and 1 is all the way to the right. It is also possible to
   *          pan somewhere in between by using fractional values.
   */
  // !!!!Need to fix late start panning and volume control
  @Override
  public void setPan(double pan) {
    super.setPan(pan);
    setPan();
  }

  private void setPan() {
    float side = 2 * (float) pan - 1;
    // System.out.println("Side is " + side);
    try {
      if (dataLine != null) {
        Control panControl = dataLine.getControl(FloatControl.Type.PAN);
        ((FloatControl) panControl).setValue(side);
      }
    } catch (Exception e) {
    }
  }

  @Override
  public void setVolume(double volume) {
    super.setVolume(volume);
    gain = (float) Math.log10(volume) * 20.0f;
    // System.out.println("gain is " + gain);
    setVolume();
  }

  private void setVolume() {
    try {
      if (dataLine != null) {
        Control panControl =
            dataLine.getControl(FloatControl.Type.MASTER_GAIN);
        ((FloatControl) panControl).setValue(gain);
      }
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  @Override
  public void play() {
    start();
  }

  @Override
  public void play(double pan) {
    setPan(pan);
    start();
  }

  @Override
  public void stop() {
    playing = false;
    dataLine.stop();
    activePlayable.remove(this);
  }

  @Override
  public boolean isPlaying() {
    return playing;
  }

  @Override
  public boolean isFinishedPlaying() {
    return finishedPlaying;
  }

  @Override
  public void mute() {
    muted = true;
  }

  @Override
  public void turnSoundOn() {
    muted = false;
  }

  @Override
  public void setLooping(boolean looping) {
    if (looping) {
      loopsLeft = Integer.MAX_VALUE;
    } else {
      loopsLeft = 0;
    }
  }

  @Override
  public void setLooping(int loops) {
    loopsLeft = loops - 1;
  }

  @Override
  public boolean isLooping() {
    return loopsLeft > 0;
  }

  public URL getSoundURL() {
    return soundFile;
  }

  @Override
  public void loop() {
    setLooping(true);
    start();
  }

  public void stopLooping() {
    setLooping(false);
  }

  public static void main(String[] args) throws InterruptedException,
                                        MalformedURLException,
                                        Exception {
    // System.out.println("Running SampleSound.");
    URL soundFile = SampledSound.class.getResource(
    // new URL("http://iima.javawide.org/media.php?f=DingLower.wav");
        // "../examples/alarmbeep/resources/DingLower.wav");
        "VideoGameTrack.mp3");
    // "DingLower.wav");
    for (int i = 0; i < 1; i++) {
      SampledSound playable = new SampledSound(soundFile);
      // synchronized(playable){playable.wait(2001);}
      // System.out.println("starting sound");
      // playable.setLooping(true);
      playable.play(1);
      // playable.setVolume(10);
      playable.turnSoundOn();
      // synchronized(playable){playable.wait(3000);}
      /*
       * Sequencer sequencer=MidiSystem.getSequencer(); URL
       * url=SampledSound.class.getResource("YMCA.mid");
       * System.out.println("url is "+url); Synthesizer
       * synth=MidiSystem.getSynthesizer(); for(MidiChannel control:
       * synth.getChannels()) { control.controlChange(8, 127); }
       * Sequence sequence=MidiSystem.getSequence(url);
       * sequencer.setSequence(sequence); sequencer.open();
       * sequencer.start();
       * synchronized(playable){playable.wait(20001);} sequencer.stop();
       * synchronized(playable){playable.wait(20001);}
       * sequencer.start();
       */
    }
    // playable=new SinglePlayable(soundFile);
    // synchronized(playable){playable.wait(2001);}
    // playable.start();
    // synchronized(playable){playable.wait(2000);}
    // playable.start();
    /*
     * for(int i=0; i<10; i++) playable.update(0.0);
     * playable.muted=true; //playable.stop(); for(int i=0; i<10; i++)
     * playable.update(0.0); playable.start(); playable.muted=false;
     * for(int i=0; i<10; i++) playable.update(0.0);
     * playable.muted=true;
     */
  }

  @Override
  public double getClipLength() {
    return clipLength;
  }

  @Override
  public SequentialSound getDuplicate() {
    SampledSound sampled = new SampledSound(soundFile);
    sampled.setVolume(getVolume());
    sampled.setPan(getPan());
    sampled.paused = paused;
    sampled.muted = muted;
    sampled.loopsLeft = loopsLeft;
    return sampled;
  }

  @Override
  public int getLoopsLeft() {
    return loopsLeft;
  }

  @Override
  public boolean isLoaded() {
    return loaded;
  }

  @Override
  public boolean isMuted() {
    return muted || masterMuted;
  }

  @Override
  public boolean isPaused() {
    return (!playing && !finishedPlaying) || !masterPlaying;
  }

  @Override
  public void pause() {
    playing = false;
  }

  @Override
  public void resume() {
    playing = true;
  }

  /**
   * Only works to setClipPosition to the beginning.
   */
  @Override
  public void setClipPosition(double time) {
    if (time == 0) {
      resetStream();
    }
  }

  public void mediaFullyLoaded() {
    // System.out.println("alerted of load complete");
    prepareToPlay();
  }
}
