package fang2.media;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;

/**
 * This class facilitates loading and caching
 * content from a URL off the main execution
 * thread.  Since the FANG Engine advanceFrame
 * must happen quickly (about 1/25th of a second),
 * media must be either loaded off the main
 * execution thread or delay the game.  This class
 * loads and stores the exact contents of the URL
 * which is usually in compressed format, and has
 * options for loading off and on the main
 * execution thread.
 * @author jam
 *
 */
public class RawMediaCache extends Thread
{
    private static final ThreadLocal<RawMediaCache> RAW=
        new ThreadLocal<RawMediaCache>()
        {
            protected synchronized RawMediaCache initialValue()
            {
            	return SINGLETON;
                //this was return new RawMediaCache, but with the changed
                //threading, the threadlocal does not seem to make as much
                //sense.  In future versions, perhaps everything here should
                //be made static.
            }
        };
    private static final RawMediaCache SINGLETON=new RawMediaCache();
    /**size of the buffer for reading and writing sound*/
    private static final int BUFFER_SIZE = 8096;
    /**map of url to content*/
    private LinkedHashMap < URL, byte[] > cache =
        new LinkedHashMap < URL, byte[] > ();
    /**the list of url's to load*/
    private LinkedList<URL> queue =
        new LinkedList<URL>();
    /**those to notify when the url loads*/
    private HashMap < URL, LinkedList < MediaLoadListener >> listeners =
        new HashMap < URL, LinkedList < MediaLoadListener >> ();
    /**the buffer used for reading/writing sound*/
    private byte[] byteBuffer = new byte[BUFFER_SIZE];
    /**the maximum megabytes to cache*/
    private int megabyteLimit = 4;
    private long bytesStored = 0;

    private RawMediaCache()
    {
        start();
    }

    public static int getQueueLength()
    {
    	return RAW.get().queue.size();
    }
    
    public static void load(URL url)
    {
        //System.out.println("loading");
        synchronized (RAW.get().queue)
        {
            if (!RAW.get().queue.contains(url))
                RAW.get().queue.add(url);
            RAW.get().queue.notify();
        }
    }

    public static byte[] loadAndWait(URL url)
    {
        return loadData(url);
    }

    public static void load(URL url, MediaLoadListener listener)
    {
        // bcl -- never read -- long startTime = System.currentTimeMillis();
        synchronized (RAW.get().queue)
        {
            if (!RAW.get().queue.contains(url))
            {
                RAW.get().queue.add(url);
                RAW.get().queue.notify();
            }
            if (listener != null)
            {
                if (!RAW.get().listeners.containsKey(url))
                    RAW.get().listeners.put(url, new LinkedList<MediaLoadListener>());
                RAW.get().listeners.get(url).add(listener);
            }
        }
        //System.out.println("Loading took: " + (System.currentTimeMillis() - startTime));
    }

    public static boolean isCurrentlyLoading(URL url)
    {
        boolean loading = false;
        synchronized (RAW.get().queue)
        {
            loading = RAW.get().queue.contains(url);
        }
        return loading;
    }

    public static boolean isFullyLoaded(URL url)
    {
        return RAW.get().cache.containsKey(url) && RAW.get().cache.get(url) != null;
    }

    public static boolean hasLoadError(URL url)
    {
        return RAW.get().cache.containsKey(url) && RAW.get().cache.get(url) == null;
    }

    public static InputStream getCachedStream(URL url)
    {
        InputStream input = null;
        byte[] data;
        if (!isFullyLoaded(url))
            data = loadAndWait(url);
        else
            data = RAW.get().cache.get(url);
        input = new ByteArrayInputStream(data);
        //implement Least Recently Used cache replacement
        RAW.get().cache.remove(url);
        RAW.get().cache.put(url, data);
        return input;
    }

    private static byte[] loadData(URL url)
    {
        synchronized (RAW.get().cache)
        {
            try
            {
                ByteArrayOutputStream outBuffer =
                    new ByteArrayOutputStream();
                InputStream input;
                input = url.openStream();
                int bytesRead = input.read(RAW.get().byteBuffer);
                while (bytesRead >= 0)
                {
                    outBuffer.write(RAW.get().byteBuffer, 0, bytesRead);
                    bytesRead = input.read(RAW.get().byteBuffer);
                }
                byte[] compressedData = outBuffer.toByteArray();
                if (compressedData.length <= RAW.get().megabyteLimit << 20)
                    RAW.get().cache.put(url, compressedData);
                else
                {
                    double megabytes = compressedData.length / (double)(1 << 20);
                    megabytes = Math.round(megabytes * 100) / 100.0;
                    System.err.println("Too big: " + url + "\n" +
                                       "Size of URL is " + megabytes + " MB." + "\n" +
                                       "Size of cache is " + RAW.get().megabyteLimit + " MB.");
                }
                RAW.get().bytesStored += compressedData.length;
                maintainCacheLimit();
                return compressedData;
            }
            catch (IOException e)
            {
                RAW.get().cache.put(url, null);
                e.printStackTrace();
                return null;
            }
        }
    }

    private static void maintainCacheLimit()
    {
        if (RAW.get().bytesStored > RAW.get().megabyteLimit << 20)
        {
            ArrayList<URL> removalList = new ArrayList<URL>();
            for (URL key: RAW.get().cache.keySet())
            {
                removalList.add(key);
                RAW.get().bytesStored -= RAW.get().cache.get(key).length;
                if (RAW.get().bytesStored < RAW.get().megabyteLimit << 20)
                    break;
            }
            for (URL key: removalList)
            {
                //System.out.println("removing: " + key);
                RAW.get().cache.remove(key);
            }
        }
    }

    public void run()
    {
        URL url = null;
        while (true)
        {
            synchronized (queue)
            {
                while (queue.isEmpty())
                {
                    try
                    {
                        //System.out.println("Waiting");
                        queue.wait();
                        //System.out.println("Done waiting");
                    }
                    catch (InterruptedException e)
                    {
                        e.printStackTrace();
                    }
                }
                //System.out.println("loading url");
                url = queue.getFirst();
            }
            //do outside of synchronization to
            //avoid unnecessary blocking
            loadData(url);
            synchronized (queue)
            {
                queue.removeFirst();
            }
            //System.out.println("alerting listeners");
            if (listeners.containsKey(url))
            {
                for (MediaLoadListener l: listeners.get(url))
                {
                    //System.out.println("alerting " + l);
                    l.mediaFullyLoaded();
                }
            }
        }
    }

    public static interface MediaLoadListener
    {
        public void mediaFullyLoaded();
    }

    /**
     * @param args
     */
    public static void main(String[] args)
    {
        // TODO Auto-generated method stub

    }

}
