package fang2.media;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.LinkedList;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Sequence;
import javax.sound.midi.Sequencer;


public class MidiSound extends SequentialSound
{
    private Sequencer sequencer;
    private URL soundFile;
    private boolean loaded = false;
    private boolean muted;
    private boolean paused;
    private long startTime;

    /**the offline loader of sounds*/
    private static Loader loader;
    private static HashMap < URL, byte[] > cachedSounds =
        new HashMap < URL, byte[] > ();
    /**the sounds that are currently playing*/
/*-- bcl - never read locally (and private)  
    private static HashSet<MidiSound> activePlayable =
        new HashSet<MidiSound>();
*/

    public MidiSound(URL soundFile)
    {
        this.soundFile = soundFile;
        if (loader == null)
        {
            loader = new Loader();
            loader.start();
        }
        loaded = cachedSounds.containsKey(soundFile);
        if (!loaded) {
          loader.load(this);
        } else {
          prepareForPlaying();
        }
    }

    static class Loader extends Thread
    {
        /**the work that must be done*/
        private LinkedList<MidiSound> queue =
            new LinkedList<MidiSound>();

        /**adds to the queue of existing work*/
        public void load(MidiSound playable)
        {
            synchronized (queue)
            {
                queue.add(playable);
                queue.notify();
            }
        }

        /**empties the queue of work one at a time*/
        @Override
        public void run()
        {
            while (true)
            {
                synchronized (queue)
                {
                    if (!queue.isEmpty())
                    {
                        MidiSound toLoad = queue.removeFirst();
                        toLoad.loadDataFromURL();
                        toLoad.prepareForPlaying();
                    }
                    else
                    {
                        try
                        {
                            queue.wait();
                        }
                        catch (InterruptedException e)
                        {
                            e.printStackTrace();
                        }
                    }
                }
            }
        }
    }

    private void prepareForPlaying()
    {
        try
        {
            sequencer = MidiSystem.getSequencer();
            MidiSystem.getSequence(new ByteArrayInputStream(cachedSounds.get(soundFile)));
            Sequence sequence = MidiSystem.getSequence(new ByteArrayInputStream(cachedSounds.get(soundFile))); //MidiSystem.getSequence(soundFile);
            sequencer.setSequence(sequence);
            sequencer.open();
            if (startTime > 0)
            {
                long position = (System.currentTimeMillis() - startTime) * 1000;
                sequencer.setMicrosecondPosition(position);
                //System.out.println("starting at "+position/1000000.0);
                sequencer.start();
            }
            loaded = true;
        }
        catch (MidiUnavailableException e)
        {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        catch (InvalidMidiDataException e)
        {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        catch (IOException e)
        {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

    private void loadDataFromURL()
    {
        try
        {
            if (!cachedSounds.containsKey(soundFile))
            {
                ByteArrayOutputStream outBuffer = new ByteArrayOutputStream();
                InputStream input = soundFile.openStream();
                byte[] byteBuffer = new byte[32768];
                int bytesRead = input.read(byteBuffer);
                while (bytesRead >= 0)
                {
                    outBuffer.write(byteBuffer, 0, bytesRead);
                    bytesRead = input.read(byteBuffer);
                }
                byte[] compressedData = outBuffer.toByteArray();
                cachedSounds.put(soundFile, compressedData);
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }
    /**
    * @param args
    */
    public static void main(String[] args)
    {
        URL url = SampledSound.class.getResource("YMCA.mid");
        MidiSound sound = new MidiSound(url);
        sound.play();
    }

    @Override
    public double getClipLength()
    {
        return sequencer.getMicrosecondLength() / 1000000.0;
    }

    @Override
    public double getClipPosition()
    {
        return sequencer.getMicrosecondPosition() / 1000000.0;
    }

    @Override
    public SequentialSound getDuplicate()
    {
        MidiSound sound = new MidiSound(soundFile);
        sound.setVolume(getVolume());
        sound.setPan(getPan());
        if (isPaused()) {
          sound.pause();
        } else {
          sound.resume();
        }
        if (isMuted()) {
          sound.mute();
        } else {
          sound.turnSoundOn();
        }
        return sound;
    }

    @Override
    public int getLoopsLeft()
    {
        return sequencer.getLoopCount();
    }

    /**caching not yet fully implemented*/
    @Override
    public boolean isLoaded()
    {
        return loaded;
    }

    @Override
    public boolean isMuted()
    {
        return muted;
    }

    /**not yet implemented*/
    @Override
    public boolean isPaused()
    {
        return paused;
    }

    @Override
    public boolean isPlaying()
    {
        return sequencer.isRunning();
    }

    @Override
    public void loop()
    {
        sequencer.setLoopCount(Sequencer.LOOP_CONTINUOUSLY);
    }

    @Override
    public void mute()
    {
        if (muted == false)
        {
            muted = true;
            int numTracks = sequencer.getSequence().getTracks().length;
            for (int track = 0; track < numTracks; track++) {
              sequencer.setTrackMute(track, true);
            }
        }
    }

    @Override
    public void pause()
    {
        if (paused == false)
        {
            paused = true;
            sequencer.stop();
        }
    }

    @Override
    public void play()
    {
        if (!loaded) {
          startTime = System.currentTimeMillis();
        } else
        {
            paused = false;
            sequencer.setMicrosecondPosition(0);
            sequencer.start();
        }
    }

    @Override
    public void setLooping(boolean doLoop)
    {
        if (doLoop) {
          sequencer.setLoopCount(Sequencer.LOOP_CONTINUOUSLY);
        } else {
          sequencer.setLoopCount(0);
        }
    }

    @Override
    public void setLooping(int loops)
    {
        sequencer.setLoopCount(loops);
    }

    /**not yet implemented*/
    @Override
    public void setPan(double pan)
    {
        super.setPan(pan);
    }

    /**not yet implemented*/
    @Override
    public void setVolume(double volume)
    {
        super.setVolume(volume);
    }

    @Override
    public void stop()
    {
        sequencer.stop();
    }

    /**not yet implemented*/
    @Override
    public void turnSoundOn()
    {
        if (muted == true)
        {
            muted = false;
            int numTracks = sequencer.getSequence().getTracks().length;
            for (int track = 0; track < numTracks; track++) {
              sequencer.setTrackMute(track, false);
            }
        }
    }
    @Override
    public void setClipPosition(double time)
    {
        sequencer.setMicrosecondPosition((long)(time*1000000.0));
    }

}