package fang2.core;

import fang2.attributes.Location2D;
import fang2.attributes.Vector2D;

/**
 * The base class for all next generation transformers. Transformers are
 * "animators" which can be attached to a sprite. While updating state
 * during each video game loop, FANG calls {@link #advance(double)} for
 * every {@link TransformerAdapter}-derived class in the system. The
 * transformer also gets a chance, during each frame of the game, to
 * modify the {@link Sprite} with which it is associated. The {@link
 * #updateSprite(Sprite)} method is called each frame.
 *
 * <p>By default, {@link #updateSprite(Sprite)} updates the location,
 * rotation, and scale of the given sprite. 
 *
 * <p>Note: All values (deltaLocation, deltaRotation, deltaScale) are
 * <i>relative</i> by default. It is possible to override the update
 * methods to change that and some {@link TransformerAdapter} do ((@see
 * WrapTransformerNG} for example).
 */
public class TransformerAdapter 
  implements Transformer {
  private Location2D deltaLocation;
  private double deltaRotationRadians;
  private final double deltaScale;

  /**
   * Construct a new {@link TransformerAdapter} with identity settings.
   */
  public TransformerAdapter() {
    this(new Location2D(), 0.0, 1.0);
  }

  /**
   * Construct a new {@link TransformerAdapter} with the given values for the
   * instantaneous delta values.
   *
   * @param  deltaLocation  change in location in screens/second
   * @param  deltaRotation  change in rotation in radians/second
   * @param  deltaScale     change in scale in %/second (1.00 is
   *                        identity).
   */
  public TransformerAdapter(Location2D deltaLocation, double deltaRotation,
    double deltaScale) {
    this.deltaLocation = deltaLocation;
    this.deltaRotationRadians = deltaRotation;
    this.deltaScale = deltaScale;
  }

  /**
   * Advance the internal values of this {@link TransformerAdapter} by the
   * given time step. The result should be that the various attributes
   * of this transformer are set to the delta value for the current time
   * step.
   *
   * @param  dT  elapsed time since the last call to advance
   */
  public void advance(double dT) {
    // NOTHING TO DO HERE
  }

  /**
   * Get the amount of location this {@link TransformerAdapter} is applying
   * at the current time (instantaneous velocity).
   *
   * @return  the delta of the location in screens
   */
  public Location2D getLocation() {
    return deltaLocation;
  }

  /**
   * Get the amount of rotation this {@link TransformerAdapter} is applying
   * at the current time (instantaneous angular velocity)
   *
   * @return  the delta of the rotation in radians
   */
  public double getRotation() {
    return deltaRotationRadians;
  }

  /**
   * Get the amount of rotation this {@link TransformerAdapter} is applying
   * at the current time (instantaneous angular velocity)
   *
   * @return  the delta of the rotation in degrees
   */
  public final double getRotationDegrees() {
    return Math.toDegrees(getRotation());
  }

  /**
   * Get the amount of rotation this {@link TransformerAdapter} is applying
   * at the current time (instantaneous angular velocity)
   *
   * @return  the delta of the rotation in radians
   */
  public final double getRotationRadians() {
    return getRotation();
  }

  /**
   * Get the amount of rotation this {@link TransformerAdapter} is applying
   * at the current time (instantaneous angular velocity)
   *
   * @return  the delta of the rotation in revolutions
   */
  public final double getRotationRevolutions() {
    return getRotation() / (Math.PI * 2);
  }

  /**
   * Get the percentage scale change this {@link TransformerAdapter} is
   * applying at the current time
   *
   * @return  the delta of the scale as a fraction: 1.00 is identity
   */
  public double getScale() {
    return deltaScale;
  }

  /**
   * Called before advance on each frame. Any composite/chained {@link
   * TransformerAdapter} are required to call this on any wrapped
   * transformers every frame.
   *
   * @param  dT  time since last advance in seconds.
   */
  public void nonMaskableAdvance(double dT) {
    // NOTHING TO DO HERE
  }

  /**
   * Set the delta location (instantaneous velocity) of the {@link
   * TransformerAdapter}
   *
   * @param  location  the value to set
   */
  public void setLocation(Location2D location) {
    this.deltaLocation = location;
  }

  /**
   * Set the delta location (instantaneous velocity) of the {@link
   * TransformerAdapter}
   *
   * @param  location  the value to set
   */
  public void setLocation(Vector2D location) {
    this.deltaLocation = new Location2D(location);
  }

  /**
   * Set the delta rotation (instantaneous angular velocity) of this
   * {@link TransformerAdapter}
   *
   * @param  radians  rotation in radians
   */
  public void setRotation(double radians) {
    this.deltaRotationRadians = radians;
  }

  /**
   * Set the delta rotation (instantaneous angular velocity) of this
   * {@link TransformerAdapter}
   *
   * @param  degrees  rotation in degrees
   */
  public final void setRotationDegrees(double degrees) {
    setRotation(Math.toRadians(degrees));
  }

  /**
   * Set the delta rotation (instantaneous angular velocity) of this
   * {@link TransformerAdapter}
   *
   * @param  radians  rotation in radians
   */
  public final void setRotationRadians(double radians) {
    setRotation(radians);
  }

  /**
   * Set the delta rotation (instantaneous angular velocity) of this
   * {@link TransformerAdapter}
   *
   * @param  revolutions  rotation in revolutions
   */
  public final void setRotationRevolutions(double revolutions) {
    setRotation(revolutions * Math.PI * 2);
  }

  /**
   * Apply the delta values to the associated values in the {@link
   * Sprite}; this is a helper method which should be called during
   * advance to make sure the {@link Sprite} reflects the values
   * intended by the {@link TransformerAdapter}.
   *
   * @param  sprite  the sprite to update: updated in place!
   */
  public void updateSprite(Sprite sprite) {
    sprite.translate(deltaLocation);
    sprite.rotate(deltaRotationRadians);
    sprite.scale(deltaScale);
  }

}
