package fang2.core;

/**
 * An interface for objects which provide behaviors. Intended to
 * transform sprites. During a given instance of the game loop, FANG
 * calls nonMaskableAdvance with frame update time (intended for
 * filtering transformers). Then advance, and finally, later, every
 * sprite calls updateSprite on itself.
 */
public interface Transformer {
  /**
   * Advance this transformer one frame. This is called each frame
   * before any calls to updateSprite.
   *
   * @param  dT  time (in seconds) since last call to advance
   */
  void advance(double dT);

  /**
   * Things which must happen whether this transformer is active (or
   * even enabled) or not. One use for this is to pump filtered
   * transformers (below a Keyboard transformer which filters actual
   * advance calls based on keyboard state).
   *
   * @param  dT  time (in seconds) since last call to advance
   */
  void nonMaskableAdvance(double dT);

  /**
   * Apply current frame settings to the given sprite. Since a
   * transformer might transform multiple sprites, this is called by the
   * sprite itself in {@link Sprite#applyTransformerNG()}
   *
   * @param  sprite  the sprite to transform
   */
  void updateSprite(Sprite sprite);
}
