package fang2.core;

import fang2.attributes.Box2D;
import fang2.attributes.Location2D;
import fang2.attributes.Palette;
import fang2.attributes.Vector2D;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.GraphicsConfiguration;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.Transparency;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.VolatileImage;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * This class provides a structure for representing sprites which have
 * size, shape, orientation, and color. These sprites will appear when
 * add to the AnimationCanvas. Sprites also optionally have a tracker
 * which will update the Sprite's properties at each time interval.
 *
 * @author  Jam Jenkins and the FANG team
 */
public class Sprite {
  public static final int DEBUG_BOUNDING_BOX = 0x00000001;

  public static final int DEBUG_INTERSECTS = 0x00000002;

  private static BasicStroke debugStroke = null;

  /** default color for sprites (if no color is set/specified). */
  private static Color FANG_BLUE = Palette.getColor("FANG Blue");

  /** default color for sprites (if no color is set/specified). */
  private static Color FANG_WHITE = Palette.getColor("white");

  /** 2Pi */
  private static final double TwoPI = 2.0 * Math.PI;

  protected static Color debugBoundingBoxColor = new Color(128, 50, 50);

  public static Color getDebugBoundingBoxColor() {
    return debugBoundingBoxColor;
  }

  /**
   * gets the default color which sprites are made with
   *
   * @return  the default color of the sprites
   */
  public static Color getDefaultColor() {
    return FANG_BLUE;
  }

  /**
   * gets the default color which sprites are made with
   *
   * @return  the default color of the sprites
   */
  public static Color getDefaultOutlineColor() {
    return FANG_WHITE;
  }

  /**
   * Simulates the surface normal used for bouncing the moving object
   * off of the stationary object. Normal is in the direction from the
   * surface of the stationary object to the center of the moving
   * shape's bounding box.
   *
   * @param   stationary  the object not in motion
   * @param   moving      the object that will bounce off of the
   *                      stationary object
   *
   * @return  the angle (in standard position) of the normal in radians
   */
  public static double getNormalVector(Shape stationary, Shape moving) {
    Area movingArea = new Area(moving);
    Area stationaryArea = new Area(stationary);
    Area intersecting = new Area(stationaryArea);
    intersecting.intersect(new Area(movingArea));
    if (intersecting.isEmpty()) {
      return Double.NaN;
    }
    Rectangle2D movingBounds = moving.getBounds2D();
    Rectangle2D overlapBounds = intersecting.getBounds2D();
    Point2D.Double normal = new Point2D.Double(movingBounds
        .getCenterX() - overlapBounds.getCenterX(),
        movingBounds.getCenterY() - overlapBounds.getCenterY());
    return Math.atan2(normal.y, normal.x);
  }

  /**
   * Determines the surface normal between the moving and the stationary
   * object (the moving object's normal vector is calculated)
   *
   * @param   stationary  object in motion
   * @param   moving      object not in motion
   *
   * @return  the surface normal between the stationary and moving
   *          sprites
   */
  public static double getNormalVector(Sprite stationary,
    Sprite moving) {
    return getNormalVector(stationary.getShape(), moving.getShape());
  }

  /**
   * Determine if two objects implementing the {@link java.awt.Shape
   * Shape} interface intersect. Note that this method is not cheap (it
   * uses the {@link java.awt.geom.Area#intersect(Area)
   * Area.intersects(Area)} method.
   *
   * @param   one  one of the {@link java.awt.Shape Shape}s to be tested
   *               for intersection.
   * @param   two  the other {@link java.awt.Shape Shape} to be tested
   *               for intersection
   *
   * @return  returns true if the two shapes intersect, false otherwise
   */

  public static boolean intersects(Shape one, Shape two) {
    Area total = new Area(new Area(one));
    total.intersect(new Area(two));
    return !total.isEmpty();
  }

  /**
   * Set the color for all debug bounding boxes.
   *
   * @param  color  the {@link java.awt.Color Color} with which to
   *                display any bounding boxes used for debugging
   */
  public static void setDebugBoundingBoxColor(Color color) {
    debugBoundingBoxColor = color;
  }

  /**
   * sets the default color that all sprites start with. Calling this
   * method only affects Sprites created after calling this method.
   *
   * @param  defaultColor
   */
  public static void setDefaultColor(Color defaultColor) {
    Sprite.FANG_BLUE = defaultColor;
  }

  /**
   * Get the debug stroke (for drawing debug bounding box and facing and
   * the like). Note that the {@link #debugStroke} field is static and
   * an example of a singleton pattern. It is initialized to null; if
   * the debug stroke is ever needed, then it should be fetched by
   * calling this method so that the stroke is constructed on demand the
   * first time it is fetched.
   *
   * @return  the debug stroke for drawing boxes
   */
  private static BasicStroke getDebugStroke() {
    if (debugStroke == null) {
      debugStroke = new BasicStroke(0.01f);
    }
    return debugStroke;
  }

  /**
   * the transform of this Sprite in the context of its containing
   * graphic context (game or composite sprite)
   */
  public AffineTransform transform;

  private int debug = 0;

  /**
   * true indicates the Sprite will be removed from the AnimationCanvas
   */
  private boolean destroy = false;

  /** whether the sprite should be displayed and tracked */
  private boolean enabled = true;

  /**
   * the location when this sprite was last drawn. This value is used in
   * conjunction with the pathLength in order to draw the sprite along
   * it's previous path.
   */
  private Point2D.Double oldLocation;

  /**
   * the scale when this sprite was last drawn. This value is used in
   * conjunction with the pathLength in order to draw the sprite along
   * it's previous path.
   */
  private double oldScale;

  /**
   * how many sprites to draw in the direction of the sprite's previous
   * location. By default this is zero.
   */
  private int pathLength = 0;

  /** The stroke used to draw outlines and the like */
  private BasicStroke stroke = new BasicStroke(0.04f,
      BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL);

  /** use bounding box for bounce geometry */
  private boolean useBoundingBox = false;

  /**
   * whether the sprite should be displayed. Note: a Sprite's visibility
   * has no effect its tracking.
   */
  private boolean visible = true;

  /** the fill color of the shape, FANG_BLUE by default */
  protected Color color;

  /** device for creating VolatileImages */
  protected GraphicsConfiguration config;

  /** bitmap of the shape */
  protected VolatileImage image;

  /**
   * whether to draw the shape directly or to draw to an image first
   * then draw the image
   */
  protected boolean optimizeForBitmap = false;

  /** the outline around the shape, initially null */
  protected Shape outline;

  /** the fill color of the shape, FANG_WHITE by default */
  protected Color outlineColor;

  /** the width of the outline when shown */
  protected double outlineWidth = -1;

  /** shape should initially be centered at (0, 0) */
  protected GeneralPath shape;

  protected List<Transformer> transformers;

  /**
   * initialize to an empty shape and the default values for scale,
   * rotation, an rotation. The Sprite must be centered around (0, 0)
   * and width and height 1.
   */
  public Sprite() {
    transform = new AffineTransform();
    color = FANG_BLUE;
    outlineColor = FANG_WHITE;
    shape = new GeneralPath();
    transformers = new ArrayList<Transformer>();
  }

  /**
   * initializes to a given shape and the default values for scale and
   * rotation. The Sprite must be centered around (0, 0) and width and
   * height 1.
   *
   * @param  shape  the original shape of this sprite. Note: the size
   *                and location of the given shape are not stored
   *                within the Sprite. Instead they can be set once the
   *                Sprite has been constructed.
   */
  public Sprite(Shape shape) {
    this();
    setShape(shape);
    double startingScale = Math.max(shape.getBounds2D().getHeight(),
        shape.getBounds2D().getWidth());
    // System.out.println(this + "Starting scale: " + startingScale);
    setScale(startingScale);
  }

  /**
   * Adds a {@link fang2.core.Transformer TransformerNG} to the
   * collection of {@link fang2.core.Transformer TransformerNG}s that
   * are attached to this Sprite.
   *
   * @param  transformerToAdd  the {@link fang2.core.Transformer
   *                           Transformer} to add to the collection
   */
  public void addTransformer(Transformer transformerToAdd) {
    if(!transformers.contains(transformerToAdd)){
      transformers.add(transformerToAdd);
      dirtyTransformers();
    }
  }

  /**
   * Called after TransfomerNG are all updated. Applies all attached
   * transformers to this sprite. The application (the call to
   * updateSprite) has to come from the sprite because any number of
   * sprites might refer to a single transformer.
   */
  public void applyTransformerNG() {
    if (isEnabled()) {
      for (Transformer t : transformers) {
        t.updateSprite(this);
      }
    }
  }
  
  /**
   * This method is called every frame advance.  It is possible to
   * override this method to gain access to game resources such as
   * the player mouse and keyboard.
   */
  public void update(){}

  /**
   * Copy rhs's transformer set to our set. The sets are then
   * independent while individual transformers are shared by refernce.
   *
   * @param  rhs  the right-hand side in the assignment of transformers.
   */
  public void copyTransformersFrom(Sprite rhs) {
    for (Transformer transformer : rhs.getTransformers()) {
      addTransformer(transformer);
    }
  }

  /**
   * Turn off all transformer/tracker objects associated with this
   * {@link Sprite}
   */
  public void disableTransformer() {
    enabled = false;
  }

  /**
   * Turn on all transformer/tracker objects associated with this {@link
   * Sprite}
   */
  public void enableTransformer() {
    enabled = true;
  }

  /**
   * Move this sprite forward according to its current facing the given
   * distance
   *
   * @param  distance  distance, in screens, to move this thing forward.
   */
  public void forward(double distance) {
    translate(getFacingVector().multiply(distance));
  }

  /**
   * gets the number of sprites which are being drawn from the sprite's
   * previous location. By default, only the sprite's current location
   * is drawn. Setting the number to draw as greater than one can help
   * show the path that fast moving sprites take.
   *
   * @return  the number of sprites to draw in the direction of the
   *          previous location.
   */
  public int getBlurLength() {
    return pathLength;
  }

  /**
   * Get the bounding box around the {@code Sprite} in its current
   * position. The bounding box is based on the bounding box of the
   * value returned from the {@link #getShape} method. bcl - 20081201 -
   * modified to return the correct bounds; had been using the {@code
   * Sprite}'s {@link #getX} and {@link #getY} method results which put
   * the upper-left corner of the box in the center of the sprite. This
   * is a fundamental change and should be thoroughly vetted.
   *
   * @return  minimal {@link Box2D} containing all of the the bits of
   *          this {@code Sprite}
   */
  public Box2D getBounds2D() {
    Rectangle2D bounds = getShape().getBounds2D();
    return new Box2D(bounds);

    /*
     * return new Box2D(bounds.getMinX() - bounds.getWidth() / 2,
     * bounds.getMinY() - bounds.getHeight() / 2, bounds.getWidth(),
     * bounds.getHeight());
     */
  }

  /**
   * returns the fill color of the Sprite
   *
   * @return  the fill color
   */
  public Color getColor() {
    return color;
  }

  /**
   * gets the portion of the screen each dash takes up. The return value
   * of this method changes as the size of the sprite changes.
   *
   * @return  the portion of the screen each dash takes up
   */
  public double getDashLength() {
    float[] dashes = stroke.getDashArray();
    if (dashes.length == 0) {
      return 0;
    } else {
      return dashes[0] * getSize();
    }
  }

  /**
   * gets the on/off pattern of the dashes used in drawing the outline.
   * The pattern is an ordered list of on and off alternating. For
   * example 0.2f, 0.1f, 0.4f, 0.3f means 0.2f of outline, skip 0.1f,
   * 0.4f of outline skip 0.3f then repeat. The pattern sizes change
   * when the size of the sprite changes.
   *
   * @return  the on/off pattern of the dashes used in drawing the
   *          outline
   */
  public float[] getDashPattern() {
    float[] dashes = stroke.getDashArray();
    for (int i = 0; i < dashes.length; i++) {
      dashes[i] *= getSize();
    }
    return dashes;
  }

  /**
   * Get the current debug setting of this {@code Sprite}.
   *
   * @return  debug flag value
   */
  public int getDebug() {
    return debug;
  }

  /**
   * Get the vector along which we are facing.
   *
   * @return  The facing vector.
   */
  public Vector2D getFacingVector() {
    // -rotation because of reversed y-axis in the screen space
    return new Vector2D(-getRotationDegrees(), 1);
  }

  /**
   * Get the height (in screens) of the {@code Sprite}.
   *
   * @return  height (in screens) of the sprite
   */
  public double getHeight() {
    return getBounds2D().getHeight();
  }

  /**
   * the location of the shape on the canvas
   *
   * @return  the location of the shape
   */
  public Location2D getLocation() {
    return new Location2D(internalGetLocation());
  }

  /**
   * the maximum x-value covered by this {@code Sprite}; in screens
   *
   * @return  maximum x-value in screens of area covered by this
   *          sprite's bounding box
   */
  public double getMaxX() {
    return getBounds2D().getMaxX();
  }

  /**
   * the maximum y-value covered by this {@code Sprite}; in screens
   *
   * @return  maximum y-value in screens of area covered by this
   *          sprite's bounding box
   */
  public double getMaxY() {
    return getBounds2D().getMaxY();
  }

  /**
   * the minimum x-value covered by this {@code Sprite}; in screens
   *
   * @return  minimum x-value in screens of area covered by this
   *          sprite's bounding box
   */
  public double getMinX() {
    return getBounds2D().getMinX();
  }

  /**
   * the minimum y-value covered by this {@code Sprite}; in screens
   *
   * @return  minimum y-value in screens of area covered by this
   *          sprite's bounding box
   */
  public double getMinY() {
    return getBounds2D().getMinY();
  }

  /**
   * gets whether the shape should be drawn directly to the screen or if
   * the shape should first be drawn to an image and then the image is
   * used to draw the shape to the screen. When the sprite is optimized
   * for bitmaps, potentially complex shapes with curves and/or many
   * vertexes can be simplified into a 2D array of pixels using the
   * VolatileImage. If the shape is not complex, then it is probably
   * just faster to draw the shape directly to the screen. By default,
   * all Sprites are not optimized for bitmap. The decision on whether
   * to use bitmaps or not is based upon the speed with or without using
   * the bitmaps since the rendering will look identical.
   *
   * @return  true if the shapes are flattened into pixels on a 2D image
   *          and this image should be used for drawing to screen. False
   *          indicates the shape is drawn directly to screen without
   *          first caching it into a VolatileImage.
   */
  public boolean getOptimizedForBitmap() {
    return optimizeForBitmap;
  }

  /**
   * Get the current orientation of the sprite in radians. Orientation
   * is a synonym for Rotation.
   *
   * @return  rotation from initial facing in radians
   */
  public double getOrientation() {
    return getRotation();
  }

  /**
   * Get the current orientation of the sprite in degrees. Orientation
   * is a synonym for Rotation.
   *
   * @return  rotation from initial facing in degrees
   */
  public double getOrientationDegrees() {
    return getRotationDegrees();
  }

  /**
   * Get the current orientation of the sprite in radians. Orientation
   * is a synonym for Rotation.
   *
   * @return  rotation from initial facing in radians
   */
  public double getOrientationRadians() {
    return getRotationRadians();
  }

  /**
   * Get the number of full turns in the current rotation. Note that
   * this returns a <b>fractional</b> part of the last rotation.
   * Orientation is a synonym for Rotation.
   *
   * @return  fractional number of full turns in the current rotation
   */
  public double getOrientationRevolutions() {
    return getRotationRevolutions();
  }

  /**
   * returns the outline color of the Sprite
   *
   * @return  the outline color
   */
  public Color getOutlineColor() {
    return outlineColor;
  }

  /**
   * gets the portion of the screen used in drawing the outline. Note:
   * this method returns the width of the outline without respect to
   * whether the outline is currently being drawn or not. Use the method
   * isOutlineVisible to determine if the outline is currently being
   * shown.
   *
   * @return  thickness of the outline in screens
   */
  public double getOutlineThickness() {
    return outlineWidth / 2 * getScale();
  }

  /**
   * gets the current amount of rotation in radians
   *
   * @return  the positive rotation in radians less than 2*PI and zero
   *          or more
   */
  public double getRotation() {
    AffineTransform copy = new AffineTransform(transform);
    copy.translate(-copy.getTranslateX(), -copy.getTranslateY());
    double scale = internalGetScale();
    copy.scale(1 / scale, 1 / scale);
    double theta = Math.acos(copy.getScaleX());
    if (Math.asin(-copy.getShearX()) < 0) {
      theta *= -1;
    }
    return theta;
  }

  /**
   * gets the orientation of the sprite in degrees. This method calls
   * getRotation and converts the radians into degrees.
   *
   * @return  the rotation in degrees
   */
  public double getRotationDegrees() {
    return Math.toDegrees(getRotation());
  }

  /**
   * get the orientation of the sprite in radians
   *
   * @return  the rotation in degrees
   */
  public double getRotationRadians() {
    return getRotation();
  }

  /**
   * gets the orientation of the sprite in revolutions. This method
   * calls getRotation and converts the radians into revolutions.
   *
   * @return  the rotation in revolutions
   */
  public double getRotationRevolutions() {
    return getRotation() / TwoPI;
  }

  /**
   * the scaling factor
   *
   * @return  the ratio of the current size to the orignal size of the
   *          shape
   */
  public double getScale() {
    return internalGetScale();
  }

  /**
   * gets the shape in its current position, size, and orientation
   *
   * @return  the shape
   */
  public Shape getShape() {
    return shape.createTransformedShape(transform);
  }

  public double getSize() {
    return getScale();
  }

  /**
   * determines whether the bounding box is used for intersections of
   * the exact shape is used
   *
   * @return  true if a bounding box is being used in tests for
   *          intersection, false if the actual shape is being used
   */
  public boolean getUseBoundingBox() {
    return useBoundingBox;
  }

  public double getWidth() {
    return getBounds2D().getWidth();
  }

  /**
   * returns the x-coordinate of the sprite on the canvas
   *
   * @return  x-coordinate of sprite
   */
  public double getX() {
    return internalGetLocation().x;
  }

  /**
   * returns the y-coordinate of the sprite on the canvas
   *
   * @return  y-coordinate of sprite
   */
  public double getY() {
    return internalGetLocation().y;
  }

  /**
   * makes the sprite invisible. This is the same as calling
   * setVisible(false). Sprites are visible by default. When using hide
   * sprites continue to be affected by Trackers. removeFromCanvas is
   * different from hide in that once removed from the canvas, sprites
   * are no longer affected by Trackers.
   */
  public void hide() {
    setVisible(false);
  }

  /**
   * makes the outline not shown. By default, the outline of shapes is
   * not drawn.
   */
  public void hideOutline() {
    outline = null;
  }

  /**
   * determines if a point is within the sprite's shape
   *
   * @param   x  x-coordinate of the point to test
   * @param   y  y-coordinate of the point to test
   *
   * @return  true if point is in the shape, false otherwise
   */
  public boolean intersects(double x, double y) {
    Area area = new Area(getShape());
    return area.contains(x, y);
  }

  /**
   * determines if a point is within the sprite's shape
   *
   * @param   point  the point to test
   *
   * @return  true if point is in the shape, false otherwise
   */
  public boolean intersects(Location2D point) {
    return intersects(point.x, point.y);
  }

  /**
   * determines if a point is within the sprite's shape
   *
   * @param   point  the point to test
   *
   * @return  true if the point is in the shape, false otherwise
   */
  public boolean intersects(Point2D.Double point) {
    return intersects(point.x, point.y);
  }

  /**
   * determines if another Sprite intersects this Sprite. Intersection
   * is a geometric property of the sprites and does not depend on
   * whether the sprites are on the canvas or visible.
   *
   * @param   sprite  another Sprite
   *
   * @return  true if the other Sprite intersects this Sprite, false if
   *          there is no intersection
   */
  public boolean intersects(Sprite sprite) {
    // bcl - uses fixed getBounds2D rather than Shape.
    Rectangle2D spriteBounds = sprite.getBounds2D().Rectangle2D();
    Rectangle2D bounds = getBounds2D().Rectangle2D();

    if ((debug & DEBUG_INTERSECTS) != 0) {
      System.out.println("intersects: ");
      System.out.println("  this = " + this + " bounds = [" +
        bounds.getX() + ", " + bounds.getY() + ", " +
        bounds.getWidth() + ", " + bounds.getHeight() + "]");
      System.out.println("  sprite = " + sprite + " spriteBounds = " +
        spriteBounds);
    }

    // check simplest case first:
    // if axis-aligned bounding boxes don't intersect, shapes can't
    if (!bounds.intersects(spriteBounds)) {
      return false;
    }
    // now check to see if rotated bounding boxes intersect
    Area boundsRotated = getUnrotatedBoundaryArea();
    Area spriteBoundsRotated = sprite.getUnrotatedBoundaryArea();
    
    boundsRotated = rotate(boundsRotated, getRotation()); 
    spriteBoundsRotated = rotate(spriteBoundsRotated, sprite.getRotation()); 

    // if neither rotated bounding box intersect with
    // the other upright bounding box, then sprites cannot intersect
    if (!boundsRotated.intersects(spriteBounds) ||
        !spriteBoundsRotated.intersects(bounds)) {
      return false;
    }

    // if rotated bounding boxes do not intersect with
    // each other, sprites cannot intersect
    Area rotatedIntersection = new Area(boundsRotated);
    rotatedIntersection.intersect(spriteBoundsRotated);
    if (rotatedIntersection.isEmpty()) {
      return false;
    }

    // if both are using bounding boxes,
    // then they do intersect
    if (sprite.getUseBoundingBox() && getUseBoundingBox()) {
      return true;
    }

    // since the boxes do intersect, must determine
    // if the shape and box intersect
    else if (getUseBoundingBox()) {
      // they only intersect if the sprite shape
      // intersects both the upright and the rotated
      // bounding boxes of this
      Area transformedArea = new Area(sprite.getShape());
      transformedArea.intersect(boundsRotated);
      return sprite.getShape().intersects(bounds) &&
        !transformedArea.isEmpty();
    }
    // since the boxes do intersect, must determine
    // if the shape and box intersect
    else if (sprite.getUseBoundingBox()) {
      // they only intersect if this shape
      // intersects both the upright and the rotated
      // bounding boxes of the sprite
      Area transformedArea = new Area(getShape());
      transformedArea.intersect(spriteBoundsRotated);
      return getShape().intersects(spriteBounds) &&
        !transformedArea.isEmpty();
    }
    // hardest case: boxes intersect and can't use bounding
    // boxes, must use intersection of actual shapes
    // this could take time if the shapes are complex
    else {
      Area one = new Area(getShape());
      Area two = new Area(sprite.getShape());
      one.intersect(two);
      return !one.isEmpty();
    }
  }
  
  private Area rotate( Area area, double radians )
  {
    Rectangle2D bounds2D = area.getBounds2D();
    AffineTransform affine = new AffineTransform();
    affine.translate( bounds2D.getCenterX(), bounds2D.getCenterY() );
    affine.rotate( radians );
    affine.translate( -bounds2D.getCenterX(), -bounds2D.getCenterY() );
    area.transform( affine );
    return area;
  }  

  /**
   * determines the Sprite's ability to display and update
   *
   * @return  true if Sprite can display and update, false otherwise
   */
  public boolean isEnabled() {
    return enabled;
  }

  /**
   * returns whether the Sprite is currently displayable
   *
   * @return  true if Sprite can be displayed, false otherwise
   */
  public boolean isVisible() {
    return visible;
  }

  /**
   * draws the shape in the proper location, orientation, and size If
   * overridden, this method must not add or remove sprites from the
   * canvas. If adding or removing sprites is desired, schedule this
   * action to occur zero seconds from now by creating an alarm which
   * performs the desired adding or removing.
   *
   * @param  brush  the Graphics used to draw the shape
   */
  public void paint(Graphics2D brush) {
    if (optimizeForBitmap) {
      if (image == null) {
        config = brush.getDeviceConfiguration();
        image = createImage();
        restoreImage();
      }
      if (image.contentsLost()) {
        restoreImage();
      }
      Point2D.Double location = internalGetLocation();
      brush.drawImage(image, (int) location.x - (image.getWidth() / 2),
        (int) location.y - (image.getHeight() / 2), null);
    } else {
      AffineTransform original = brush.getTransform();
      Color originalColor = brush.getColor();
      brush.transform(transform);
      brush.setColor(color);

      paintShape(brush);

      if (outline != null) {
        brush.setColor(outlineColor);
        brush.fill(outline);
      }
      brush.setColor(originalColor);
      brush.setTransform(original);
    }
    if ((debug & DEBUG_BOUNDING_BOX) != 0) {
      drawDebugShapes(brush);
    }
  }

  /**
   * provides a mechanism for enabling and visibility. This method also
   * handles drawing sprites in the direction of the previous location
   * for fast moving sprites with the blur length set to greater than
   * zero.
   *
   * @param  brush  the Graphics2D to draw on
   */
  public void paintInternal(Graphics2D brush) {
    if (oldLocation == null) {
      oldLocation = getLocation();
      oldScale = getScale();
    }
    double currentScale = getScale();
    Point2D.Double currentLocation = getLocation();
    if (visible && enabled) {
      if ((pathLength > 0) &&
          (!oldLocation.equals(currentLocation) ||
            (oldScale != currentScale))) {
        Color original = getColor();
        int alpha = original.getAlpha();
        for (int i = 0; i < pathLength; i++) {
          double factor = (i + 1.0) / (pathLength + 2.0);
          setColor(new Color(original.getRed(), original.getGreen(),
              original.getBlue(), (int) (alpha * factor)));
          setScale(oldScale + (factor * (currentScale - oldScale)));
          setLocation(oldLocation.x +
            (factor * (currentLocation.x - oldLocation.x)),
            oldLocation.y +
            (factor * (currentLocation.y - oldLocation.y)));
          paint(brush);
        }
        setColor(original);
        setScale(currentScale);
        setLocation(currentLocation);
        paint(brush);
      } else {
        paint(brush);
      }
    }
    oldScale = currentScale;
    oldLocation = currentLocation;
  }

  /**
   * removes the Sprite from the canvas the next time advanceFrame is
   * called. removeFromCanvas is different from hide (or
   * setVisible(false)) in that once removed from the canvas, sprites
   * are no longer affected by Trackers. When using hide (or
   * setVisible(false)) sprites continue to be affected by Trackers.
   */
  public void removeFromCanvas() {
    destroy = true;
    enabled = false;
  }

  /**
   * Remove the given {@link Transformer} from the list of
   * transformers connected to this {@link Sprite}.
   *
   * @param  transformToEliminate  the {@link Transformer} to remove
   */
  public void removeTransformer(Transformer transformToEliminate) {
    transformers.remove(transformToEliminate);
    dirtyTransformers();
  }

  /**
   * sets the orientation with regard to the current orientation
   *
   * @param  rotation  the additional rotation in radians
   */
  public void rotate(double rotation) {
    setRotation(getRotation() + rotation);
  }

  /**
   * sets the orientation with regard to the current orientation. This
   * method converts the degrees to radians and calls rotate.
   *
   * @param  degrees  the additional rotation in degrees
   */
  public void rotateDegrees(double degrees) {
    rotate(Math.toRadians(degrees));
  }

  public void rotateRadians(double radians) {
    rotate(radians);
  }

  /**
   * sets the orientation with regard to the current orientation. This
   * method converts the revolutions to radians and calls rotate. One
   * revolution is the same as 360 degrees or 2*PI radians.
   *
   * @param  revolutions  the additional rotation in revolution
   */
  public void rotateRevolutions(double revolutions) {
    rotate(revolutions * Math.PI * 2);
  }

  /**
   * scale with regard to the current scaling. The resulting scale will
   * be the product of the current scale and the scale parameter.
   *
   * @param  scale  the factor by which to change the current scale,
   *                must be nonzero
   */
  public void scale(double scale) {
    setScale(scale * internalGetScale());
  }

  /**
   * adds a sequence of translucent sprites from the sprite's previous
   * location to the current location in increasing opacity. By default,
   * only the sprite's current location is drawn. Setting the number to
   * draw as greater than one can help show the path that fast moving
   * sprites take.
   *
   * @param  length  how many sprites to draw from the previous position
   */
  public void setBlurLength(int length) {
    pathLength = length;
  }

  /**
   * determines the fill color of the shape. All shapes are filled by
   * default and the default color of the fill is blue.
   *
   * @param  c  the fill color
   */
  public void setColor(Color c) {
    color = c;
    if (optimizeForBitmap) {
      restoreImage();
    }
  }

  /**
   * sets the length of the dashes used in drawing the outline. This
   * length changes when the size of the sprite changes.
   *
   * @param  length  the length of each dash as a portion of the screen
   */
  public void setDashLength(double length) {
    float[] dashes = {
        (float) (length / getSize()), (float) (length / getSize())
      };
    stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(),
        stroke.getLineJoin(), stroke.getMiterLimit(), dashes,
        stroke.getDashPhase());
    if (outline != null) {
      outline = null;
      showOutline();
    }
  }

  /**
   * sets the on/off pattern of the dashes used in drawing the outline.
   * The pattern is an ordered list of on and off alternating. For
   * example 0.2f, 0.1f, 0.4f, 0.3f means 0.2f of outline, skip 0.1f,
   * 0.4f of outline skip 0.3f then repeat. The pattern sizes change
   * when the size of the sprite changes.
   *
   * @param  dashes  the on/off pattern of the dashes used in drawing
   *                 the outline
   */
  public void setDashPattern(float... dashes) {
    for (int i = 0; i < dashes.length; i++) {
      dashes[i] /= getSize();
    }
    stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(),
        stroke.getLineJoin(), stroke.getMiterLimit(), dashes,
        stroke.getDashPhase());
    if (outline != null) {
      outline = null;
      showOutline();
    }
  }

  /**
   * Set the debug flag for this {@code Sprite} to the given value.
   * Currently defined values for this flag (which can be {@code |}'ed
   * together) are: {@code Sprite. DEBUG_BOUNDING_BOX} - draw bounding
   * boxes around the sprite every frame from now on. {@code
   * DEBUG_INTERSECTS} - dump debug data to stderr during collision
   * detection tests.
   *
   * @param  i  New value for the debug flag.
   */
  public void setDebug(int i) {
    debug = i;
  }

  /**
   * sets the Sprite's ability to display and update
   *
   * @param  able  true indicates the Sprite can display and update,
   *               false otherwise
   */
  public void setEnabled(boolean able) {
    enabled = able;
  }

  /**
   * sets the location without regard to its current location
   *
   * @param  x  the horizontal location (in screens)
   * @param  y  the vertical location (in screens)
   */
  public void setLocation(double x, double y) {
    transform.setTransform(transform.getScaleX(), transform.getShearY(),
      transform.getShearX(), transform.getScaleY(), x, y);
  }

  /**
   * sets the location without regard to its current location
   *
   * @param  location  the sprite's new location (in screens)
   */

  public void setLocation(Location2D location) {
    setLocation(location.x, location.y);
  }

  /**
   * sets the location without regard to its current location
   *
   * @param  location  the new location
   */
  public void setLocation(Point2D.Double location) {
    setLocation(location.x, location.y);
  }

  /**
   * Note: this method is current disabled. If the method is desired,
   * email the author of the FANG Engine to request it be added back in.
   * sets whether the shape should be drawn directly to the screen or if
   * the shape should first be drawn to an image and then the image is
   * used to draw the shape to the screen. When the sprite is optimized
   * for bitmaps, potentially complex shapes with curves and/or many
   * vertexes can be simplified into a 2D array of pixels using the
   * VolatileImage. If the shape is not complex, then it is probably
   * just faster to draw the shape directly to the screen. By default,
   * all Sprites are not optimized for bitmap. The decision on whether
   * to use bitmaps or not is based upon the speed with or without using
   * the bitmaps since the rendering will look identical.
   *
   * @param  enableBitmap  true indicates the shapes should be flattened
   *                       into pixels on a 2D image and this image
   *                       should be used for drawing to screen. False
   *                       indicates the shape should be drawn directly
   *                       to screen without first caching it into a
   *                       VolatileImage.
   */
  public void setOptimizedForBitmap(boolean enableBitmap) {
    // optimizeForBitmap=enableBitmap;
  }

  /**
   * set the rotation of this {@code Sprite} without regard to its
   * current rotation.
   *
   * @param  radians  new rotation value in CCW radians away from the
   *                  positive x-axis
   */
  public void setOrientation(double radians) {
    setRotation(radians);
  }

  /**
   * set the rotation of this {@code Sprite} without regard to its
   * current rotation.
   *
   * @param  degrees  new rotation value in CCW degrees from the
   *                  positive x axis
   */
  public void setOrientationDegrees(double degrees) {
    setRotation(Math.toRadians(degrees));
  }

  /**
   * set the rotation of this {@code Sprite} without regard to its
   * current rotation.
   *
   * @param  radians  new rotation value in CCW radians away from the
   *                  positive x-axis
   */
  public void setOrientationRadians(double radians) {
    setRotation(radians);
  }

  /**
   * set the rotation of this {@code Sprite} without regard to its
   * current rotation.
   *
   * @param  revolutions  new rotation value in CCW revolutions (full
   *                      circle = 1.0) away from the positive x-axis
   */
  public void setOrientationRevolutions(double revolutions) {
    setRotation(revolutions * Math.PI * 2.0);
  }

  /**
   * determines the outline color of the shape. By default the outline
   * is white and not visible. In order to see the outline, call
   * showOutline.
   *
   * @param  c  the outline color
   */
  public void setOutlineColor(Color c) {
    outlineColor = c;
  }

  /**
   * sets the portion of the screen that the outline should fill. The
   * outline of the shape is drawn around the edges of the shape and is
   * always in the internal part of the shape.
   *
   * @param  thickness  the width of the outline as a portion of the
   *                    screen
   */
  public void setOutlineThickness(double thickness) {
    if (thickness >= 0) {
      outlineWidth = thickness * 2 / getScale();
    } else {
      outlineWidth = 0;
    }
    if (outline != null) {
      outline = null;
      showOutline();
    }
  }

  /**
   * this is a helper method that either calls showOutline or
   * hideOutline
   *
   * @param  visible  true means call showOutline, false means call
   *                  hideOutline
   */
  public void setOutlineVisible(boolean visible) {
    if (visible) {
      showOutline();
    } else {
      hideOutline();
    }
  }

  /**
   * sets the orientation without regard to the current orientation
   *
   * @param  rotation  the new orientation in radians counter-clockwise
   *                   for the original orientation
   */
  public void setRotation(double rotation) {
    transform.rotate(rotation - getRotation());
    if (optimizeForBitmap) {
      restoreImage();
    }
  }

  /**
   * sets the orientation without regard to the current orientation.
   * This method converts the degrees to radians and calls setRotation.
   *
   * @param  degrees  the new orientation in degrees from the original
   *                  orientation
   */
  public void setRotationDegrees(double degrees) {
    setRotation(Math.toRadians(degrees));
  }

  /**
   * sets the orientation without regard to the current orientation. The
   * method uses radians directly.
   *
   * @param  radians  the new orientation in radians counter clockwise
   *                  from the original orientation
   */
  public void setRotationRadians(double radians) {
    setRotation(radians);
  }

  /**
   * sets the orientation without regard to the current orientation.
   * This method converts the revolutions to radians and calls
   * setRotation. One revolution is the same as 360 degrees or 2*PI
   * radians.
   *
   * @param  revolutions  the new orientation in revolutions from the
   *                      original orientation
   */
  public void setRotationRevolutions(double revolutions) {
    setRotation(revolutions * Math.PI * 2);
  }

  /**
   * sets the absolute scale without regard to its current value, must
   * be nonzero. If zero is sent in as the parameter, a very small
   * number will be used instead.
   *
   * @param  scale  the new scale
   */
  public void setScale(double scale) {
    if (scale == 0) {
      scale = 0.00001;
    }
    double determinant =
      (transform.getScaleX() * transform.getScaleY()) -
      (transform.getShearX() * transform.getShearY());
    scale = scale / Math.sqrt(determinant);
    if (scale == 1) {
      return;
    }
    transform.setTransform(scale * transform.getScaleX(),
      scale * transform.getShearY(), scale * transform.getShearX(),
      scale * transform.getScaleY(), transform.getTranslateX(),
      transform.getTranslateY());
    if (optimizeForBitmap) {
      restoreImage();
    }
  }

  public void setSize(double size) {
    setScale(size);
  }

  /**
   * Sets the parameter which determines if the actual shape or its
   * minimal bounding box is used in tests for intersection. Using exact
   * intersection is the default.
   *
   * @param  box  true indicates a bounding box should be used, false
   *              indicates the actual shape should be used
   */
  public void setUseBoundingBox(boolean box) {
    useBoundingBox = box;
  }

  /**
   * sets if the Sprite will be displayed when added to an
   * AnimationCanvas. Calling setVisible(true) is the same as calling
   * show(). Calling setVisible(false) is the same as calling hide().
   * When using hide sprites continue to be affected by Trackers.
   * removeFromCanvas is different from hide in that once removed from
   * the canvas, sprites are no longer affected by Trackers.
   *
   * @param  vis  true indicates to display, false to hide
   */
  public void setVisible(boolean vis) {
    visible = vis;
  }

  /**
   * sets the location's x-coordinate without regard to the current
   * location. y-coordinate is unchanged
   *
   * @param  newX  - new x-coordinate
   */
  public void setX(double newX) {
    setLocation(newX, internalGetLocation().y);
  }

  /**
   * sets the location's y-coordinate without regard to the current
   * location. x-coordinate is unchanged
   *
   * @param  newY  - new y-coordinate
   */
  public void setY(double newY) {
    setLocation(internalGetLocation().x, newY);
  }

  /**
   * makes the sprite visible if it is on the canvas. Sprites are
   * visible by default.
   */
  public void show() {
    setVisible(true);
  }

  /**
   * makes the outline of the shape appear around the edges of the
   * shape. Note: this method does not change the overall shape -
   * instead it makes visible the edges by painting a line around them.
   * If this is the first time the method is called, the thickness of
   * the outline will be set to a default value. This outline thickness
   * will change as the size of the sprite changes.
   */
  public void showOutline() {
    if (outlineWidth < 0) {
      outlineWidth = 0.02 / getScale();
    }
    if (outline == null) {
      stroke = new BasicStroke((float) (outlineWidth),
          stroke.getEndCap(), stroke.getLineJoin(),
          stroke.getMiterLimit(), stroke.getDashArray(), 0.0f);
      // the below two lines simplify the shape to make sure outlines
      // work correctly
      // sometimes shapes have internal lines which would cause
      // undesired artifacts
      // when drawing the outline
      Area cleanerShape = new Area(shape.getBounds2D());
      cleanerShape.intersect(new Area(shape));
      Area area = new Area(stroke.createStrokedShape(cleanerShape));
      area.intersect(new Area(shape));
      outline = area;
    }
  }

  /**
   * Return a string representation of the {@code Sprite}. The type,
   * location, size, and color of the {@code Sprite}.
   */
  @Override
  public String toString() {
    return getClass().getName() + "[x = " + getX() + ", y = " + getY() +
      "], " + "[w = " + getWidth() + ", h = " + getHeight() +
      "] color = " + Palette.getColorName(getColor());
  }

  /**
   * moves the shape with regard to its current location
   *
   * @param  x  the amount to move horizontally location
   * @param  y  the amount to move vertically location
   */
  public void translate(double x, double y) {
    setLocation(x + transform.getTranslateX(),
      y + transform.getTranslateY());
  }

  /**
   * Moves the shape with regard to its current location
   *
   * @param  delta  the directional info for the move
   */
  public void translate(Location2D delta) {
    translate(delta.x, delta.y);
  }

  /**
   * moves the shape with regard to its current location
   *
   * @param  delta  the amount and direction to move the shape from its
   *                current location
   */
  public void translate(Point2D.Double delta) {
    translate(delta.x, delta.y);
  }

  /**
   * Moves the shape with regard to its current location
   *
   * @param  deltaV  the vector along which to move.
   */
  public void translate(Vector2D deltaV) {
    translate(deltaV.getX(), deltaV.getY());
  }

  /**
   * moves sprite horizontally relative to current location
   *
   * @param  deltaX  - amount to move horizontally
   */
  public void translateX(double deltaX) {
    translate(deltaX, 0.0);
  }

  /**
   * moves sprite vertically relative to current location
   *
   * @param  deltaY  - amount to move vertically
   */
  public void translateY(double deltaY) {
    translate(0.0, deltaY);
  }

  /**
   * creates a volatile image appropriately sized
   *
   * @return  a volatile image which will be used to display the sprite
   */
  private VolatileImage createImage() {
    Rectangle2D bounds = getShape().getBounds2D();
    return config.createCompatibleVolatileImage(
        (int) bounds.getWidth() + 1, (int) bounds.getHeight() + 1,
        Transparency.TRANSLUCENT);
  }

  /**
   * Inform the game (or the game's canvas) that the transformer list it
   * has is dirty and must be reloaded.
   */
  private void dirtyTransformers() {
    AnimationCanvas canvas = Game.getCurrentGame().getCanvas();
    if ((canvas != null)) {// && canvas.containsSprite(this)) {
      canvas.allTransformersDirty = true;
    }
  }

  private Area getUnrotatedBoundaryArea() {
    double originalRotation = getRotation();

    setRotation(0);// rotate back to cannonical orientation
    // bcl - using fixed getBounds2D
    Rectangle2D unrotatedBoundingBox = getBounds2D().Rectangle2D();
    setRotation(originalRotation);// rotate back to original
    // orientation

    GeneralPath path = new GeneralPath(unrotatedBoundingBox);
    return new Area(path);
  }

  /**
   * uses the transform to get the location
   *
   * @return  the location as obtained from the transform
   */
  private Point2D.Double internalGetLocation() {
    return new Point2D.Double(transform.getTranslateX(),
        transform.getTranslateY());
  }

  /**
   * gets the square root of the determinant of the transformation
   * matrix
   *
   * @return  the scaling factor
   */
  private double internalGetScale() {
    double determinant =
      (transform.getScaleX() * transform.getScaleY()) -
      (transform.getShearX() * transform.getShearY());
    return Math.sqrt(determinant);
  }

  /**
   * called when the volatile image expires. This method is only called
   * when optimizing the sprite for bitmap display.
   */
  private void restoreImage() {
    if (image == null) {
      return;
    }
    int status = image.validate(config);
    Rectangle2D bounds = getShape().getBounds2D();
    if ((status == VolatileImage.IMAGE_INCOMPATIBLE) ||
        (bounds.getWidth() > image.getWidth()) ||
        (bounds.getHeight() > image.getHeight())) {
      image = createImage();
    }
    Graphics2D brush = image.createGraphics();
    brush.setBackground(new Color(0, 0, 0, 0));
    brush.clearRect(0, 0, image.getWidth(), image.getHeight());
    brush.translate(-bounds.getMinX() +
      ((image.getWidth() - bounds.getWidth()) / 2),
      -bounds.getMinY() +
      ((image.getHeight() - bounds.getHeight()) / 2));
    brush.setColor(color);
    RenderingHints hints = new RenderingHints(null);
    hints.put(RenderingHints.KEY_RENDERING,
      RenderingHints.VALUE_RENDER_SPEED);
    hints.put(RenderingHints.KEY_ANTIALIASING,
      RenderingHints.VALUE_ANTIALIAS_ON);
    brush.addRenderingHints(hints);
    brush.fill(getShape());
  }

  /**
   * Draw debug shapes. This method draws the bounding box.
   *
   * @param  brush  The graphics context in which to draw the box
   */
  protected void drawDebugShapes(Graphics2D brush) {
    setDebugBrush(debugBoundingBoxColor, brush);
    Box2D bounds = getBounds2D();
    brush.draw(bounds.Rectangle2D());
  }

  /**
   * Return the list of transformers.
   *
   * @return  the list of all of the transformers currently on this
   *          sprite. Reference is live.
   */
  protected List<Transformer> getTransformers() {
    return transformers;
  }

  /**
   * determines if the sprite should be removed from the AnimationCanvas
   *
   * @return  true if Sprite should be removed, false otherwise
   */
  protected boolean isDestroyed() {
    return destroy;
  }

  /**
   * resizes to be 1 by 1 and centers around origin (0, 0)
   */
  protected void normalize() {
    Rectangle2D bounds = shape.getBounds2D();
    double max = Math.max(bounds.getWidth(), bounds.getHeight());
    // resizes to be 1x1
    shape.transform(AffineTransform.getScaleInstance(1 / max, 1 / max));
    bounds = shape.getBounds2D();
    // centers around (0, 0)
    shape.transform(AffineTransform.getTranslateInstance(
        (-bounds.getWidth() / 2) - bounds.getX(),
        (-bounds.getHeight() / 2) - bounds.getY()));
  }

  protected void paintShape(Graphics2D brush) {
    brush.fill(shape);
  }

  /**
   * sets shape to shape s and uses the location and size information.
   * Calling this method affects the scale and location of the sprite.
   * The new size is the maximum dimension of the shape and the new
   * location is the center of the shape.
   *
   * @param  s  the new shape of the sprite
   */
  protected void setAbsoluteShape(Shape s) {
    Rectangle2D bounds = s.getBounds2D();
    double scale = Math.max(bounds.getWidth(), bounds.getHeight());
    double x = bounds.getCenterX();
    double y = bounds.getCenterY();
    setShape(s);
    setLocation(x, y);
    setSize(scale);
  }

  /**
   * Sets the drawing context's brush to be a thin brush with the given
   * color; this method is provided so that Sprite subclasses can easily
   * set up to paint debugging information.
   *
   * @param  brushColor
   * @param  brush
   */
  protected void setDebugBrush(Color brushColor, Graphics2D brush) {
    brush.setStroke(getDebugStroke());
    brush.setColor(brushColor);
  }

  /**
   * sets shape to shape s. Calling this method has no effect on the
   * scale, location, or orientation of the sprite
   *
   * @param  s  the new shape of the sprite
   */
  protected void setShape(Shape s) {
    shape.reset();
    shape.append(s, true);
    normalize();
  }

  /**
   * Add all {@link Transformer} associated with this sprite to the
   * allTransformers set.
   *
   * @param  allTransformers  the set of all transformers; provided by
   *                          the caller.
   */
  /* package */ public void reportTransformers(Set<Transformer> allTransformers) {
    if (isEnabled()) {
      for (Transformer t : transformers) {
        allTransformers.add(t);
      }
    }
  }
}
