package fang2.core;

import java.lang.ref.WeakReference;
import java.net.URL;
import java.util.ArrayList;

import fang2.media.MidiSound;
import fang2.media.RawMediaCache;
import fang2.media.SampledSound;
import fang2.media.SequentialSound;


/**
 * This class that is capable of playing sampled
 * sounds and MIDIs.  Games should use this class
 * to play sounds.  The formats wav and midi are
 * automatically supported.  
 * 
 * In order to play mp3
 * files, you need to add the following jar files
 * to your classpath.  All are available from the
 * <a href="http://www.javazoom.net/mp3spi/mp3spi.html">
 * JavaZoom website</a>.
 * <ol><li>Java Layer 1.0 (jl1.0.jar)
 *     <li>Tritonus Share (tritonus_share-0.3.6.jar)
 *     <li>MP3SPI (mp3spi1.9.4.jar)
 * </ol>
 *
 * In order to play ogg vorbis
 * files, you need to add the following jar files
 * to your classpath.  All are available from the
 * <a href="http://www.javazoom.net/vorbisspi/vorbisspi.html">
 * JavaZoom website</a>.
 * <ol><li>JOGG from JCraft (jogg-0.0.7.jar)
 *     <li>Jorbis from JCraft (jorbis-0.0.15.jar)
 *     <li>OGGSPI (vorbisspi1.0.3.jar)
 * </ol>
 * 
 * @author Jam Jenkins
 */
public class Sound extends SequentialSound
{
    /**all sounds are kept here for muting and pausing.
     * This uses weak references because sound should be
     * able to be garbage collected once they are no longer
     * being used.*/
    private static ArrayList < WeakReference < Sound >> allSounds =
        new ArrayList < WeakReference < Sound >> ();
    /**the default start is muted*/
    private static boolean muted = true;
    /**the default start is paused*/
    private static boolean paused = true;

    /**storage for each of the concurrently playing clips*/
    private ArrayList<SequentialSound> clips;
    /**the name of the sound file*/
    private URL soundFile;


    /**makes the sound from a given URL*/
    public Sound(URL soundFile)
    {
        initialize(soundFile);
    }

    /**makes the sound from a give file name*/
    public Sound(String url)
    {
        initialize(GameLoop.getGameResource(url));
    }


    /**starts loading the sound into memory*/
    private void initialize(URL soundFile)
    {
        allSounds.add(new WeakReference<Sound>(this));
        this.soundFile = soundFile;
        clips = new ArrayList<SequentialSound>();
        String name = soundFile.toString().toLowerCase();
        if (name.endsWith("mp3") ||
                name.endsWith(".wav") ||
                name.endsWith(".ogg"))
        {
            clips.add(new SampledSound(soundFile));
        }
        else if (name.endsWith(".mid") ||
                 name.endsWith("midi"))
        {
            clips.add(new MidiSound(soundFile));
        }
        else
        {
            System.err.println("Unsupported sound format:" +
                               name);
        }
        mute();
        pause();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#getClipLength()
     */
    public double getClipLength()
    {
        return clips.get(0).getClipLength();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#getClipPosition()
     */
    public double getClipPosition()
    {
        double clipPosition = 0;
        for (SequentialSound s: clips)
        {
            if (s.isPlaying())
            {
                clipPosition = s.getClipPosition();
                break;
            }
        }
        return clipPosition;
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#getDuplicate()
     */
    public SequentialSound getDuplicate()
    {
        return new Sound(soundFile);
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#getLoopsLeft()
     */
    public int getLoopsLeft()
    {
        return clips.get(0).getLoopsLeft();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#getPan()
     */
    public double getPan()
    {
        return clips.get(0).getPan();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#getVolume()
     */
    public double getVolume()
    {
        return clips.get(0).getVolume();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#isFinishedPlaying()
     */
    public boolean isFinishedPlaying()
    {
        boolean finished = true;
        for (SequentialSound s: clips)
            finished = finished && s.isFinishedPlaying();
        return finished;
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#isLoaded()
     */
    public boolean isLoaded()
    {
        return clips.get(0).isLoaded();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#isLooping()
     */
    public boolean isLooping()
    {
        return clips.get(0).isLooping();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#isMuted()
     */
    public boolean isMuted()
    {
        return clips.get(0).isMuted();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#isPaused()
     */
    public boolean isPaused()
    {
        return clips.get(0).isPaused();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#isPlaying()
     */
    public boolean isPlaying()
    {
        boolean playing = false;
        for (SequentialSound s: clips)
            playing = playing || s.isPlaying();
        return playing;
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#loop()
     */
    public void loop()
    {
        for (SequentialSound s: clips)
            s.loop();
    }

    /**
     * mutes all of the current sounds
     */
    public static void muteAll()
    {
        muted = true;
        ArrayList < WeakReference < Sound >> toRemove =
            new ArrayList < WeakReference < Sound >> ();
        for (WeakReference<Sound> weak: allSounds)
        {
            Sound sound = weak.get();
            if (sound == null)
                toRemove.add(weak);
            else
                sound.mute();
        }
        allSounds.removeAll(toRemove);
    }

    public static void turnAllSoundOn()
    {
        muted = false;
        ArrayList < WeakReference < Sound >> toRemove =
            new ArrayList < WeakReference < Sound >> ();
        for (WeakReference<Sound> weak: allSounds)
        {
            Sound sound = weak.get();
            if (sound == null)
                toRemove.add(weak);
            else
                sound.turnSoundOn();
        }
        allSounds.removeAll(toRemove);
    }

    public static void pauseAll()
    {
        paused = true;
        ArrayList < WeakReference < Sound >> toRemove =
            new ArrayList < WeakReference < Sound >> ();
        for (WeakReference<Sound> weak: allSounds)
        {
            Sound sound = weak.get();
            if (sound == null)
                toRemove.add(weak);
            else
                sound.pause();
        }
        allSounds.removeAll(toRemove);
    }

    public static void resumeAll()
    {
        paused = false;
        ArrayList < WeakReference < Sound >> toRemove =
            new ArrayList < WeakReference < Sound >> ();
        for (WeakReference<Sound> weak: allSounds)
        {
            Sound sound = weak.get();
            if (sound == null)
                toRemove.add(weak);
            else
                sound.resume();
        }
        allSounds.removeAll(toRemove);
    }

    /**
     * mutes all currently playing sounds.  This has
     * no effect on sounds that are to be played in
     * the future.
     */
    public void mute()
    {
        for (SequentialSound s: clips)
            s.mute();
    }
    /**
     * pauses all currently playing sounds.  This has
     * no effect on sounds that are to be played in
     * the future.
     */
    public void pause()
    {
        for (SequentialSound s: clips)
            s.pause();
    }

    /**
     * gets a SequentialSound that is ready
     * to play.  A SequentialSound is ready
     * when it has either played to completion
     * or when it has yet to play the first time.
     * @return a SequentialSound ready to play
     */
    private SequentialSound getNextAvailable()
    {
        SequentialSound available = null;
        for (SequentialSound s: clips)
            if (s.isLoaded() && s.isFinishedPlaying())
            {
                available = s;
                break;
            }
        if (available == null)
        {
            // bcl -- never used -- long start = System.currentTimeMillis();
            available = clips.get(0).getDuplicate();
            // System.out.println("getting duplicate: " + (start - System.currentTimeMillis()));
            clips.add(available);
        }
        return available;
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#play()
     */
    public void play()
    {
        SequentialSound sound = getNextAvailable();
        sound.play();
        if (muted)
            sound.mute();
        if (paused)
            sound.pause();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#play(double)
     */
    public void play(double pan)
    {
        SequentialSound sound = getNextAvailable();
        sound.play(pan);
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#resume()
     */
    public void resume()
    {
        for (SequentialSound s: clips)
            s.resume();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#setLooping(boolean)
     */
    public void setLooping(boolean doLoop)
    {
        for (SequentialSound s: clips)
            s.setLooping(doLoop);
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#setLooping(int)
     */
    public void setLooping(int loops)
    {
        for (SequentialSound s: clips)
            s.setLooping(loops);
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#setPan(double)
     */
    public void setPan(double pan)
    {
        for (SequentialSound s: clips)
            s.setPan(pan);
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#setVolume(double)
     */
    public void setVolume(double volume)
    {
        for (SequentialSound s: clips)
            s.setVolume(volume);
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#start()
     */
    public void start()
    {
        for (SequentialSound s: clips)
            s.start();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#stop()
     */
    public void stop()
    {
        for (SequentialSound s: clips)
            s.stop();
    }

    /* (non-Javadoc)
     * @see fang2.SequentialSound#turnSoundOn()
     */
    public void turnSoundOn()
    {
        for (SequentialSound s: clips)
            s.turnSoundOn();
    }

    public static void main(String[] args) throws InterruptedException
    {
        URL url = SampledSound.class.getResource("VideoGameTrack.mp3");
        //url=SampledSound.class.getResource("YMCA.mid");
        RawMediaCache.loadAndWait(url);
        //url=SampledSound.class.getResource("VideoGameTrackCopy1.mp3");
        //RawMediaCache.loadAndWait(url);
        //url=SampledSound.class.getResource("VideoGameTrackCopy2.mp3");
        //RawMediaCache.loadAndWait(url);
        url = SampledSound.class.getResource("DingLower.wav");
        //RawMediaCache.loadAndWait(url);
        //URL url=SampledSound.class.getResource("YMCA.mid");
        Sound sound = new Sound(url);
        sound.setVolume(.2);
        sound.setPan(1);
        //sound.play();
        Sound sound2 = new Sound(url);
        sound2.setVolume(.2);
        sound2.setPan(0);
        //synchronized(sound){sound.wait(1000);}
        synchronized (url)
        {
            url.wait(1000);
        }
        // bcl -- never read -- long start1 = System.currentTimeMillis();
        sound2.play();
        // bcl -- never read --long start2 = System.currentTimeMillis();
        //sound.play();
        //System.out.println("start difference=" + (start2 - start1));
        //sound.setVolume(8);
        //url=SampledSound.class.getResource("YMCA.mid");
        //sound=new Sound(url);
        //sound.play();
        /*synchronized(sound){sound.wait(100);}
        sound.play();
        synchronized(sound){sound.wait(100);}
        sound.play();
        synchronized(sound){sound.wait(100);}
        sound.play();*/
    }


    @Override
    public void setClipPosition(double time)
    {
        for (SequentialSound s: clips)
            s.setClipPosition(time);
    }
}