package fang2.core;

import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.io.Externalizable;
import java.util.Observer;

import fang2.attributes.Location2D;


/**
 * Stores the mouse positions and clicks for use in the gaming engine.
 * 
 * @author Jam Jenkins
 */
public class Mouse implements Externalizable
{
    /** used for serialization versioning */
    private static final long serialVersionUID = 1L;

    /** position of the mouse */
    private Location2D mousePosition;

    /** cached position of the mouse */
    private Location2D lastMousePosition;

    /** canvas for getting scaled Location2Ds */
    private transient AnimationCanvas canvas;

    /** position of mouse click */
    private Location2D mouseClick;

    /** position of left mouse click */
    private Location2D leftClick;

    /** position of middle mouse click */
    private Location2D middleClick;

    /** position of right mouse click */
    private Location2D rightClick;

    /** whether mouse is down or not */
    private boolean mouseDown;

    /** whether left mouse is down or not */
    private transient boolean leftDown;

    /** whether middle mouse is down or not */
    private transient boolean middleDown;

    /** whether right mouse is down or not */
    private transient boolean rightDown;

    /** observer to update on mouse changes */
    private Observer observer;

    /** internal listener, so as not confuse those methods with game methods */
    protected Listener listener = new Mouse.Listener();


    /**
     * Creates a new instance of Mouse
     */
    public Mouse ()
    {
        mouseDown = false;
        leftDown = false;
        middleDown = false;
        rightDown = false;
        lastMousePosition = new Location2D();
    }


    /**
     * returns a string representation of the mouse position
     */
    public String toString ()
    {
        return "Mouse at " + mousePosition;
    }


    /**
     * sets the canvas.
     * 
     * The size of the canvas is used to scale the mouse position by the 
     * inverse of the canvas size in order to make the positions from 
     * (0, 0) to (1, 1). If the canvas is rectangular, it scales from 0 to
     * 1 on the short side and 0 to n on the long side.
     * 
     * @param canvas the canvas used for scaling the positions of the mouse
     */
    public void setCanvas (AnimationCanvas canvas)
    {
        this.canvas = canvas;
    }

    public void addListeners()
    {
        this.canvas.addMouseListener(listener);
        this.canvas.addMouseMotionListener(listener);
    }

    /**
     * writes the position and clicking of the mouse to the output stream.
     * 
     * @param out the output stream to write to
     */
    public void writeExternal (java.io.ObjectOutput out)
    throws java.io.IOException
    {
        out.writeBoolean(mouseDown);
        writePoint(out, mousePosition);
        writePoint(out, mouseClick);
        writePoint(out, leftClick);
        writePoint(out, middleClick);
        writePoint(out, rightClick);
    }


    /**
     * writes a point to the output stream
     * 
     * @param out the output stream to write to
     * @param point the point to write
     * @throws java.io.IOException if an error occurs with the networked
     *         connection
     */
    private void writePoint (java.io.ObjectOutput out, Location2D point)
    throws java.io.IOException
    {
        if (point == null)
        {
            out.writeBoolean(false);
        }
        else
        {
            out.writeBoolean(true);
            out.writeDouble(point.getX());
            out.writeDouble(point.getY());
        }
    }


    /**
     * reads in the point from the input stream
     * 
     * @param in the input stream to read from
     * @param point the place to store the read point
     * @return the point read in
     * @throws java.io.IOException if there is an error in the game's network
     *         connection
     */
    private Location2D readPoint (java.io.ObjectInput in, Location2D point)
    throws java.io.IOException
    {
        if (in.readBoolean())
        {
            if (point != null)
            {
                point.setLocation(in.readDouble(), in.readDouble());
            }
            else
            {
                point = new Location2D(in.readDouble(), in.readDouble());
            }
        }
        return point;
    }


    /**
     * reads in the mouse from the input stream.
     * 
     * @param in the input stream to read from
     */
    public void readExternal (java.io.ObjectInput in)
    throws java.io.IOException
    {
        boolean originalDown = mouseDown;
        mouseDown = in.readBoolean();
        mousePosition = readPoint(in, mousePosition);
        if (mousePosition != null)
        {
            lastMousePosition.setLocation(mousePosition.getX(),
                                          mousePosition.getY());
        }
        mouseClick = readPoint(in, mouseClick);
        leftClick = readPoint(in, leftClick);
        middleClick = readPoint(in, middleClick);
        rightClick = readPoint(in, rightClick);
        if (!originalDown && mouseDown)
        {
            if (leftClick != null)
                leftDown = true;
            if (middleClick != null)
                middleDown = true;
            if (rightClick != null)
                rightDown = true;
        }
        else if (originalDown && !mouseDown)
        {
            leftDown = false;
            middleDown = false;
            rightDown = false;
        }
    }


    /**
     * clear all pending mouse events
     */
    public void clear ()
    {
        mousePosition = null;
        mouseDown = false;
        leftDown = false;
        middleDown = false;
        rightDown = false;
        clearClicks();
    }


    /**
     * clears all mouse clicks
     */
    public void clearClicks ()
    {
        mouseClick = null;
        leftClick = null;
        rightClick = null;
        middleClick = null;
        leftDown = false;
        middleDown = false;
        rightDown = false;
    }


    /**
     * gets the last position of the mouse.
     * 
     * Subsequent calls to getMousePosition will return the same 
     * position until the mouse moves to a different location.
     * 
     * @return the last position of the mouse
     */
    public Location2D getLocation ()
    {
        return new Location2D(lastMousePosition);
    }


    /**
     * determines the last position of a click.
     * 
     * @return last clicked position,
     *         null if not clicked since last frame advance
     */
    public Location2D getClickLocation ()
    {
        if (mouseClick == null)
            return null;
        else
            return new Location2D(mouseClick);
    }


    /**
     * determines the last position of a left click
     * 
     * @return last clicked position,
     *         null if not clicked since last frame advance
     */
    public Location2D getLeftClickLocation ()
    {
        if (leftClick == null)
            return null;
        else
            return new Location2D(leftClick);
    }


    /**
     * determines the last position of a middle click.
     * 
     * @return last clicked position,
     *         null if not clicked since last frame advance
     */
    public Location2D getMiddleClickLocation ()
    {
        if (middleClick == null)
            return null;
        else
            return new Location2D(middleClick);
    }


    /**
     * determines the last position of a right click.
     * 
     * @return last clicked position,
     *         null if not clicked since last frame advance
     */
    public Location2D getRightClickLocation ()
    {
        if (rightClick == null)
            return null;
        else
            return new Location2D(rightClick);
    }


    /**
     * determines if the mouse is currently pressed.
     * 
     * Clicking occurs when the mouse is released.
     * 
     * @return true if the mouse is currently down, false otherwise
     */
    public boolean buttonPressed ()
    {
        return mouseDown;
    }

    /**
     * determines if the mouse is currently pressed.
     * 
     * Clicking occurs when the mouse is released.
     * 
     * @return true if the mouse is currently down, false otherwise
     */
    public boolean leftPressed ()
    {
        return leftDown;
    }

    /**
     * determines if the mouse is currently pressed.
     * 
     * Clicking occurs when the mouse is released.
     * 
     * @return true if the mouse is currently down, false otherwise
     */
    public boolean middlePressed ()
    {
        return middleDown;
    }

    /**
     * determines if the mouse is currently pressed.
     * 
     * Clicking occurs when the mouse is released.
     * 
     * @return true if the mouse is currently down, false otherwise
     */
    public boolean rightPressed ()
    {
        return rightDown;
    }


    /**
     * sets the observer to update when the mouse changes
     */
    public void setObserver (Observer observer)
    {
        this.observer = observer;
    }


    /**
     * notify observers of change in mouse
     */
    public void notifyObserver ()
    {
        if (observer != null)
        {
            observer.update(null, null);
        }
    }


    /**
     * converts the point on the screen to the scaled point using the canvas' size.
     * 
     * @param point the screen position
     * @return the relative position on the canvas
     */
    private Location2D getPoint2D (Point point)
    {
        if (canvas == null)
        {
            return new Location2D();
        }
        else
        {
            double min = Math.min(canvas.getWidth(), canvas.getHeight());
            Location2D clipped = new Location2D(point.x / min, point.y / min);
            if (canvas.getAspect() > 1)
            {
                clipped.setLocation(Math.min(canvas.getAspect(), clipped.getX()),
                                    Math.min(1, clipped.getY()));
            }
            else
            {
                clipped.setLocation(Math.min(1, clipped.getX()),
                                    Math.min(1 / canvas.getAspect(), clipped.getY()));
            }
            clipped.setLocation(Math.max(0, clipped.getX()),
                                Math.max(0, clipped.getY()));
            return clipped;
        }
    }


    /**
     * Internal class that listens for mouse events
     *
     * @author Robert C. Duvall
     */
    protected class Listener implements MouseListener, MouseMotionListener
    {
        /**
         * stores the MouseEvent for later polling.
         * 
         * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
         */
        public void mouseClicked (java.awt.event.MouseEvent mouseEvent)
        {}


        /**
         * stores the MouseEvent for later polling.
         * 
         * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
         */
        public void mouseDragged (java.awt.event.MouseEvent mouseEvent)
        {
            Location2D point = getPoint2D(mouseEvent.getPoint());
            if (!point.equals(mousePosition))
            {
                mousePosition = point;
                notifyObserver();
            }
        }


        /**
         * stores the MouseEvent for later polling.
         * 
         * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
         */
        public void mouseEntered (java.awt.event.MouseEvent mouseEvent)
        {}


        /**
         * stores the MouseEvent for later polling.
         * 
         * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
         */
        public void mouseExited (java.awt.event.MouseEvent mouseEvent)
        {}


        /**
         * stores the MouseEvent for later polling.
         * 
         * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
         */
        public void mouseMoved (java.awt.event.MouseEvent mouseEvent)
        {
            Location2D point = getPoint2D(mouseEvent.getPoint());
            if (!point.equals(mousePosition))
            {
                mousePosition = point;
                notifyObserver();
            }
        }


        /**
         * stores the MouseEvent for later polling.
         * 
         * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
         */
        public void mousePressed (java.awt.event.MouseEvent mouseEvent)
        {
            mouseDown = true;
            mouseClick = getPoint2D(mouseEvent.getPoint());
            if (mouseEvent.getButton() == MouseEvent.BUTTON1)
            {
                leftClick = mouseClick;
            }
            else if (mouseEvent.getButton() == MouseEvent.BUTTON2)
            {
                middleClick = mouseClick;
            }
            else if (mouseEvent.getButton() == MouseEvent.BUTTON3)
            {
                rightClick = mouseClick;
            }
            notifyObserver();
        }


        /**
         * stores the MouseEvent for later polling.
         * 
         * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
         */
        public void mouseReleased (java.awt.event.MouseEvent mouseEvent)
        {
            mouseDown = false;
            notifyObserver();
        }
    }
}