package fang2.core;

import java.awt.event.KeyEvent;
import java.io.Externalizable;
import java.util.Observer;
import java.awt.KeyEventDispatcher;
import java.awt.KeyboardFocusManager;
import static java.awt.event.KeyEvent.*;
/**
 * This class uses polling rather than events for keyboard input.
 * 
 * @author Jam Jenkins and the FANG Team
 */
public class Keyboard implements Externalizable
{
    /** used for serialization versioning */
    private static final long serialVersionUID = 1L;

    /** key pressed */
    private char key;

    /** observer, used for signaling updates */
    private Observer observer;

    /** internal listener, so as not confuse those methods with game methods */
    protected Listener listener = new Keyboard.Listener();

    /**
     * Creates a new instance of Keyboard
     */
    public Keyboard()
    {
        clear();
    }

    /**
     * returns the key that is pressed in string form
     */
    public String toString()
    {
        return key + " is last pressed";
    }

    /**
     * Clear all stored keys. Sets the stored key to Keyboard.NONE
     */
    public void clear()
    {
        key = KeyEvent.CHAR_UNDEFINED; // bcl - modified to return Keyboard.NONE rather than a KeyEvent value
    }

    /**
     * writes the key to the output stream
     * 
     * @param out
     *          the output stream to write to
     */
    public void writeExternal(java.io.ObjectOutput out)
    throws java.io.IOException
    {
        out.writeChar(key);
    }

    /**
     * reads the key from the input stream
     * 
     * @param in
     *          the input stream to read from
     */
    public void readExternal(java.io.ObjectInput in) throws java.io.IOException
    {
        char tmp = in.readChar();
        if (tmp != KeyEvent.CHAR_UNDEFINED)
        {
            key = tmp;
        }
    }

    /**
     * Returns the last key pressed. After a key is pressed, the keyboard hanlders
     * store it. Thus subsequent calls to this method will return the same value
     * until the event is cleared (in which case the key becomes Keyboard.NONE) or
     * a different key is pressed, whichever happens first.
     * 
     * @return the last key pressed or Keyboard.NONE if no key was pressed since
     *         events were cleared
     */
    public char getLastKey()
    {
        return key;
    }

    /**
     * Synonym for {@link #getLastKey}.
     * 
     * @return the last key pressed or, if none has been pressed since events were
     *         cleared, returns Keyboard.NONE
     */
    public char getKeyPressed()
    {
        return getLastKey();
    }

    /**
     * Returns the last key pressed (if there is one) converted to lowercase.
     * Returns Keyboard.NONE if no key was pressed since events were cleared.
     * 
     * @return lowercase of last key pressed (non-letters unchanged) or
     *         Keyboard.NONE
     */
    public char getLowerCaseKeyPressed()
    {
        return Character.toLowerCase(getLastKey());
    }

    /**
     * Returns the last key pressed (if there is one) converted to UPPERCASE.
     * Returns Keyboard.NONE if no key was pressed since events were cleard.
     * 
     * @return uppercase of last key pressed (non-letters unchanged) or
     *         Keyboard.NONE
     */
    public char getUpperCaseKeyPressed()
    {
        return Character.toUpperCase(getLastKey());
    }

    /**
     * Was the up-arrow key pressed as the last key?
     * 
     * @return true if up-arrow key was last key pressed; false otherwise
     */
    public boolean upPressed()
    {
        return VK_UP == key || VK_KP_UP == key;
    }

    /**
     * Was the down-arrow key pressed as the last key?
     * 
     * @return true if down-arrow key was last key pressed; false otherwise
     */
    public boolean downPressed()
    {
        return VK_DOWN == key || key == VK_KP_DOWN;
    }

    /**
     * Was the left-arrow key pressed as the last key?
     * 
     * @return true if left-arrow key was last key pressed; false otherwise
     */
    public boolean leftPressed()
    {
        return VK_LEFT == key || key == VK_KP_LEFT;
    }

    /**
     * Was the right-arrow key pressed as the last key?
     * 
     * @return true if right-arrow key was last key pressed; false otherwise
     */
    public boolean rightPressed()
    {
        return VK_RIGHT == key || key == VK_KP_RIGHT;
    }

    /**
     * determines if getLastKey will return a key pressed.
     * 
     * getLastKey returns an undefined char when no key has been pressed.
     * 
     * @return true if a key has been pressed, false otherwise
     */
    public boolean keyPressed()
    {
        return key != KeyEvent.CHAR_UNDEFINED;
    }

    /**
     * sets the last key pressed
     * 
     * @param key
     *          the last key pressed
     */
    public void setLastKey(char key)
    {
        this.key = key;
        if (observer != null)
        {
            observer.update(null, null);
        }
    }

    /**
     * sets keyboard up for this canvas.
     * 
     * @param canvas
     */
    public void setCanvas(AnimationCanvas canvas)
    {
        KeyboardFocusManager.getCurrentKeyboardFocusManager()
        .addKeyEventDispatcher(listener);
    }

    /**
     * sets observer
     * 
     * @param observer
     */
    public void setObserver(Observer observer)
    {
        this.observer = observer;
    }

    /**
     * Internal class that listens for key events
     * 
     * @author Robert C. Duvall
     */
    protected class Listener implements KeyEventDispatcher
    {
        /**
         * sets key and updates the observer
         * 
         * @param e
         *          the event to store
         * @return false
         */
        public boolean dispatchKeyEvent(KeyEvent e)
        {
            if (e.getID() == KeyEvent.KEY_PRESSED)
            {
               char letter;
               if(e.getKeyChar()==KeyEvent.CHAR_UNDEFINED)
               {
                  letter=(char)e.getKeyCode();
               }
               else
                  letter=e.getKeyChar();
               setLastKey(letter);
            }
            return false;
        }
    }
}
