package fang2.core;

import java.awt.*;
import java.lang.reflect.*;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;

import fang2.attributes.*;
import fang2.media.*;
import fang2.sprites.*;
import fang2.ui.*;


/**
 * This class redirects all method calls to the originally extended
 * JApplet.
 *
 * @author  Jam Jenkins and the FANG Team
 */
@SuppressWarnings("serial")
public abstract class GameRedirection
  extends GameLoop {
  protected Game firstGame;

  /** used by the first game to call the correct advance method */
  protected Game currentGame;

  protected ArrayList<Game> upcomingGames;

  protected ArrayList<Game> addedGames;

  protected boolean finishingGame;

  protected boolean restarting;
  
  protected boolean activated=false;

  /**
   * the alarms which will persist across levels. When a level ends all
   * alarms in not in this collection will be cancelled.
   */
  private final HashSet<Alarm> persistentAlarms = new HashSet<Alarm>();

  /**
   * the sprites which will persist across levels. When a level ends all
   * sprites in not in this collection will be removed from the canvas.
   */
  private final HashSet<Sprite> persistentSprites =
    new HashSet<Sprite>();

  /**
   * the sounds which will persist across levels. When a level ends all
   * sounds not in this collection will be terminated.
   */
  private final HashSet<Sound> persistentSounds = new HashSet<Sound>();

  /**
   * Constructs a game level; permits creation of a container of game
   * levels and linking of the finish of one game to the beginning of
   * another game/level.
   *
   * @param  gameDimensions   the dimensions, in pixels, of the game
   *                          canvas part of the game window. The play
   *                          area size.
   * @param  backgroundColor  color to wipe the backtground to
   */
  public GameRedirection(Dimension gameDimensions,
    Color backgroundColor) {
    super(gameDimensions, backgroundColor);
    initGameRedirection();
  }

  /**
   * Handles GameRedirection initialization for any constructor. Factors
   * out the common constructor code to avoid duplication (and missing
   * updates).
   */
  private void initGameRedirection() {
    firstGame = (Game) this;
    if (currentGame == null) {
      // System.out.println("current game is null");
      currentGame = (Game) this;
    }
    upcomingGames = new ArrayList<Game>();
    addedGames = new ArrayList<Game>();
    finishingGame = false;
    restarting = false;
    resetTime();
  }

  protected void copyFirstFields() {
    // public fields
    random = firstGame.random;
    player = firstGame.player;
    timeInterval = firstGame.timeInterval;
    players = firstGame.players;
    player = firstGame.player;
    playersSelectable = firstGame.playersSelectable;
    serverSelectable = firstGame.serverSelectable;
    sessionSelectable = firstGame.sessionSelectable;
    fullScreen = firstGame.fullScreen;

    // protected fields
    client = firstGame.client;
    client.setGameLoop(currentGame);

    canvas = firstGame.canvas;
    help = firstGame.help;
    setCreateOwnFrame(firstGame.isCreatingOwnFrame());

    setNumberOfPlayers(firstGame.getNumberOfPlayers());
    setServerName(firstGame.getServerName());
  }

  @Override
  protected Game getSharedCurrentGame() {
    return currentGame;
  }

  /**
   * Get a reference to the current game without having to know anything
   * about the type of the game. Note: this method must only be called
   * from the AWT Event Thread.
   *
   * @return  a reference to the current game, or null if not called
   *          from the AWT Event Thread
   */
  public static Game getCurrentGame() {
    //error checking would be good here, but it does not work
    //when the game starts up because the thread that creates
    //the applet may not be the dispatch thread??
    //if (java.awt.EventQueue.isDispatchThread()) {
      return sharedCurrentGame;
    //} else {
      // return currentGame;
    //  return null;
    //}
  }

  /**
   * Add a game to the list of games waiting to be played.
   *
   * @param  game  the Game to add to the end of the list of upcoming
   *               games
   */
  public void addGame(Game game) {
    addedGames.add(game);
  }

  /**
   * Finish the game. advance and advanceFrame will not be called again
   * for this GameRedirection. Note that any games added (through calls
   * to addGame) are, at this point, added to the upcomingGames and
   * removed from addedGames.
   */
  public void finishGame() {
    currentGame.finishingGame = true;
    currentGame.activated=false;
  }

  /**
   * Finishes a level. The current game is cleaned up and the next game
   * (if any are waiting) in upcomingGames is started.
   */
  protected void reallyFinishGame() {
	  addedGames.addAll(upcomingGames);
	  upcomingGames.clear();
	  upcomingGames.addAll(addedGames);
	  addedGames.clear();
	  currentGame.finishingGame = false;
	  cleanUp();
	  removeLevelObjects();
	  if (upcomingGames.size() != 0) {
		  Game upcoming = upcomingGames.get(0);
		  upcomingGames.remove(0);
		  upcoming.firstGame = firstGame;
		  upcoming.copyFirstFields();
		  upcoming.upcomingGames.clear();
		  upcoming.upcomingGames.addAll(currentGame.upcomingGames);
		  currentGame = upcoming;
		  firstGame.currentGame = upcoming;
		  firstGame.initializeCurrentGame();
		  ((GameRedirection)upcoming).setupWhenMediaLoaded();
    }
  }

  /**
   * Add a level. This method should not be used; use {@link #addGame}
   * instead.
   *
   * @deprecated
   */
  @Deprecated
  @Override
  public final void addLevel(GameLevel level) {
  }

  /**
   * Finish a level. This method should not be used; use {@link
   * #finishGame} instead.
   *
   * @deprecated
   */
  @Deprecated
  @Override
  public final void finishLevel() {
  }

  /**
   * Get the next game level. This method should not be used; use
   *
   * @deprecated
   */
  @Deprecated
  @Override
  public final GameLevel getNextLevel() {
    return null;
  }

  /**
   * @deprecated
   */
  @Deprecated
  @Override
  public final void advanceLevel() {
  }

  /**
   * @deprecated
   */
  @Deprecated
  @Override
  public final int getLevelNumber() {
    return -1;
  }

  /**
   * @deprecated
   */
  @Deprecated
  @Override
  public final void setNextLevel(GameLevel level) {
  }

  /**
   * @deprecated
   */
  @Deprecated
  @Override
  public final void advanceFrame(double secondsElapsed) {
    currentGame.advance(secondsElapsed);
    if (currentGame.restarting) {
      currentGame.reallyStartOver();
    }
    if (currentGame.finishingGame) {
      currentGame.reallyFinishGame();
    }
  }

  /**
   * Advance with time elapsed since previous call to advance. By
   * default, just calls the zero parameter version of advance.
   *
   * @param  deltaTime  number of seconds since last call to advance
   */
  public void advance(double deltaTime) {
    advance();
  }

  /**
   * An abstract advance method. Must be implemented by classes
   * implementing GameRedirection.
   */
  public abstract void advance();

  /**
   * removes all the non-persistent sprites, alarms, and sounds
   */
  protected void removeLevelObjects() {
    Sprite[] allSprites = canvas.getAllSprites();
    for (Sprite sprite : allSprites) {
      if (!persistentSprites.contains(sprite)) {
        canvas.removeSprite(sprite);
      }
    }
    // Sound.clearAllExcept(persistentSounds.toArray(new Sound[0]));
    Alarm[] allAlarms = getAlarms();
    for (Alarm alarm : allAlarms) {
      if (!persistentAlarms.contains(alarm)) {
        cancelAlarm(alarm);
      }
    }
  }

  /**
   * starts the game over. This consists of removing all the sprites,
   * sounds, and alarms, then calling the startGame method. All of this
   * will occur after the next termination of the advanceFrame method.
   * Therefore, the game should be able to continue without error for at
   * least one more frame before the startGame method is called.
   */
  @Override
  public void startOver() {
    currentGame.restarting = true;
  }

  protected void reallyStartOver() {
    currentGame.restarting = false;
    restoreCursor();
    resetTime();
    canvas.removeAllSprites();
    // Sound.clearAll();
    cancelAllAlarms();
    client.clearInput();
    cleanUp();
    upcomingGames.clear();
    addedGames.clear();
    currentGame = firstGame;
    firstGame.startGame();
  }

  /**
   * this method will be called by the FANG Engine just before
   * terminating the level in case the level needs an opportunity to
   * perform some final operations
   */
  public void cleanUp() {
  }

  /**
   * adds an alarm to the collection of alarms which will persist after
   * the level ends.
   *
   * @param  alarm  the alarm to keep after the level terminates
   */
  public void persist(Alarm alarm) {
    persistentAlarms.add(alarm);
  }

  /**
   * adds a sprite to the collection of sprites which will persist after
   * the level ends.
   *
   * @param  sprite  the sprite to keep after the level terminates
   */
  public void persist(Sprite sprite) {
    persistentSprites.add(sprite);
  }

  /**
   * adds a sound to the collection of sounds which will persist after
   * the level ends.
   *
   * @param  sound  the sound to keep after the level terminates
   */
  public void persist(Sound sound) {
    persistentSounds.add(sound);
  }

  /**
   * copies all of the peristent alarms, sprites, and sounds into this
   * level's persistence collections
   *
   * @param  alarms   the alarms which came from the last level
   * @param  sprites  the sprites which came from the last level
   * @param  sounds   the sounds which came from the last lelel
   */
  public void initializePersistantState(Alarm[] alarms,
    Sprite[] sprites, Sound[] sounds) {
    for (Alarm alarm : alarms) {
      persistentAlarms.add(alarm);
    }
    for (Sprite sprite : sprites) {
      persistentSprites.add(sprite);
    }
    for (Sound sound : sounds) {
      persistentSounds.add(sound);
    }
  }

  public abstract void setup();

  private void setupWhenMediaLoaded()
	{
		if (RawMediaCache.getQueueLength() == 0 || getNumberOfPlayers()>1)
		{
			setup();
			activated=true;
		}
		else
		{
			final int resources = RawMediaCache.getQueueLength();
			final StringSprite status = new StringSprite("Loading 0/"
					+ resources + "\n" + "Please wait...");
			status.setLineHeight(0.1);
			status.setLocation(0.25, 0.5);
			status.leftJustify();
			status.topJustify();
			currentGame.addSprite(status);
			final LineSprite line = new LineSprite(0.25, 0.4, 0.25, 0.4);
			line.setColor(Palette.getColor("white"));
			final LineSprite underline = new LineSprite(0.25, 0.4, 0.75, 0.4);
			underline.setColor(Palette.getColor("white", 120));
			// underline.setLineThickness(0.3);
			// currentGame.addSprite(line);
			currentGame.addSprite(underline);
			currentGame.addSprite(line);
			currentGame.pause();
			new Thread(new Runnable()
			{
				public void run()
				{
					Runnable runnable = new Runnable()
					{
						public void run()
						{
							int loaded = resources
									- RawMediaCache.getQueueLength();
							status.setText("Loading: " + loaded + "/"
									+ resources + "\n" + "Please wait...");
							line.setEnd(0.25 + loaded * 0.5 / resources, 0.4);
							currentGame.getCanvas().paintImmediately();
						}
					};
					while (RawMediaCache.getQueueLength() > 0)
					{
						try
						{
							EventQueue.invokeAndWait(runnable);
						} catch (InterruptedException e1)
						{
							// TODO Auto-generated catch block
							e1.printStackTrace();
						} catch (InvocationTargetException e1)
						{
							// TODO Auto-generated catch block
							e1.printStackTrace();
						}
						synchronized (this)
						{
							try
							{
								this.wait(100);
							} catch (InterruptedException e)
							{
								// TODO Auto-generated catch block
								e.printStackTrace();
							}
						}
					}
					currentGame.removeSprite(status);
					currentGame.removeSprite(line);
					currentGame.removeSprite(underline);
					setup();
					activated=true;
					resume();
				}
			}).start();
		}
	}
  
  @Override
  public final void startGame() {
    currentGame = (Game) this;
    ErrorConsole.clear();
    setupWhenMediaLoaded();
  }

  protected void copyFrom(Game game) {
  }

  // --------Start GameLoop Methods--------
  @Override
  public final void toggleSound() {
    if (firstGame == this) {
      super.toggleSound();
    } else {
      firstGame.toggleSound();
    }
  }

  /**
   * toggle mute/audible
   *
   * @deprecated  use of this method is depricated; use toggleSound
   *              instead
   */
  @Deprecated
  @Override
  protected final void toggleAudible() {
    toggleSound();
  }

  @Override
  public boolean isMuted() {
    if (firstGame == this) {
      return super.isMuted();
    } else {
      return firstGame.isMuted();
    }
  }

  @Override
  public void muteSound() {
    if (firstGame == this) {
      super.muteSound();
    } else {
      firstGame.muteSound();
    }
  }

  @Override
  public void playSound() {
    if (firstGame == this) {
      super.playSound();
    } else {
      firstGame.playSound();
    }
  }

  @Override
  public void pause() {
    if (firstGame == this) {
      super.pause();
    } else {
      firstGame.pause();
    }
  }

  @Override
  public void resume() {
    if (firstGame == this) {
      super.resume();
    } else {
      firstGame.resume();
    }
  }

  @Override
  public void startGameImmediately() {
    if (firstGame == this) {
      super.startGameImmediately();
    } else {
      firstGame.startGameImmediately();
    }
  }

  @Override
  public void playSoundImmediately() {
    if (firstGame == this) {
      super.playSoundImmediately();
    } else {
      firstGame.playSoundImmediately();
    }
  }

  @Override
  public void begin() {
    if (firstGame == this) {
      super.begin();
    } else {
      firstGame.begin();
    }
  }

  @Override
  public final void setServerName(String server) {
    if (firstGame == this) {
      super.setServerName(server);
    } else {
      firstGame.setServerName(server);
    }
  }

  @Override
  public final void setSessionName(String sessionName) {
    if (firstGame == this) {
      super.setSessionName(sessionName);
    } else {
      firstGame.setSessionName(sessionName);
    }
  }

  @Override
  public String getSessionName() {
    if (firstGame == this) {
      return super.getSessionName();
    } else {
      return firstGame.getSessionName();
    }
  }

  @Override
  public String getServerName() {
    if (firstGame == this) {
      return super.getServerName();
    } else {
      return firstGame.getServerName();
    }
  }

  @Override
  public final void setGameName(String gameName) {
    if (firstGame == this) {
      super.setGameName(gameName);
    } else {
      firstGame.setGameName(gameName);
    }
  }

  @Override
  public String getGameName() {
    if (firstGame == this) {
      return super.getGameName();
    } else {
      return firstGame.getGameName();
    }
  }

  @Override
  public final void setNumberOfPlayers(int numPlayers) {
    if (firstGame == this) {
      super.setNumberOfPlayers(numPlayers);
    } else {
      firstGame.setNumberOfPlayers(numPlayers);
    }
  }

  @Override
  public int getNumberOfPlayers() {
    if (firstGame == this) {
      return super.getNumberOfPlayers();
    } else {
      return firstGame.getNumberOfPlayers();
    }
  }

  @Override
  public void sendMessage(Object localMessage) {
    if (firstGame == this) {
      super.sendMessage(localMessage);
    } else {
      firstGame.sendMessage(localMessage);
    }
  }

  @Override
  public final void connect(String server, String gameName,
    String sessionName, int players) {
    if (firstGame == this) {
      super.connect(server, gameName, sessionName, players);
    } else {
      firstGame.connect(server, gameName, sessionName, players);
    }
  }

  @Override
  public final void stop() {
    if (firstGame == this) {
      super.stop();
    } else {
      firstGame.stop();
    }
  }

  @Override
  public final void disconnect() {
    if (firstGame == this) {
      super.disconnect();
    } else {
      firstGame.disconnect();
    }
  }

  @Override
  public boolean isPaused() {
    if (firstGame == this) {
      return super.isPaused();
    } else {
      return firstGame.isPaused();
    }
  }

  @Override
  public final void togglePause() {
    if (firstGame == this) {
      super.togglePause();
    } else {
      firstGame.togglePause();
    }
  }

  @Override
  protected final void pauseToggle() {
    if (firstGame == this) {
      super.pauseToggle();
    } else {
      firstGame.pauseToggle();
    }
  }

  @Override
  public final void serverSaysPauseToggle() {
    if (firstGame == this) {
      super.serverSaysPauseToggle();
    } else {
      firstGame.serverSaysPauseToggle();
    }
  }

  @Override
  public int getID() {
    if (firstGame == this) {
      return super.getID();
    } else {
      return firstGame.getID();
    }
  }

  @Override
  public Player getPlayer(int playerIndex) {
    if (firstGame == this) {
      return super.getPlayer(playerIndex);
    } else {
      return firstGame.getPlayer(playerIndex);
    }
  }

  @Override
  public Player getPlayer(String name) {
    if (firstGame == this) {
      return super.getPlayer(name);
    } else {
      return firstGame.getPlayer(name);
    }
  }

  @Override
  public final boolean gameIsOver() {
    if (firstGame == this) {
      return super.gameIsOver();
    } else {
      return firstGame.gameIsOver();
    }
  }

  public boolean isGameOver() {
    if (firstGame == this) {
      return super.gameIsOver();
    } else {
      return firstGame.gameIsOver();
    }
  }

  // !!!!May need to add more here!!!!
  @Override
  public void setGameOver(boolean gameOver) {
    if (firstGame == this) {
      super.setGameOver(gameOver);
    } else {
      firstGame.setGameOver(gameOver);
    }
  }

  @Override
  public Player getPlayer() {
    if (firstGame == this) {
      return super.getPlayer();
    } else {
      return firstGame.getPlayer();
    }
  }

  // --------End GameLoop Methods--------

  // --------Start FrameAdvancer Methods--------

  @Override
  public void setCanvas(AnimationCanvas canvas) {
    if (firstGame == this) {
      super.setCanvas(canvas);
    } else {
      firstGame.setCanvas(canvas);
    }
  }

  @Override
  public AnimationCanvas getCanvas() {
    if (firstGame == this) {
      return super.getCanvas();
    } else {
      return firstGame.getCanvas();
    }
  }

  @Override
  public double getTime() {
    if (firstGame == this) {
      return super.getTime();
    } else {
      return firstGame.getTime();
    }
  }

  @Override
  public double getScreenRefreshRate() {
    if (firstGame == this) {
      return super.getScreenRefreshRate();
    } else {
      return firstGame.getScreenRefreshRate();
    }
  }

  @Override
  public void setMinimumModelFrameRate(int framesPerSecond) {
    if (firstGame == this) {
      super.setMinimumModelFrameRate(framesPerSecond);
    } else {
      firstGame.setMinimumModelFrameRate(framesPerSecond);
    }
  }

  @Override
  public final void scheduleRelative(Alarm alarm, double relative) {
    if (firstGame == this) {
      super.scheduleRelative(alarm, relative);
    } else {
      firstGame.scheduleRelative(alarm, relative);
    }
  }

  @Override
  public final void scheduleAbsolute(Alarm alarm, double absolute) {
    if (firstGame == this) {
      super.scheduleAbsolute(alarm, absolute);
    } else {
      firstGame.scheduleAbsolute(alarm, absolute);
    }
  }

  @Override
  public final void cancelAlarm(Alarm alarm) {
    if (firstGame == this) {
      super.cancelAlarm(alarm);
    } else {
      firstGame.cancelAlarm(alarm);
    }
  }

  @Override
  public final void cancelAllAlarms() {
    if (firstGame == this) {
      super.cancelAllAlarms();
    } else {
      firstGame.cancelAllAlarms();
    }
  }

  @Override
  public final Alarm[] getAlarms() {
    if (firstGame == this) {
      return super.getAlarms();
    } else {
      return firstGame.getAlarms();
    }
  }

  @Override
  public void schedule(AlarmAdapter action, double delay) {
    if (firstGame == this) {
      super.schedule(action, delay);
    } else {
      firstGame.schedule(action, delay);
    }
  }

  @Override
  public void cancel(AlarmAdapter action) {
    if (firstGame == this) {
      super.cancel(action);
    } else {
      firstGame.cancel(action);
    }
  }

  @Override
  public void cancelAllTimedActions() {
    if (firstGame == this) {
      super.cancelAllTimedActions();
    } else {
      firstGame.cancelAllTimedActions();
    }
  }

  @Override
  public AlarmAdapter[] getAllTimedActions() {
    if (firstGame == this) {
      return super.getAllTimedActions();
    } else {
      return firstGame.getAllTimedActions();
    }
  }

  @Override
  public void updateModel(double time) {
	  if (firstGame == this && (firstGame!=currentGame || activated)) {
	      super.updateModel(time);
	    } else if(activated){
	      firstGame.updateModel(time);
	    }
  }

  @Override
  public void setCursor(URL url) {
    if (firstGame == this) {
      super.setCursor(url);
    } else {
      firstGame.setCursor(url);
    }
  }

  @Override
  public void removeCursor() {
    if (firstGame == this) {
      super.removeCursor();
    } else {
      firstGame.removeCursor();
    }
  }

  @Override
  public void restoreCursor() {
    if (firstGame == this) {
      super.restoreCursor();
    } else {
      firstGame.restoreCursor();
    }
  }

  @Override
  public final void refreshScreen() {
    if (firstGame == this) {
      super.refreshScreen();
    } else {
      firstGame.refreshScreen();
    }
  }

  @Override
  public void resetTime() {
    if (firstGame == this) {
      super.resetTime();
    } else {
      firstGame.resetTime();
    }
  }

  // --------End FrameAdvancer Methods--------

  // --------Start GameWindow Methods--------
  @Override
  public void runAsApplication() {
    if (firstGame == this) {
      super.runAsApplication();
    } else {
      firstGame.runAsApplication();
    }
  }

  @Override
  public void setTitle(String topTitle) {
    if (firstGame == this) {
      super.setTitle(topTitle);
    } else {
      firstGame.setTitle(topTitle);
    }
  }

  @Override
  public void setHelp(String filename) {
    if (firstGame == this) {
      super.setHelp(filename);
    } else {
      firstGame.setHelp(filename);
    }
  }

  @Override
  public void setHelpText(String helpText) {
    if (firstGame == this) {
      super.setHelpText(helpText);
    } else {
      firstGame.setHelpText(helpText);
    }
  }
  // --------End GameWindow Methods--------
}
