package fang2.core;

import java.awt.Color;
import java.awt.Dimension;
import java.net.URL;
import java.util.Random;

import fang2.attributes.Location2D;
import fang2.attributes.Palette;
import fang2.core.Player;
import fang2.ui.*;

/**
 * The class representing a FANG video game. This is an "empty" game
 * box. By extending it, a program gets the pretty window, the buttons,
 * and the behavior of every FANG video game. Because the box is empty,
 * there is no actual game contained in this class. Programmers extend
 * Game to create their own games. The extension is usually by defining
 * new versions of {@link #setup()} and {@link #advance(double)} (or
 * {@link #advance()} if you don't care about elapsed time between
 * frames). Create and initialize game attributes in the
 * {@link #setup()} method and then update their state in the
 * {@link #advance(double)} method. In FANG, a game runs something like
 * this (the following is not actually Java): 
 * <pre>
 * create Window,
 * create Game g; 
 * g.setup(); 
 * while (!g.isGameOver()) { 
 *   draw all known Sprites 
 *   check for input from all input devices 
 *   calculate timeSinceLastCall to advance 
 *   g.advance(timeSinceLastCall);
 * }
 * </pre>  
 * 
 * @author Jam Jenkins and the FANG Team
 */
@SuppressWarnings("serial")
public class Game
  extends GameRedirection {
  /** the color to use when clearing the background */
  public static final Color DEFAULT_BACKGROUND_COLOR = Color.BLACK;

  /** default size of canvas */
  public static final Dimension DEFAULT_GAME_DIMENSIONS =
      new Dimension(400, 400);

  /**
   * Create a new Game with the default size and default background
   * color. See {@link #DEFAULT_GAME_DIMENSIONS} and
   * {@link #DEFAULT_BACKGROUND_COLOR} for definitions of the default
   * values. The game will keep the same aspect ratio (ratio between
   * width and height) even when resized after creation.
   */
  public Game() {
    this(DEFAULT_GAME_DIMENSIONS, DEFAULT_BACKGROUND_COLOR);
  }

  public Game(Color backgroundColor) {
    this(DEFAULT_GAME_DIMENSIONS, backgroundColor);
  }

  /**
   * Create a new Game with the default size and default background
   * color. See {@link #DEFAULT_GAME_DIMENSIONS} and
   * {@link #DEFAULT_BACKGROUND_COLOR} for definitions of the default
   * values. The game will keep the same aspect ratio (ratio between
   * width and height) even when resized after creation.
   * 
   * @param gameDimensions initial dimensions of the game field in
   *          pixels
   */
  public Game(Dimension gameDimensions) {
    this(gameDimensions, DEFAULT_BACKGROUND_COLOR);
  }

  /**
   * Create a new Game with the given size and default background color.
   * See {@link #DEFAULT_BACKGROUND_COLOR} for definitions of the
   * default values. The game will keep the same aspect ratio (ratio
   * between width and height) even when resized after creation.
   * 
   * @param gamePixelWidth initial width of the game field in pixels
   * @param gamePixelHeight initial height of the game field in pixels
   */
  public Game(int gamePixelWidth, int gamePixelHeight) {
    this(new Dimension(gamePixelWidth, gamePixelHeight),
        DEFAULT_BACKGROUND_COLOR);
  }

  /**
   * Create a new Game with the given size and given background color.
   * The game will keep the same aspect ratio (ratio between width and
   * height) even when resized after creation.
   * 
   * @param gamePixelWidth initial width of the game field in pixels
   * @param gamePixelHeight initial height of the game field in pixels
   * @param backgroundColor inital background color of the game field
   */
  public Game(int gamePixelWidth, int gamePixelHeight,
              Color backgroundColor) {
    this(new Dimension(gamePixelWidth, gamePixelHeight),
        backgroundColor);
  }

  /**
   * Create a new Game with the given size and given background color.
   * The game will keep the same aspect ratio (ratio between width and
   * height) even when resized after creation.
   * 
   * @param gameDimensions initial dimensions of the game field in
   *          pixels
   * @param backgroundColor inital background color of the game field
   */
  public Game(Dimension gameDimensions, Color backgroundColor) {
    super(gameDimensions, backgroundColor);
  }
  
  /**call this method only from the constructor*/
  public void initializePlayerNames()
  {
	  //don't re-initialize names
	  for(int i=0; i<getNumberOfPlayers(); i++)
	  {
		  if(!getPlayerName(i).equals(Player.getDefaultName(i)))
		  {
			  return;
		  }
	  }
	  addedGames.add(0, new PlayerNameInput());
	  addedGames.add(1, this);
	  finishGame();
  }


  // ----------Start Mouse methods---------

  /**
   * Get location for a given player. Returns the location (a point) in
   * screen coordinates (0.0-1.0 for both x- and y-coordinates).
   * 
   * @param id - the identifier of the player to get the mouse for
   * @return the location of the mouse of the given player
   */
  public Location2D getMouse2D(int id) {
    return getPlayer(id).getMouse().getLocation();
  }

  /**
   * Return the x-coordinate of the mouse for a given player id.
   * 
   * @param id - player id to get coordinate for
   * @return the screen x-coordinate of the given mouse during this
   *         frame (0.0 - 1.0 on screen)
   */
  public double getMouseX(int id) {
    return getMouse2D(id).x;
  }

  /**
   * Return the y-coordinate of the mouse for a given player id.
   * 
   * @param id - player id to get coordinate for
   * @return the screen y-coordinate of the given mouse during this
   *         frame (0.0 - 1.0 on screen)
   */
  public double getMouseY(int id) {
    return getMouse2D(id).y;
  }

  /**
   * Get location for the default (local) player's mouse. Returns the
   * location (a point) in screen coordinates (0.0-1.0 for both x- and
   * y-coordinates).
   * 
   * @return the location of the mouse of the given player
   */
  public Location2D getMouse2D() {
    return getMouse2D(0);
  }

  /**
   * Return the x-coordinate of the mouse for the default (local)
   * player.
   * 
   * @return the screen x-coordinate of the mouse during this frame (0.0
   *         - 1.0 on screen)
   */
  public double getMouseX() {
    return getMouseX(0); // bcl - modified to parallel getClickX...
    // makes no
    // sense to do this two different ways
  }

  /**
   * Return the y-coordinate of the mouse for the default (local)
   * player.
   * 
   * @return the screen y-coordinate of the mouse during this frame (0.0
   *         - 1.0 on screen)
   */
  public double getMouseY() {
    return getMouseY(0);
  }

  /**
   * Get location for the given player's mouse when any button was
   * clicked. Returns the location (a point) in screen coordinates
   * (0.0-1.0 for both x- and y-coordinates) where any button was
   * clicked.
   * 
   * @param id - player id to get coordinate for
   * @return the location of the mouse of the given player; null if the
   *         player did not click this frame
   */
  public Location2D getClick2D(int id) {
    return getPlayer(id).getMouse().getClickLocation();
  }

  /**
   * Return the x-coordinate of the given player's mouse when any button
   * was clicked. If no button was clicked, will return the special
   * numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @param id player to find mouse click location for
   * @return x-coordinate (in screens) where mouse was clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getClickX(int id) {
    if (getClick2D(id) == null) {
      return Double.NaN;
    } else {
      return getClick2D(id).x;
    }
  }

  /**
   * Return the y-coordinate of the given player's mouse when any button
   * was clicked. If no button was clicked, will return the special
   * numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @param id player to find mouse click location for
   * @return y-coordinate (in screens) where mouse was clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getClickY(int id) {
    if (getClick2D(id) == null) {
      return Double.NaN;
    } else {
      return getClick2D(id).y;
    }
  }

  /**
   * Get location for the default (local) player's mouse when any button
   * was clicked. Returns the location (a point) in screen coordinates
   * (0.0-1.0 for both x- and y-coordinates) where any button was
   * clicked or null if no click.
   * 
   * @return the location of the mouse of the given player; null if the
   *         player did not click this frame
   */
  public Location2D getClick2D() {
    return getClick2D(0);
  }

  /**
   * Return the x-coordinate of the default (local) player's mouse when
   * any button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @return x-coordinate (in screens) where mouse was clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getClickX() {
    return getClickX(0);
  }

  /**
   * Return the y-coordinate of the default (local) player's mouse when
   * any button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @return y-coordinate (in screens) where mouse was clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getClickY() {
    return getClickY(0);
  }

  /**
   * Get location for the given player's mouse when the left button was
   * clicked. Returns the location (a point) in screen coordinates
   * (0.0-1.0 for both x- and y-coordinates) where any button was
   * clicked.
   * 
   * @param id - player id to get coordinate for
   * @return the location of the mouse of the given player at left
   *         click; null if the player did not click this frame
   */
  public Location2D getLeftClick2D(int id) {
    return getPlayer(id).getMouse().getLeftClickLocation();
  }

  /**
   * Return the x-coordinate of the given player's mouse when left
   * button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @param id player to find mouse click location for
   * @return x-coordinate (in screens) where mouse was left clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getLeftClickX(int id) {
    if (getLeftClick2D(id) == null) {
      return Double.NaN;
    } else {
      return getLeftClick2D(id).x;
    }
  }

  /**
   * Return the y-coordinate of the given player's mouse when left
   * button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @param id player to find mouse click location for
   * @return y-coordinate (in screens) where mouse was left clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getLeftClickY(int id) {
    if (getLeftClick2D(id) == null) {
      return Double.NaN;
    } else {
      return getLeftClick2D(id).y;
    }
  }

  /**
   * Get location for the default (local) player's mouse when the left
   * button was clicked. Returns the location (a point) in screen
   * coordinates (0.0-1.0 for both x- and y-coordinates) where any
   * button was clicked.
   * 
   * @return the location of the mouse of the given player at left
   *         click; null if the player did not click this frame
   */
  public Location2D getLeftClick2D() {
    return getLeftClick2D(0);
  }

  /**
   * Return the x-coordinate of the default (local) player's mouse when
   * left button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @return x-coordinate (in screens) where mouse was left clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getLeftClickX() {
    return getLeftClickX(0);
  }

  /**
   * Return the y-coordinate of the default (local) player's mouse when
   * left button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @return y-coordinate (in screens) where mouse was left clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getLeftClickY() {
    return getLeftClickY(0);
  }

  /**
   * Get location for the given player's mouse when the middle button
   * was clicked. Returns the location (a point) in screen coordinates
   * (0.0-1.0 for both x- and y-coordinates) where any button was
   * clicked.
   * 
   * @param id - player id to get coordinate for
   * @return the location of the mouse of the given player at middle
   *         click; null if the player did not click this frame
   */
  public Location2D getMiddleClick2D(int id) {
    return getPlayer(id).getMouse().getMiddleClickLocation();
  }

  /**
   * Return the x-coordinate of the given player's mouse when middle
   * button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @param id player to find mouse click location for
   * @return x-coordinate (in screens) where mouse was middle clicked
   *         (if clicked this frame); {@code Double.NaN} if no mouse
   *         click this frame
   */
  public double getMiddleClickX(int id) {
    if (getMiddleClick2D(id) == null) {
      return Double.NaN;
    } else {
      return getMiddleClick2D(id).x;
    }
  }

  /**
   * Return the y-coordinate of the given player's mouse when middle
   * button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @param id player to find mouse click location for
   * @return y-coordinate (in screens) where mouse was middle clicked
   *         (if clicked this frame); {@code Double.NaN} if no mouse
   *         click this frame
   */
  public double getMiddleClickY(int id) {
    if (getMiddleClick2D(id) == null) {
      return Double.NaN;
    } else {
      return getMiddleClick2D(id).y;
    }
  }

  /**
   * Get location for the default (local) player's mouse when the middle
   * button was clicked. Returns the location (a point) in screen
   * coordinates (0.0-1.0 for both x- and y-coordinates) where any
   * button was clicked.
   * 
   * @return the location of the mouse of the given player at middle
   *         click; null if the player did not click this frame
   */
  public Location2D getMiddleClick2D() {
    return getMiddleClick2D(0);
  }

  /**
   * Return the x-coordinate of the default (local) player's mouse when
   * middle button was clicked. If no button was clicked, will return
   * the special numeric value <em>not a number</em> or {@code
   * Double.NaN}.
   * 
   * @return x-coordinate (in screens) where mouse was middle clicked
   *         (if clicked this frame); {@code Double.NaN} if no mouse
   *         click this frame
   */
  public double getMiddleClickX() {
    return getMiddleClickX(0);
  }

  /**
   * Return the y-coordinate of the default (local) player's mouse when
   * middle button was clicked. If no button was clicked, will return
   * the special numeric value <em>not a number</em> or {@code
   * Double.NaN}.
   * 
   * @return y-coordinate (in screens) where mouse was middle clicked
   *         (if clicked this frame); {@code Double.NaN} if no mouse
   *         click this frame
   */
  public double getMiddleClickY() {
    return getMiddleClickY(0);
  }

  /**
   * Get location for the given player's mouse when the right button was
   * clicked. Returns the location (a point) in screen coordinates
   * (0.0-1.0 for both x- and y-coordinates) where any button was
   * clicked.
   * 
   * @param id - player id to get coordinate for
   * @return the location of the mouse of the given player at right
   *         click; null if the player did not click this frame
   */
  public Location2D getRightClick2D(int id) {
    return getPlayer(id).getMouse().getRightClickLocation();
  }

  /**
   * Return the x-coordinate of the given player's mouse when right
   * button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @param id player to find mouse click location for
   * @return x-coordinate (in screens) where mouse was right clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getRightClickX(int id) {
    if (getRightClick2D(id) == null) {
      return Double.NaN;
    } else {
      return getRightClick2D(id).x;
    }
  }

  /**
   * Return the y-coordinate of the given player's mouse when right
   * button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @param id player to find mouse click location for
   * @return y-coordinate (in screens) where mouse was right clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getRightClickY(int id) {
    if (getRightClick2D(id) == null) {
      return Double.NaN;
    } else {
      return getRightClick2D(id).y;
    }
  }

  /**
   * Get location for the default (local) player's mouse when the right
   * button was clicked. Returns the location (a point) in screen
   * coordinates (0.0-1.0 for both x- and y-coordinates) where any
   * button was clicked.
   * 
   * @return the location of the mouse of the given player at right
   *         click; null if the player did not click this frame
   */
  public Location2D getRightClick2D() {
    return getRightClick2D(0);
  }

  /**
   * Return the x-coordinate of the default (local) player's mouse when
   * right button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @return x-coordinate (in screens) where mouse was right clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getRightClickX() {
    return getRightClickX(0);
  }

  /**
   * Return the y-coordinate of the default (local) player's mouse when
   * right button was clicked. If no button was clicked, will return the
   * special numeric value <em>not a number</em> or {@code Double.NaN}.
   * 
   * @return y-coordinate (in screens) where mouse was right clicked (if
   *         clicked this frame); {@code Double.NaN} if no mouse click
   *         this frame
   */
  public double getRightClickY() {
    return getRightClickY(0);
  }

  /**
   * Was any button on the given player's mouse pressed during the last
   * frame?
   * 
   * @param id player whose mouse should be checked
   * @return true if any button on the mouse was pressed this frame,
   *         false otherwise
   */
  public boolean mousePressed(int id) {
    return getPlayer(id).getMouse().buttonPressed();
  }

  /**
   * Was any button on the default (local) player's mouse pressed during
   * the last frame?
   * 
   * @return true if any button on the mouse was pressed this frame,
   *         false otherwise
   */
  public boolean mousePressed() {
    return mousePressed(0);
  }

  /**
   * Was the left button on the given player's mouse pressed during the
   * last frame?
   * 
   * @param id player whose mouse should be checked
   * @return true if any button on the mouse was pressed this frame,
   *         false otherwise
   */
  public boolean mouseLeftPressed(int id) {
    return getPlayer(id).getMouse().leftPressed();
  }

  /**
   * Was the left button on the default (local) player's mouse pressed
   * during the last frame?
   * 
   * @return true if any button on the mouse was pressed this frame,
   *         false otherwise
   */
  public boolean mouseLeftPressed() {
    return mouseLeftPressed(0);
  }

  /**
   * Was the middle button on the given player's mouse pressed during
   * the last frame?
   * 
   * @param id player whose mouse should be checked
   * @return true if any button on the mouse was pressed this frame,
   *         false otherwise
   */
  public boolean mouseMiddlePressed(int id) {
    return getPlayer(id).getMouse().middlePressed();
  }

  /**
   * Was the middle button on the default (local) player's mouse pressed
   * during the last frame?
   * 
   * @return true if any button on the mouse was pressed this frame,
   *         false otherwise
   */
  public boolean mouseMiddlePressed() {
    return mouseMiddlePressed(0);
  }

  /**
   * Was the right button on the given player's mouse pressed during the
   * last frame?
   * 
   * @param id player whose mouse should be checked
   * @return true if any button on the mouse was pressed this frame,
   *         false otherwise
   */
  public boolean mouseRightPressed(int id) {
    return getPlayer(id).getMouse().rightPressed();
  }

  /**
   * Was the right button on the default (local) player's mouse pressed
   * during the last frame?
   * 
   * @return true if any button on the mouse was pressed this frame,
   *         false otherwise
   */
  public boolean mouseRightPressed() {
    return mouseRightPressed(0);
  }

  // ----------End Mouse methods---------

  // ----------Start Keyboard methods---------
  /**
   * Returns the last key pressed for the given player.
   * 
   * @param id - identifer for player owning the keyboard to be checked
   * @return the last key pressed or Keyboard.NONE if no key was pressed
   *         this frame
   */
  public char getKeyPressed(int id) {
    return getPlayer(id).getKeyboard().getLastKey();
  }

  /**
   * Returns the last key pressed for the default (local) player.
   * 
   * @return the last key pressed or Keyboard.NONE if no key was pressed
   *         this frame
   */
  public char getKeyPressed() {
    return getKeyPressed(0);
  }

  /**
   * Returns the last key pressed for the given player converted to
   * lowercase.
   * 
   * @param id - identifer for player owning the keyboard to be checked
   * @return the last key pressed in lowercase (non-letters unchanged)
   *         or Keyboard.NONE if no key was pressed this frame
   */
  public char getLowerCaseKeyPressed(int id) {
    return Character.toLowerCase(getKeyPressed(id));
  }

  /**
   * Returns the last key pressed for the default (local) player
   * converted to lowercase.
   * 
   * @return the last key pressed in lowercase (non-letters unchanged)
   *         or Keyboard.NONE if no key was pressed this frame
   */
  public char getLowerCaseKeyPressed() {
    return getLowerCaseKeyPressed(0);
  }

  /**
   * Returns the last key pressed for the given player converted to
   * UPPERCASE.
   * 
   * @param id - identifer for player owning the keyboard to be checked
   * @return the last key pressed in UPPERCASE (non-letters unchanged)
   *         or Keyboard.NONE if no key was pressed this frame
   */
  public char getUpperCaseKeyPressed(int id) {
    return Character.toUpperCase(getKeyPressed(id));
  }

  /**
   * Returns the last key pressed for the default (local) player
   * converted to UPPERCASE.
   * 
   * @return the last key pressed in UPPERCASE (non-letters unchanged)
   *         or Keyboard.NONE if no key was pressed this frame
   */
  public char getUpperCaseKeyPressed() {
    return getUpperCaseKeyPressed(0);
  }

  /**
   * Was the up-arrow key pressed as the last key by the given player?
   * 
   * @param id identifier for the player owning the keyboard to be
   *          checked
   * @return true if up-arrow key was last key pressed; false otherwise
   */
  public boolean upPressed(int id) {
    return getPlayer(id).getKeyboard().upPressed();
  }

  /**
   * Was the up-arrow key pressed as the last key by the default (local)
   * player?
   * 
   * @return true if up-arrow key was last key pressed; false otherwise
   */
  public boolean upPressed() {
    return upPressed(0);
  }

  /**
   * Was the down-arrow key pressed as the last key by the given player?
   * 
   * @param id identifier for the player owning the keyboard to be
   *          checked
   * @return true if down-arrow key was last key pressed; false
   *         otherwise
   */
  public boolean downPressed(int id) {
    return getPlayer(id).getKeyboard().downPressed();
  }

  /**
   * Was the down-arrow key pressed as the last key by the default
   * (local) player?
   * 
   * @return true if down-arrow key was last key pressed; false
   *         otherwise
   */
  public boolean downPressed() {
    return downPressed(0);
  }

  /**
   * Was the right-arrow key pressed as the last key by the given
   * player?
   * 
   * @param id identifier for the player owning the keyboard to be
   *          checked
   * @return true if right-arrow key was last key pressed; false
   *         otherwise
   */
  public boolean rightPressed(int id) {
    return getPlayer(id).getKeyboard().rightPressed();
  }

  /**
   * Was the right-arrow key pressed as the last key by the default
   * (local) player?
   * 
   * @return true if right-arrow key was last key pressed; false
   *         otherwise
   */
  public boolean rightPressed() {
    return rightPressed(0);
  }

  /**
   * Was the left-arrow key pressed as the last key by the given player?
   * 
   * @param id identifier for the player owning the keyboard to be
   *          checked
   * @return true if left-arrow key was last key pressed; false
   *         otherwise
   */
  public boolean leftPressed(int id) {
    return getPlayer(id).getKeyboard().leftPressed();
  }

  /**
   * Was the left-arrow key pressed as the last key by the default
   * (local) player?
   * 
   * @return true if left-arrow key was last key pressed; false
   *         otherwise
   */
  public boolean leftPressed() {
    return leftPressed(0);
  }

  /**
   * Was any key pressed by the given player?
   * 
   * @param id identifier for the player owning the keyboard to be
   *          checked
   * @return true if a key was pressed this frame; false otherwise
   */
  public boolean keyPressed(int id) {
    return getPlayer(id).getKeyboard().keyPressed();
  }

  /**
   * Was any key pressed by the default (local) player?
   * 
   * @return true if a key was pressed this frame; false otherwise
   */
  public boolean keyPressed() {
    return keyPressed(0);
  }

  // ----------End Keyboard methods---------

  // ----------Start Canvas methods---------
  /**
   * Set the aspect ratio of the AnimationCanvas.
   * <b>This method should only be called from either setup or advance.</b>
   * @param aspect the new aspect ratio
   */
  public void setAspect(double aspect) {
    getCanvas().setAspect(aspect);
  }

  /**
   * Restore the cursor; used to un-hide the cursor.
   */
  @Override
  public void restoreCursor() {
    getCanvas().restoreCursor();
  }

  /**
   * Hide the cursor; remove it from the window
   */
  @Override
  public void removeCursor() {
    getCanvas().removeCursor();
  }

  /**
   * Set the cursor to the image indicated by the given url.
   * 
   * @param url the location of the cursor resource
   */
  @Override
  public void setCursor(URL url) {
    getCanvas().setCursor(url);
  }

  /**
   * Get the largest (right-most) x-coordinate on the screen
   * 
   * @return the largest x-coordinate
   */
  public double getMaxX() {
    return getCanvas().getMaxX();
  }

  /**
   * Get the largest (lowest) y-coordinate on the screen
   * 
   * @return the largest y-coordinate
   */
  public double getMaxY() {
    return getCanvas().getMaxY();
  }

  /**
   * Add all of the sprites ({@code sprite} can be a variable number of
   * {@link Sprite Sprite} objects) to the game's canvas. Real work is
   * forwarded to the {@link AnimationCanvas AnimationCanvas} class.
   * 
   * @param sprite a list of {@link Sprite} objects to add to the top of
   *          the current scene
   */
  public void addSprite(Sprite... sprite) {
    getCanvas().addSprite(sprite);
  }

  /**
   * Add all of the sprites ({@code sprite} can be a variable number of
   * {@link Sprite Sprite} objects) to the games canvas with the given
   * z-ordering or z-level. Real work is forwarded to the
   * {@link AnimationCanvas AnimationCanvas} class.
   * 
   * @param layer z-order to apply to these sprites
   * @param sprite a list of {@link Sprite} objects to add to the given
   *          layer of the current scene
   */
  public void addSprite(double layer, Sprite... sprite) {
    getCanvas().addSprite(layer, sprite);
  }

  /**
   * Get the z-ordering (distance up from the background) of the given
   * sprite.
   * 
   * @param sprite the {@link Sprite} for which a level must be
   *          determined
   * @return the z-ordering or distance from the background up to the
   *         layer containing the sprite
   */
  public double getLayer(Sprite sprite) {
    return getCanvas().getLayer(sprite);
  }

  /**
   * Get an array of all of the sprites in a given layer.
   * 
   * @param layer what distance from the background are sprites to be
   *          found
   * @return an array containing references to all {@link Sprite} at the
   *         given level
   */
  public Sprite[] getLayer(double layer) {
    return getCanvas().getLayer(layer);
  }

  /**
   * Flatten all the layers into a single background layer (keeping the
   * overall z-ordering).
   */
  public void flattenLayers() {
    getCanvas().flattenLayers();
  }

  /**
   * Remove the given sprite or sprites (({@code sprite} can be a
   * variable number of {@link Sprite Sprite} objects) from the game.
   * Real work is forwarded to the {@link AnimationCanvas
   * AnimationCanvas} class.
   * 
   * @param sprite a list of {@link Sprite} objects to remove from the
   *          current scene
   */
  public void removeSprite(Sprite... sprite) {
    getCanvas().removeSprite(sprite);
  }

  /**
   * Remove all sprites from the canvas.
   */
  public void removeAllSprites() {
    getCanvas().removeAllSprites();
  }

  /**
   * Does the game contain the given sprite?
   * 
   * @param sprite the {@link Sprite} to check for in the game
   * @return true if the sprite is in the game (has been added with a
   *         call to {@link #addSprite}) or false otherwise
   */
  public boolean containsSprite(Sprite sprite) {
    return getCanvas().containsSprite(sprite);
  }

  /**
   * Get an array of all of the sprites in the game. This will contain
   * all of the Sprites which have been added (through
   * {@link #addSprite}) and not removed (through {@link #removeSprite}
   * ).
   * 
   * @return an array of all of the Sprites that are in the game
   */
  public Sprite[] getAllSprites() {
    return getCanvas().getAllSprites();
  }

  /**
   * Add all of the sprites ({@code sprite} can be a variable number of
   * {@link Sprite Sprite} objects) to the game's canvas. Real work is
   * forwarded to the {@link AnimationCanvas AnimationCanvas} class.
   * 
   * @param sprite a list of {@link Sprite} objects to add to the bottom
   *          of the current scene
   */
  public void addBottom(Sprite... sprite) {
    getCanvas().addBottom(sprite);
  }

  /**
   * Get the aspect ratio of the game on the screen
   * 
   * @return the aspect ratio of width to height
   */
  public double getAspect() {
    return getCanvas().getAspect();
  }

  // ----------End Canvas methods---------

  // ----------Start Palette methods---------
  /**
   * Create a solid, opaque color with the given red, green, and blue
   * values.
   * 
   * @param red the amount of red to include; range is [0..255] with 0
   *          meaning no red
   * @param green the amount of green to include; range is [0..255] with
   *          0 meaining no green
   * @param blue the amount of blue to include; range is [0..255] with 0
   *          meaning no blue
   * @return a {@link java.awt.Color Color} object for the given shade
   */
  public static Color getColor(int red, int green, int blue) {
    return Palette.getColor(red, green, blue);
  }

  /**
   * Create a solid, translucent color with the given red, gree, and
   * blue values. The opacity value determines how opaque the color is
   * with 0 being completely transparent and 255 being completely
   * opaque.
   * 
   * @param red the amount of red to include; range is [0..255] with 0
   *          meaning no red
   * @param green the amount of green to include; range is [0..255] with
   *          0 meaning no green
   * @param blue the amount of blue to include; range is [0..255] with 0
   *          meaning no blue
   * @param opacity the amount of opaqueness for the color; range is
   *          [0..255] with 0 meaning transparent and 255 meaning opaque
   * @return a {@link java.awt.Color Color} object for the given shade
   */
  public static Color getColor(int red, int green, int blue, int opacity) {
    return Palette.getColor(red, green, blue, opacity);
  }

  /**
   * Create a solid color with from the name given on <a
   * href="http://www.pitt.edu/~isg/cis/web/cgi/rgb.html">this web
   * page</a>. The colors are those understood by the XWindows windowing
   * system.
   * 
   * @param name color name
   * @return color matching name (or black with given if name is not a
   *         known color name)
   */
  public static Color getColor(String name) {
    return Palette.getColor(name);
  }

  /**
   * Create a semi-transparent color with from the name given on <a
   * href="http://www.pitt.edu/~isg/cis/web/cgi/rgb.html">this web
   * page</a>.
   * 
   * @param name color name
   * @param opacity the transparency for the new color (0-255)
   * @return color matching name with given opacity (or black with given
   *         opacity if name is not a known color name)
   */
  public static Color getColor(String name, int opacity) {
    return Palette.getColor(name, opacity);
  }

  /**
   * Get the name of the given color (specified by red, green, and blue
   * channels.
   * 
   * @param r red value [0..255]
   * @param g green value [0..255]
   * @param b blue value [0..255]
   * @return returns the name for the color (if there is one) and
   *         returns Palette.NO_SUCH_COLOR otherwise.
   */
  public static String getColorName(int r, int g, int b) {
    return Palette.getColorName(r, g, b);
  }

  /**
   * Get the name of given color (specified as a "#rrggbb" hex string
   * (note that it can safely be a "#rrggbbaa" string, too.
   * 
   * @param rgb string of the form "#rrggbb" where rr, gg, and bb are
   *          the color channel values for red, green, and blue
   *          expressed in hex
   * @return returns the name for the color (if there is one) and
   *         returns Palette.NO_SUCH_COLOR otherwise.
   */
  public static String getColorName(String rgb) {
    return Palette.getColorName(rgb);
  }

  /**
   * Get the name of the color expressed as a Java Color object.
   * 
   * @param color the color to name (if possible)
   * @return returns the name for the color (if there is one) and
   *         returns NO_SUCH_COLOR otherwise.
   */
  public static String getColorName(Color color) {
    return Palette.getColorName(color);
  }

  /**
   * Convert a "#rrggbb" string into a Java color object corresponding
   * to the given color.
   * 
   * @param rgb string of the form "#rrggbb" where rr, gg, and bb are
   *          the color channel values for red, green, and blue
   *          expressed in hex
   * @return an opaque Color corresponding to the given rgb values.
   */
  public static Color getColorFromRGB(String rgb) {
    return Palette.getColorFromRGB(rgb);
  }

  /**
   * Convert a "#rrggbbaa" string into a Java color object corresponding
   * to the given color.
   * 
   * @param rgba string of the form "#rrggbbaa" where rr, gg, and bb are
   *          the color channel values for red, green, and blue
   *          expressed in hex and aa is the alpha channel (or opacity)
   *          expressed in hex
   * @return a Color corresponding to the given rgba values.
   */
  public static Color getColorFromRGBA(String rgba) {
    return Palette.getColorFromRGBA(rgba);
  }

  /**
   * Convert a "#rrggbb" string into a Java color object corresponding
   * to the given color.
   * 
   * @param rgb string of the form "#rrggbb" where rr, gg, and bb are
   *          the color channel values for red, green, and blue
   *          expressed in hex
   * @param opacity the opacity of the color [0..255] where 255 is
   *          opaque and 0 is transparent
   * @return a Color corresponding to the given rgb and transparency
   *         values
   */
  public static Color getColorFromRGB(String rgb, int opacity) {
    return Palette.getColorFromRGB(rgb, opacity);
  }

  /**
   * Add a color to the list of colors this Palette knows. Lets you
   * define and use your own named colors. Color is specified as a Java
   * {@link java.awt.Color Color} object.
   * 
   * @param name
   * @param color
   */
  public static void addColor(String name, Color color) {
    Palette.addColor(name, color);
  }

  /**
   * Add a color to the list of colors this Palette knows. Lets you
   * define named colors which you can later use. Color is specified as
   * a Web string "#rrggbb" where rr, gg, and bb are hexidecimal numbers
   * specifying color channel values for red, green, and blue.
   * 
   * @param name the new color's name
   * @param rgb the new color's rgb value ("#rrggbb")
   */
  public static void addColor(String name, String rgb) {
    Palette.addColor(name, rgb);
  }

  /**
   * Returns a random color. The randome sequence is shared across
   * multiple games in the same multi-player session.
   * 
   * @return a randomly selected color.
   */
  public Color randomColor() {
    return Palette
        .getRandomColor((getCurrentGame() != null) ? getCurrentGame()
            .getRandom() : new Random());
  }

  // ----------End Palette methods---------

  // ----------Start Random methods---------
  /**
   * Return this {@link fang2.core.Game Game}'s random number generator.
   * The random number sequence is shared across multipe games in the
   * same multi-player session.
   * 
   * @return the {@link java.util.Random Random} used by this game to
   *         generate random numbers.
   */
  public Random getRandom() {
    return random;
  }

  /**
   * Return a randomly selected boolean value. The result is {@code
   * true} and {@code false} half of the time each.
   * 
   * @return a randomly selected boolean value.
   */
  public boolean randomBoolean() {
    return getRandom().nextBoolean();
  }

  /**
   * Generate a shared random double on the range [0.0, 1.0). Here
   * "shared" means that multiple games running in the same FANG game
   * session will see the same sequence of numbers. [0.0, 1.0) means
   * zero is included by one is excluded from the range.
   * 
   * @return random, uniformly-distributed double on the range [0.0,
   *         1.0)
   */
  public double randomDouble() {
    return getRandom().nextDouble();
  }

  /**
   * Generate a shared random double on the range [0.0, max). The random
   * sequence is shared with other games in a multi-player session. The
   * range dose not include max.
   * 
   * @param max the top end of the range of random doubles desired
   * @return random, uniformly-distributed double on the range [0.0,
   *         max).
   */
  public double randomDouble(double max) {
    return getRandom().nextDouble() * max;
  }

  /**
   * Generate a shared random double on the range [min, max). The random
   * sequence is shared with other games in a multi-player session. The
   * range includes min but excludes max.
   * 
   * @param min the bottom end of the range of random doubles desired
   * @param max the top end of the range of random doubles desired
   * @return random, uniformly-distributed double on [min, max).
   */
  public double randomDouble(double min, double max) {
    return getRandom().nextDouble() * (max - min) + min;
  }

  /**
   * Generate a shared random integer on the full 32-bit range of int.
   * The random bit sequence is shared between multiple games in the
   * same game session.
   * 
   * @return random, uniformly-distributed in on the full range of int
   */
  public int randomInt() {
    return getRandom().nextInt();
  }

  /**
   * Generate a random number on the range [0, n). Shared random
   * sequence between multi-player games.
   * 
   * @param n must be positive; the number of different values desired
   * @return random, uniformly-distributed int on the range [0, n)
   */
  public int randomInt(int n) {
    return getRandom().nextInt(n);
  }

  /**
   * Generate a random integer on the range [min, max). Shared random
   * sequence between multi-player games.
   * 
   * @param min the bottom end of the range of integers desired
   *          INCLUSIVE
   * @param max the top end of the range of integers desired EXCLUSIVE
   * @return random, uniformly-distributed int on the range [min, max)
   */
  public int randomInt(int min, int max) {
    return getRandom().nextInt(max - min) + min;
  }

  // ----------End Random methods---------

  // ----------Start Player methods---------
  /**
   * Get the number of lives the given player has remaining.
   * 
   * @param id player id to check life count for
   * @return the number of lives set for the given user
   */
  public int getLives(int id) {
    return getPlayer(id).getLives();
  }

  /**
   * Get the number of lives player 0 (the local player) has remaining.
   * 
   * @return the number of lives set for the local player
   */
  public int getLives() {
    return getLives(0);
  }

  /**
   * Set the number of lives for the given player.
   * 
   * @param lives the new number of lives for the given player
   * @param id the player id of the player to set lives for
   */
  public void setLives(int lives, int id) {
    getPlayer(id).setLives(lives);
  }

  /**
   * Set the number of lives for the local player.
   * 
   * @param lives the new number of lives for the local player
   */
  public void setLives(int lives) {
    setLives(lives, 0);
  }

  /**
   * Get the player name associated with the given player id.
   * 
   * @param id player id of the player for whom a name is desired
   * @return the name of the given player
   */
  public String getPlayerName(int id) {
    return getPlayer(id).getName();
  }

  /**
   * Get the player name associated with the local player.
   * 
   * @return the name of the local player
   */
  public String getPlayerName() {
    return getPlayerName(0);
  }

  /**
   * Set the player name for the given player.
   * 
   * @param name the new name for the given player
   * @param id the player id of the player to set a name for
   */
  public void setPlayerName(String name, int id) {
    getPlayer(id).setName(name);
  }

  /**
   * Set the player name for the local player.
   * 
   * @param name the new name for the local player
   */
  public void setPlayerName(String name) {
    setPlayerName(name, 0);
  }

  /**
   * Get the score associated with the given player id
   * 
   * @param id player id for whom to fetch the score
   * @return the score of the given player id
   */
  public int getScore(int id) {
    return getPlayer(id).getScore();
  }

  /**
   * Get the score associated with the local player
   * 
   * @return the score of the local player
   */
  public int getScore() {
    return getScore(0);
  }

  /**
   * Set the score associated with the given player id
   * 
   * @param score the new score for the given player
   * @param id the player for whom to fetch the score
   */
  public void setScore(int score, int id) {
    getPlayer(id).setScore(score);
  }

  /**
   * Set the score associated with the local player
   * 
   * @param score the new score for the local player
   */
  public void setScore(int score) {
    setScore(score, 0);
  }

  /**
   * Get the local player's player id.
   * 
   * @return the local player's id
   */
  public int getPlayerID() {
    return getID();
  }

  /**
   * Is there a message pending for the given player id?
   * 
   * @param id player id to check for pending message
   * @return true if there is a message pending, false otherwise
   */
  public boolean hasMessage(int id) {
    return getPlayer(id).hasMessage();
  }

  /**
   * Is there a message pending for the local player?
   * 
   * @return true if there is a message pending, false otherwise
   */
  public boolean hasMessage() {
    return hasMessage(0);
  }

  /**
   * Get the pending message for the given player. This is a blocking
   * call.
   * 
   * @param id player id to fetch message for
   * @return an Object representing the message sent
   */
  public Object getMessage(int id) {
    return getPlayer(id).getMessage();
  }

  /**
   * Get the pending message for the local player. This is a blocking
   * call.
   * 
   * @return an Object representing the message sent
   */
  public Object getMessage() {
    return getMessage(0);
  }

  /**
   * The standard setup method. Part of the FANG videogame startup and
   * loop.
   * 
   * Intended to be overridden by programs implementing actual games.
   * Default method defined to do nothing.
   */
  @Override
  public void setup() {
    // this method intentionally left empty
  }

  /**
   * The standard, untimed, advance-game-state method. Part of the FANG
   * videogame startup and loop.
   * 
   * Intended to be overridden by games implementing real games. By
   * default, advance() does nothing.
   */
  @Override
  public void advance() {
    // this method intentionally left empty
  }

  /**
   * Advance with time elapsed since previous call to advance. By
   * default, just calls the zero parameter version of advance.
   * 
   * @param deltaTime number of seconds since last call to advance
   */
  @Override
  public void advance(double deltaTime) {
    advance();
  }
}
