package fang2.core;

import static fang2.ui.ErrorConsole.addError;
import static fang2.ui.ErrorConsole.fixHTML;
import static fang2.ui.ErrorConsole.fixedWidth;
import static fang2.ui.ErrorConsole.getErrorFile;
import static fang2.ui.ErrorConsole.getErrorLineNumber;
import static fang2.ui.ErrorConsole.getErrorMethod;
import static fang2.ui.ErrorConsole.getLine;
import static fang2.ui.ErrorConsole.indent;

import java.awt.Color;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.KeyboardFocusManager;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;

import javax.swing.JComponent;

/**
 * This class is a JPanel that contains Sprites.<br>
 * <br>
 * The license below must remain in the software unaltered.
 *
 * @author  Jam Jenkins
 */

public class AnimationCanvas
  extends JComponent {
  /** used for serialization versioning */
  private static final long serialVersionUID = 1L;

  /** the color to use when clearing the background */
  protected static final Color DEFAULT_BACKGROUND = Color.BLACK;

  /** default size of canvas */
  protected static final Dimension DEFAULT_SIZE = new Dimension(400,
      400);

  /** width/height aspect ratio */
  private double aspect = 1;

  /** shared validity flag for allTransformers */
  /* package */ boolean allTransformersDirty = true;
  /** set of all installed transformers */
  private final Set<Transformer> allTransformers = new LinkedHashSet<Transformer>();
  
  /**
   * list of <layer, sprite> pairs that have been added but are not yet
   * actually on the canvas. Set to non-null, {@link #holder}, when
   * traversing {@link #sprites}.
   */
  private List<DeferredSpriteAddElement> deferredAdd = null;

  /**
   * the real deferred add list; constructed once and then deferredAdd
   * either points at it or not. Eliminates the call to new {@link
   * ArrayList} before every traversal of {@link #sprites}.
   */
  private final List<DeferredSpriteAddElement> holder =
    new ArrayList<DeferredSpriteAddElement>();

  /** reverse map for sprites */
  private final HashMap<Sprite, Double> reverseSprites;

  /**
   * the ordered collection of Sprites. The last added Sprite will
   * appear on top.
   */
  private final TreeMap<Double, LinkedHashSet<Sprite>> sprites;

  /**
   * constructs an empty canvas with the default size
   */
  public AnimationCanvas() {
    this(DEFAULT_SIZE);
  }

  /**
   * constructs an empty canvas with a give size and default color.
   *
   * @param  size  the width and height to make the AnimationCanvas.
   *               This width and height specify the target aspect ratio
   *               and preferred size. If the size of the
   *               AnimationCanvas expands or shrinks, it will maintain
   *               the original aspect ratio of width to height.
   */
  public AnimationCanvas(Dimension size) {
    this(size, DEFAULT_BACKGROUND);
  }

  /**
   * constructs an empty canvas with a give size and background color
   *
   * @param  size             the width and height to make the
   *                          AnimationCanvas. This width and height
   *                          specify the target aspect ratio and
   *                          preferred size. If the size of the
   *                          AnimationCanvas expands or shrinks, it
   *                          will maintain the original aspect ratio of
   *                          width to height.
   * @param  backgroundColor  the color to set the background to.
   */
  public AnimationCanvas(Dimension size, Color backgroundColor) {
    super();
    aspect = size.getWidth() / size.getHeight();
    setSize(size);
    sprites = new TreeMap<Double, LinkedHashSet<Sprite>>();
    reverseSprites = new HashMap<Sprite, Double>();
    setFocusable(true);
    setOpaque(true);
    setBackground(backgroundColor);
    setIgnoreRepaint(true);
    // grab focus whenever the mouse is over the window
    KeyboardFocusManager.getCurrentKeyboardFocusManager()
      .setGlobalCurrentFocusCycleRoot(this);
    addMouseMotionListener(new FocusGrabber());
  }

  /**
   * adds sprite to the bottom of the canvas. This method can be useful
   * for adding and/or changing the background.
   *
   * @param  sprite  the sprite to add at the bottom
   */
  public void addBottom(Sprite... sprite) {
    double key = 0;
    if (sprites.size() > 0) {
      key = sprites.firstKey() - 1;
    }
    addSprite(key, sprite);
  }

  /**
   * Add to the given layer all of the sprites in the list (this method
   * has a variable number of arguments). If the sprites data structure
   * is being traversed, the add is deferred (by making an add list
   * entry). In case of deferred add, the sprite is added (with another
   * call to this method) after the traversal ends.
   *
   * @param  layer   drawing layer in which to add the {@link Sprite}s
   *                 in the list of sprites. The higher the layer
   *                 number, the further up the Sprite is drawn
   *                 (z-ordering is by layer in increasing order)
   * @param  sprite  the list of {@link Sprite}s to be added at the
   *                 given layer.
   */
  public void addSprite(double layer, Sprite... sprite) {
    int count = 0;
    for (Sprite s : sprite) {
      /* bcl - deferred add behavior is indicated by a non-null value of
       * deferredAdd field. Field is set in beginSpritesTraversal, unset
       * in endSpritesTraversal. When adding is deferred, added sprites
       * go onto deferredAdd rather than into sprites. When cleaning up
       * after traversal, all deferred sprites are added with another
       * call to this method.
       */
      // ----- Add is deferred -----
      if (deferredAdd != null) {
        deferredAdd.add(new DeferredSpriteAddElement(s, layer));
        continue;
      }

      // ----- Add is immediate
      if (s == null) {
        NullPointerException npe = new NullPointerException(
            "Cannot add null Sprite to AnimationCanvas");
        reportNullSprite(getErrorFile(npe), getErrorLineNumber(npe),
          getErrorMethod(npe), count);
        throw npe;
      } else {
        addSingleSprite(layer, s);
      }
      count++;

      /* an immeadiate add means the transformer's list is not right */
      allTransformersDirty = true;
    }
  }

  /**
   * adds a Sprite to the canvas. The last added Sprite appears on top.
   *
   * @param  sprite  the Sprite to be added
   */
  public void addSprite(Sprite... sprite) {
    double layer = 0;
    if (sprites.size() > 0) {
      layer = sprites.lastKey() + 1;
    }
    addSprite(layer, sprite);
  }

  /**
   * determines if the Sprite exists on the AnimationCanvas. This method
   * does not indicate whether the Sprite in question is visible or
   * enabled, just whether it is currently on the AnimationCanvas.
   *
   * @param   sprite  the Sprite to check
   *
   * @return  true if th sprite does exist in the collection, false
   *          otherwise
   */
  public boolean containsSprite(Sprite sprite) {
    return reverseSprites.containsKey(sprite);
  }

  /**
   * gives each sprite a different layer where the layers start at 0 and
   * go up by one for each consecutive sprite
   */
  public void flattenLayers() {
    Sprite[] all = getAllSprites();
    sprites.clear();
    for (int i = 0; i < all.length; i++) {
      LinkedHashSet<Sprite> element = new LinkedHashSet<Sprite>();
      element.add(all[i]);
      sprites.put((double) i, element);
    }
  }

  /**
   * gets a copy of the Sprite collection
   *
   * @return  the copy of the Sprites collection in array form
   */
  public Sprite[] getAllSprites() {
    Sprite[] all = new Sprite[reverseSprites.size()];
    int index = 0;
    for (LinkedHashSet<Sprite> group : sprites.values()) {
      for (Sprite s : group) {
        all[index] = s;
        index++;
      }
    }
    return all;
  }

  /**
   * gets the aspect ratio of width to height
   *
   * @return  the aspect ratio width/height
   */
  public double getAspect() {
    return aspect;
  }

  /**
   * returns height of the canvas on which drawing can occur. The canvas
   * size is clipped to maintain the initial aspect ratio.
   *
   * @return  height in pixels
   */
  @Override
  public int getHeight() {
    if ((((double) getSize().width) / getSize().height) > aspect) {
      return getSize().height;
    }
    return (int) (getSize().width / aspect);
  }

  /**
   * gets the Sprites in the given layer
   *
   * @param   layer  the position in which to add the sprite. The
   *                 highest layer is on top.
   *
   * @return  the sprites in the layer, which could possibly be a zero
   *          length array if there is no such layer
   */
  public Sprite[] getLayer(double layer) {
    if (!sprites.containsKey(layer)) {
      return new Sprite[0];
    }
    return sprites.get(layer).toArray(new Sprite[0]);
  }

  /**
   * gets the layer on which a sprite is drawn. The highest layer is on
   * top.
   *
   * @param   sprite  the sprite which to find the layer of.
   *
   * @return  the z-ordering (or layer) where the sprite resides (if it
   *          is in the game); returns Double.NaN if the sprite is not
   *          in the game
   */
  public double getLayer(Sprite sprite) {
    if (reverseSprites.containsKey(sprite)) {
      return reverseSprites.get(sprite);
    }
    return Double.NaN;
  }

  /**
   * gets the maximum x value which can be displayed on the canvas. When
   * the canvas is square, this returns 1. When the canvas is more long
   * than it is wide, the return value will be the width/height, and
   * when the width is less than the height, the return value is 1.
   *
   * @return  the maximum displayable x value
   */
  public double getMaxX() {
    if (aspect > 1) {
      return aspect;
    }
    return 1;
  }

  /**
   * gets the maximum y value which can be displayed on the canvas. If
   * the canvas is square, the return value is 1. If the canvas is
   * taller than it is wide, the return value is height/width. If the
   * canvas is wider than it is tall, the return value is 1.
   *
   * @return  the maximum displayable y value
   */
  public double getMaxY() {
    if (aspect < 1) {
      return 1 / aspect;
    }
    return 1;
  }

  /**
   * gets the preferred size as the current size
   *
   * @return  the current size
   */
  @Override
  public Dimension getPreferredSize() {
    return getSize();
  }

  /**
   * returns width of the canvas on which drawing can occur. The canvas
   * size is clipped to maintain the initial aspect ratio.
   *
   * @return  width in pixels
   */
  @Override
  public int getWidth() {
    if ((((double) getSize().width) / getSize().height) > aspect) {
      return (int) (aspect * getSize().height);
    }
    return getSize().width;
  }

  /**
   * redraws the area which contains the Sprites
   */
  public void paintImmediately() {
    paintImmediately(0, 0, getSize().width, getSize().height);
  }

  /**
   * clears the canvas of all Sprites
   */
  public void removeAllSprites() {
    sprites.clear();
    reverseSprites.clear();
    allTransformersDirty = true;
  }

  /**
   * removes the cursor from the game canvas
   */
  public void removeCursor() {
    Toolkit toolkit = getToolkit();
    Image image = toolkit.createImage(new byte[0]);
    setCursor(getToolkit().createCustomCursor(image, new Point(),
        "none"));
  }

  /**
   * removes the Sprite from the canvas. Removing a non-existent sprite
   * has no effect and does not cause an exception.
   *
   * @param  sprite  the Sprite to be removed
   */
  public void removeSprite(Sprite... sprite) {
    if (sprite == null) {
      return;
    }
    for (Sprite s : sprite) {
      if ((s == null) || !reverseSprites.containsKey(s)) {
        continue;
      }

      removeSingleSprite(s);
      /* if any sprite was removed, allTransformers is dirty */
      allTransformersDirty = true;
    }
 }

  /**
   * Restore the graphics cursor.
   */
  public void restoreCursor() {
    this.setCursor(Cursor.getDefaultCursor());
  }
  
  /**
   * Set the screen's aspect ratio to the given value.
   * <b>This method should only be called from either setup or advance.</b>
   * @param aspect the new aspect ratio (pixel height to pixel width)
   */
  public void setAspect(double aspect) {
    this.aspect = aspect;
    Container container=getParent();
    if(container!=null)
    	container.repaint();
  }

  /**
   * sets the cursor for the game engine
   *
   * @param  url  the image to display as the cursor
   */
  public void setCursor(URL url) {
    Toolkit toolkit = getToolkit();
    Image image = toolkit.createImage(url);
    setCursor(getToolkit().createCustomCursor(image, new Point(),
        "none"));
  }

  private void gatherTransfomers() {
    allTransformers.clear();
    for (LinkedHashSet<Sprite> group : sprites.values()) {
      for (Sprite sprite : group) {
        sprite.reportTransformers(allTransformers);
      }
    }
  }
  
  /**
   * Pump {@link Transformer}s, update the {@link Sprite} data structures, and pump (legacy) Trackers. 
   * After pumping the {@link Transformer}, the destroyed sprites are cleaned up.
   * updates the cached array and removes all destroyed Sprites
   */
  public void updateSprites(double timeInterval) {
    beginSpritesTraversal();
    if (allTransformersDirty) 
      gatherTransfomers();
    
    for (Transformer transformerNG : allTransformers) {
      transformerNG.nonMaskableAdvance(timeInterval);
      transformerNG.advance(timeInterval);
    }
    
    for (LinkedHashSet<Sprite> group : sprites.values()) {
      for (Sprite sprite : group) {
        sprite.applyTransformerNG();
        sprite.update();
      }
    }
    endSpritesTraversal();

    ArrayList<Double> keysToRemove = new ArrayList<Double>();
    ArrayList<Sprite> toRemove = new ArrayList<Sprite>();
    double key;
    for (LinkedHashSet<Sprite> group : sprites.values()) {
      toRemove.clear();
      key = Double.NaN;
      for (Sprite sprite : group) {
        if (sprite.isDestroyed()) {
          toRemove.add(sprite);
          key = reverseSprites.get(sprite);
          reverseSprites.remove(sprite);
        }
      }
      group.removeAll(toRemove);
      if (group.size() == 0) {
        keysToRemove.add(key);
      }
    }
    for (Double keys : keysToRemove) {
      sprites.remove(keys);
    }
  }

  private void addSingleSprite(double layer, Sprite sprite) {
    LinkedHashSet<Sprite> element;
    if (sprites.containsKey(layer)) {
      element = sprites.get(layer);
    } else {
      element = new LinkedHashSet<Sprite>();
    }
    removeSingleSprite(sprite);
    element.add(sprite);
    reverseSprites.put(sprite, layer);
    sprites.put(layer, element);
  }

  /**
   * Call just before traversing {@link #sprites} field. This will set
   * things up so that {@link #addSprite(double, Sprite...)} will add
   * sprites to the deferred list rather than modifying sprites while it
   * is being traversed. Remember to call {@link #endSpritesTraversal()}
   * when the traversal is done.
   */
  private void beginSpritesTraversal() {
    deferredAdd = holder;
  }

  /**
   * paints a rectangle in the default background color
   *
   * @param  brush  the Graphics of the component
   */
  private void clearBackground(Graphics2D brush) {
    RenderingHints hints = new RenderingHints(null);
    hints.put(RenderingHints.KEY_RENDERING,
      RenderingHints.VALUE_RENDER_SPEED);
    hints.put(RenderingHints.KEY_ANTIALIASING,
      RenderingHints.VALUE_ANTIALIAS_ON);
    brush.addRenderingHints(hints);
    brush.setBackground(getBackground());
    brush.clearRect(0, 0, 1000, 1000);
  }

  /**
   * Clean up after a traversal of {@link #sprites}. The {@link
   * #deferredAdd} field will be checked and if there are any pending
   * adds, they will be handled before this method returns.
   */
  private void endSpritesTraversal() {
    deferredAdd = null;
    if (!holder.isEmpty()) {
      for (DeferredSpriteAddElement da : holder) {
        addSprite(da.layer, da.sprite);
      }
      holder.clear();
    }
  }

/*  private Map<Tracker, List<Sprite>> getTrackerMap() {
    HashMap<Tracker, List<Sprite>> all =
      new HashMap<Tracker, List<Sprite>>();
    for (LinkedHashSet<Sprite> layer : sprites.values()) {
      for (Sprite sprite : layer) {
        if (sprite.getTracker() == null) {
          continue;
        }
        if (sprite.getTracker() instanceof CompositeTracker) {
          CompositeTracker composite = (CompositeTracker)
            sprite.getTracker();
          for (Tracker track : composite.getFlatContents()) {
            if (!all.containsKey(track)) {
              all.put(track, new ArrayList<Sprite>());
            }
            all.get(track).add(sprite);
          }
        } else {
          Tracker track = sprite.getTracker();
          if (!all.containsKey(track)) {
            all.put(track, new ArrayList<Sprite>());
          }
          all.get(track).add(sprite);
        }
      }
    }
    return all;
  }*/

  /**
   * paints all of the Sprites
   *
   * @param  brush  the Graphics of the component
   */
  private void paintSprites(Graphics2D brush) {
    for (LinkedHashSet<Sprite> group : sprites.values()) {
      for (Sprite sprite : group) {
        sprite.paintInternal(brush);
      }
    }
  }

  /**
   * this method removes a single sprite in a non-recursive manner. For
   * example, if the sprite being removed is a CompositeSprite, only
   * that sprite is removed, and not all of its component Sprites.
   *
   * @param  spriteToRemove the {@link Sprite} to remove from this canvas
   */
  private void removeSingleSprite(Sprite spriteToRemove) {
    if ((spriteToRemove == null) || !reverseSprites.containsKey(spriteToRemove)) {
      return;
    }
    double key = reverseSprites.get(spriteToRemove);
    LinkedHashSet<Sprite> set = sprites.get(key);
    set.remove(spriteToRemove);
    if (set.size() == 0) {
      sprites.remove(key);
    }
    reverseSprites.remove(spriteToRemove);
  }

  /**
   * reports to the error console when a null sprite is added to the
   * canvas
   *
   * @param  fileName         the file where the error occurred
   * @param  lineNumber       the line where the error occurred
   * @param  methodName       the name of the method where the error
   *                          occurred
   * @param  parameterNumber  the name of the parameter that was null
   */
  private void reportNullSprite(String fileName, int lineNumber,
    String methodName, int parameterNumber) {
    String line = getLine(fileName, lineNumber);
    String fix = "";
    String diagnosis =
      "You are trying to add a null Sprite to the canvas on line " +
      lineNumber + " of the file<br>" + indent(fixedWidth(fileName)) +
      "This line is<br>" + fixedWidth(fixHTML(line)) + "<br>";
    String parameters = "";
    line = line.substring(line.indexOf("addSprite") + 9);
    line = line.substring(line.indexOf('(') + 1);
    String remaining = line;
    int parenthesisLeft = 1;
    int i = 0;
    while (parenthesisLeft > 0) {
      if (remaining.charAt(i) == '(') {
        parenthesisLeft++;
      } else if (remaining.charAt(i) == ')') {
        parenthesisLeft--;
      }
      i++;
    }
    parameters += remaining.substring(0, i - 1);
    int parenthesis = 0;
    int pNumber = 0;
    String nullParameter = null;
    int start = 0;
    for (i = 0; i < parameters.length(); i++) {
      if (parameters.charAt(i) == '(') {
        parenthesis++;
      } else if (parameters.charAt(i) == ')') {
        parenthesis--;
      }
      if ((parenthesis == 0) &&
          ((i == (parameters.length() - 1)) ||
            (parameters.charAt(i) == ','))) {
        if (pNumber == parameterNumber) {
          if (i == (parameters.length() - 1)) {
            i++;
          }
          nullParameter = parameters.substring(start, i);
          break;
        }
        pNumber++;
        start = i + 1;
      }
    }
    if (nullParameter.indexOf('(') < 0) {
      diagnosis += "The variable " + fixedWidth(nullParameter) +
        " is null. " + "The most likely cause is that " +
        fixedWidth(nullParameter) + " has not been initialized.";
      fix += "In the file " + fixedWidth(fileName) +
        " look for a line that starts with<br>" +
        indent(fixedWidth(nullParameter + " = ")) +
        "If you cannot find this line, then this is your problem. " +
        "You need a line that starts with<br>" +
        indent(fixedWidth(nullParameter + " = ")) +
        "Typically, this initialization statement " +
        "should be in the method " + fixedWidth(methodName) + ", " +
        "in the " + fixedWidth("makeSprites") + " method, or in a " +
        "method called from " + fixedWidth("makeSprites") + ", " +
        fixedWidth("startGame") + ", " + "or " +
        fixedWidth("startLevel") + ".  " + "If you have a " +
        fixedWidth("makeSprites") + " method " +
        "be sure to call it from the " + fixedWidth("startGame") +
        " or " + fixedWidth("startLevel") + " method.";
    } else {
      diagnosis += "The method " +
        fixedWidth(fixHTML(nullParameter.trim())) +
        " is returning a null value.";
      fix += "Check the method " +
        fixedWidth(fixHTML(nullParameter.trim())) +
        " to make sure a valid sprite is " +
        "being returned in all cases since null sprites " +
        "cannot be added to the canvas.";
    }
    addError(diagnosis, fix, new NullPointerException());
  }

  /**
   * paints all of the Sprites
   *
   * @param  brush  the Graphics of the component
   */
  @Override
  protected void paintComponent(Graphics brush) {
    Graphics2D copy = (Graphics2D) brush;
    double scalingFactor;
    Point2D.Double clip;
    // set the clip to the maximum size that fits within
    // the aspect ratio
    if ((((double) getSize().width) / getSize().height) > aspect) {
      clip = new Point2D.Double(aspect * getSize().height,
          getSize().height);
    } else {
      clip = new Point2D.Double(getSize().width,
          getSize().width / aspect);
    }
    scalingFactor = Math.min(clip.x, clip.y);
    copy.transform(AffineTransform.getScaleInstance(scalingFactor,
        scalingFactor));
    copy.clip(new Rectangle(0, 0, (int) clip.x, (int) clip.y));
    clearBackground(copy);
    paintSprites(copy);
  }

  /**
   * Class for handling addSprite DURING traversal of sprites. When
   * traversing sprites, a list of these is created to hold added
   * sprites. When traversal ends, list is cleaned up.
   */
  private class DeferredSpriteAddElement {
    public double layer;
    public Sprite sprite;
    public DeferredSpriteAddElement(Sprite sprite, double layer) {
      this.sprite = sprite;
      this.layer = layer;
    }
  }

  /**
   * Class for grabbing focus when the mouse moves over this
   * application. Make the {@link AnimationCanvas} sticky with the
   * focus.
   */
  class FocusGrabber
    extends MouseMotionAdapter {
    @Override
    public void mouseMoved(MouseEvent arg0) {
      if (!isFocusOwner()) {
        requestFocus();
      }
    }
  }
}
