package fang2.attributes;

import fang2.core.Game;

/**
 * This class represents a vector (direction and magnitude) within some
 * space.
 *
 * @author  Robert C. Duvall
 * @author  Brian C. Ladd
 */
public class Vector2D {
  /** This vector's facing in degrees */
  private double degrees;

  /** The magnitude of this vector in some metric space */
  private double magnitude;

  /**
   * Construct a default vector: facing and magnitude 0.
   */
  public Vector2D() {
    this(0, 0);
  }

  /**
   * Construct a new {@code Vector2D} with the given facing and
   * magnitude.
   *
   * @param  degrees    facing in degrees
   * @param  magnitude  magnitude in length units
   */
  public Vector2D(double degrees, double magnitude) {
    if (magnitude >= 0.0) {
      setDegrees(degrees);
      setMagnitude(magnitude);
    } else {
      setDegrees(degrees + 180.0);
      setMagnitude(-magnitude);
    }
  }

  /**
   * Construct a vector representing the given location (translates to
   * angle/magnitude).
   *
   * @param  location  the location to copy
   */
  public Vector2D(Location2D location) {
    this();
    setXYChange(location.getX(), location.getY());
  }

  /**
   * Construct a vector representing the difference between the two
   * given locations
   *
   * @param  source  starting location
   * @param  target  ending location
   */
  public Vector2D(Location2D source, Location2D target) {
    this();
    setXYChange(target.getX() - source.getX(),
      target.getY() - source.getY());
  }

  /**
   * Copy the value of the other vector
   *
   * @param  other  the {@code Vecor2D} to match
   */
  public Vector2D(Vector2D other) {
    this(other.getDegrees(), other.getMagnitude());
  }

  /**
   * Generate a random vector with magnitude between 0.0 and 1.0
   *
   * @return  random vector
   */
  public static final Vector2D randomVector() {
    return randomVector(Game.getCurrentGame());
  }

  /**
   * Generate a random vector with magnitude between min and max
   *
   * @param   minMagnitude  minimum magnitude; >= 0
   * @param   maxMagnitude  maximum magnitude; >= minMagnitude
   *
   * @return  random vector
   */
  public static final Vector2D randomVector(double minMagnitude,
    double maxMagnitude) {
    return randomVector(Game.getCurrentGame(), minMagnitude,
        maxMagnitude);
  }

  /**
   * Use the given game to generate random numbers; return a random
   * vector.
   *
   * @param   game  game to get randomDouble from
   *
   * @return  a random vector with magnitude on 0.0-1.0
   */
  public static final Vector2D randomVector(Game game) {
    return randomVector(game, 0.0, 1.0);
  }

  /**
   * Generate a random vector with magnitude between min and max
   *
   * @param   game          game to get randomDouble from
   * @param   minMagnitude  minimum magnitude; >= 0
   * @param   maxMagnitude  maximum magnitude; >= minMagnitude
   *
   * @return  random vector
   */
  public static final Vector2D randomVector(Game game,
    double minMagnitude, double maxMagnitude) {
    // fix up magnitude if necessary
    if (minMagnitude < 0) {
      minMagnitude = 0.0;
    }
    if (maxMagnitude < minMagnitude) {
      maxMagnitude = minMagnitude;
    }

    return new Vector2D(game.randomDouble(0.0, 360.0),
        game.randomDouble(minMagnitude, maxMagnitude));
  }

  /**
   * Increase the magnitude by a fixed amount
   *
   * @param  change  amount by which to increase magnitude
   */
  public Vector2D accelerate(double change) {
    setMagnitude(getMagnitude() + change);
    return this;
  }

  /**
   * Return a vector equal to the sum of two vectors.
   *
   * @param   v  the vector to add to this
   *
   * @return  this + v
   */
  public Vector2D add(Vector2D v) {
    Vector2D retval = new Vector2D(this);
    retval.addTo(v);
    return retval.simplify();
  }

  /**
   * Add the other vector to this vector
   *
   * @param  other  the {@code Vector2D} to add to this vector
   */
  public Vector2D addTo(Vector2D other) {
    setXYChange(getX() + other.getX(), getY() + other.getY());
    return this;
  }

  /**
   * Apply this vector to the given location
   *
   * @param   current  the starting location where this vector is
   *                   applied
   *
   * @return  the resulting location
   */
  public Location2D apply(Location2D current) {
    return new Location2D(current.getX() + getX(),
        current.getY() + getY());
  }

  /**
   * Subtract the other vector from this vector
   *
   * @param  other  the {@code Vector2D} to subtract from this vector
   */
  public Vector2D difference(Vector2D other) {
    setXYChange(getX() - other.getX(), getY() - other.getY());
    return this;
  }

  /**
   * Check whether this vector is the same as the other.
   *
   * @param   o  the object to check for equality
   *
   * @return  true if they are the same (same type, same direction, same
   *          magnitude)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    } else if (o == null) {
      return false;
    } else if (o.getClass() != getClass()) {
      return false;
    } else {
      Vector2D other = (Vector2D) o;
      return (getDegrees() == other.getDegrees()) &&
        (getMagnitude() == other.getMagnitude());
    }
  }

  /**
   * Return facing in degrees (standardized between -1 and 1 rotations)
   *
   * @return  facing in degrees
   */
  public double getDegrees() {
    // standardize between -360 and +360 (keep 360, -360, and 0 as
    // distinct values)
    final double OFFSET = 0.001;
    double sign = (degrees < 0) ? 1 : -1;
    return ((degrees + (sign * OFFSET)) % 360) - (sign * OFFSET);
  }

  /**
   * Get the magnitude of the vector
   *
   * @return  magnitude of the vector
   */
  public double getMagnitude() {
    return magnitude;
  }

  /**
   * Return a vector normal to the give vector: perpendicular and length
   * 1.0.
   *
   * @return  vector normal to us.
   */
  public Vector2D getNormal() {
    return new Vector2D(getDegrees() + 90, 1.0);
  }

  /**
   * Return facing in radians (between -2PI and 2PI)
   *
   * @return  facing in radians
   */
  public double getRadians() {
    return getDegrees() * Math.PI / 180;
  }

  /**
   * Return facing in revolutions
   *
   * @return  facing in revolutions
   */
  public double getRevolutions() {
    return getDegrees() / 360.0;
  }

  /**
   * Return the deltaX represented by this vector (in standard basis)
   *
   * @return  x-component of vector
   */
  public double getX() {
    return getMagnitude() * Math.cos(-Math.toRadians(getDegrees()));
  }

  /**
   * Return deltaY represented by this vector (in standard basis)
   *
   * @return  y-component of vector
   */
  public double getY() {
    return getMagnitude() * Math.sin(-Math.toRadians(getDegrees()));
  }

  /**
   * Return a vector equal to this one times a scalar.
   *
   * @param   d  the scalar to multiply by
   *
   * @return  this * d
   */
  public Vector2D multiply(double d) {
    return new Vector2D(degrees, magnitude * d).simplify();
  }

  /**
   * Normalize this vector. Make the magnitude exactly 1.0.
   */
  public Vector2D normalize() {
    if (magnitude < 0.0) {
      reverse();
    }
    magnitude = 1.0;
    return this;
  }

  /**
   * Reverse this vector.
   */
  public Vector2D reverse() {
    return new Vector2D(getDegrees() + 180.0, getMagnitude());
  }

  /**
   * Set the facing to the given number of degrees.
   *
   * @param  degrees  number of degrees to set facing to
   */
  public void setDegrees(double degrees) {
    while (degrees < 0) {
      degrees += 360.0;
    }
    while (degrees > 360) {
      degrees -= 360;
    }
    this.degrees = degrees;
  }

  /**
   * Set the magnitude
   *
   * @param  magnitude  the new magnitude value
   */
  public void setMagnitude(double magnitude) {
    this.magnitude = magnitude;
  }

  /**
   * Set the rotation to the given value (in radians)
   *
   * @param  radians  the number of radians to face
   */
  public void setRadians(double radians) {
    setDegrees(radians / Math.PI * 180);
  }

  /**
   * Set the rotation to the given value (in revolutions)
   *
   * @param  revolutions  the number of revolutions
   */
  public void setRevolutions(double revolutions) {
    setDegrees(revolutions * 360);
  }

  /**
   * Set the vector to represent the given coordinates in the standard
   * basis
   *
   * @param   dx  delta x, change in x-coordinate
   * @param   dy  delta y, change in y-coordinate
   *
   * @return  the vector passed in and changed
   */
  public Vector2D setXYChange(double dx, double dy) {
    setDegrees(Math.toDegrees(Math.atan2(-dy, dx)));
    setMagnitude(Math.sqrt((dx * dx) + (dy * dy)));
    return this;
  }

  /**
   * Simplify the vector: if the magnitude is negative, rotate 180
   * degrees and invert the magnitude.
   *
   * @return  the vector passed in
   */
  public Vector2D simplify() {
    if (magnitude < 0.0) {
      reverse();
      magnitude *= -1.0;
    }
    return this;
  }

  /**
   * Return a vector equal to the difference of two vectors.
   *
   * @param   v  the vector to subtract from this
   *
   * @return  this - v
   */
  public Vector2D subtract(Vector2D v) {
    Vector2D retval = new Vector2D(this);
    retval.addTo(v.multiply(-1));
    return retval.simplify();
  }

  /**
   * Make a string representation of the vector.
   *
   * @return  string representation of the value of the vector
   */
  @Override
  public String toString() {
    return getClass().getName() + "[dir=" + getDegrees() + ", speed=" +
      getMagnitude() + "]";
  }

  /**
   * Turn relative to the current facing
   *
   * @param  degrees  amount to turn
   */
  public void turnDegrees(double degrees) {
    setDegrees(getDegrees() + degrees);
  }

  /**
   * Turn relative to current facing.
   *
   * @param  radians  amount to turn
   */
  public void turnRadians(double radians) {
    turnDegrees(radians / Math.PI * 180);
  }

  /**
   * Turn relative to current facing
   *
   * @param  revolutions  amount to turn
   */
  public void turnRevolutions(double revolutions) {
    turnDegrees(revolutions * 360);
  }
}
