package fang2.attributes;

import java.awt.Color;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;

/**
 * This utility class provides easy access to colors by name (either an
 * X Windows named color (see
 * http://en.wikipedia.org/wiki/X11_color_names), FANG Blue (the FANG
 * default color), SCG Red (the red used in the first edition of Simple
 * Computer Games) or a hexadecimal string as Web page colors are
 * specified. The colors themselves are modeled using the standard Java
 * {@link Color} class.
 *
 * @author  Robert C. Duvall
 * @author  Brian C. Ladd
 */
public class Palette {
  /** map canonical (no spaces, lowercase) onto full names */
  private static Map<String, String> canonicalToFullnameMap;

  /** map full names onto colors */
  private static Map<String, Color> colorMap;

  /** map colors back to full names */
  private static Map<Color, String> colorToFullnameMap;

  /** a constant reference to the named color */
  public static final Color FANG_BLUE;

  /**
   * The value returned when a color name is sought for a color which is
   * not in the list of named colors.
   */
  public static String NO_SUCH_COLOR = "Not Found.";

  /** minimum and maximum values for each color channel */
  public static final int COLOR_CHANNEL_MAX = 255;
  public static final int COLOR_CHANNEL_MIN = 0;

  /** opacity setting for fully opaque color */
  public static final int OPAQUE = COLOR_CHANNEL_MAX;

  /**
   * Static initialization block. Creates the table of colors during
   * start-up.
   */
  static {
    Map<String, String> ourColorNames;
    ourColorNames = new HashMap<String, String>();
    ourColorNames.put("Alice Blue", "#F0F8FF");
    ourColorNames.put("Antique White", "#FAEBD7");
    ourColorNames.put("Aqua", "#00FFFF");
    ourColorNames.put("Aquamarine", "#7FFFD4");
    ourColorNames.put("Azure", "#F0FFFF");
    ourColorNames.put("Beige", "#F5F5DC");
    ourColorNames.put("Bisque", "#FFE4C4");
    ourColorNames.put("Black", "#000000");
    ourColorNames.put("Blanched Almond", "#FFEBCD");
    ourColorNames.put("Blue", "#0000FF");
    ourColorNames.put("Blue Violet", "#8A2BE2");
    ourColorNames.put("Brown", "#A52A2A");
    ourColorNames.put("Burlywood", "#DEB887");
    ourColorNames.put("Cadetblue", "#5F9EA0");
    ourColorNames.put("Chartreuse", "#7FFF00");
    ourColorNames.put("Chocolate", "#D2691E");
    ourColorNames.put("Coral", "#FF7F50");
    ourColorNames.put("Cornflower Blue", "#6495ED");
    ourColorNames.put("Cornsilk", "#FFF8DC");
    ourColorNames.put("Crimson", "#DC143C");
    ourColorNames.put("Cyan", "#00FFFF");
    ourColorNames.put("Dark Blue", "#00008B");
    ourColorNames.put("Dark Cyan", "#008B8B");
    ourColorNames.put("Dark Goldenrod", "#B8860B");
    ourColorNames.put("Dark Gray", "#A9A9A9");
    ourColorNames.put("Dark Green", "#006400");
    ourColorNames.put("Dark Khaki", "#BDB76B");
    ourColorNames.put("Dark Magenta", "#8B008B");
    ourColorNames.put("Dark Olive Green", "#556B2F");
    ourColorNames.put("Dark Orange", "#FF8C00");
    ourColorNames.put("Dark Orchid", "#9932CC");
    ourColorNames.put("Dark Red", "#8B0000");
    ourColorNames.put("Dark Salmon", "#E9967A");
    ourColorNames.put("Dark Sea Green", "#8FBC8F");
    ourColorNames.put("Dark Slate Blue", "#483D8B");
    ourColorNames.put("Dark Slate Gray", "#2F4F4F");
    ourColorNames.put("Dark Turquoise", "#00CED1");
    ourColorNames.put("Dark Violet", "#9400D3");
    ourColorNames.put("Deep Pink", "#FF1493");
    ourColorNames.put("Deep Sky Blue", "#00BFFF");
    ourColorNames.put("Dim Gray", "#696969");
    ourColorNames.put("Dim Grey", "#696969");
    ourColorNames.put("Dodger Blue", "#1E90FF");
    ourColorNames.put("Fire Brick", "#B22222");
    ourColorNames.put("Floral White", "#FFFAF0");
    ourColorNames.put("Forest Green", "#228B22");
    ourColorNames.put("Fuchsia", "#FF00FF");
    ourColorNames.put("Gainsboro", "#DCDCDC");
    ourColorNames.put("Ghost White", "#F8F8FF");
    ourColorNames.put("Gold", "#FFD700");
    ourColorNames.put("Goldenrod", "#DAA520");
    ourColorNames.put("Gray", "#808080");
    ourColorNames.put("Grey", "#808080");
    ourColorNames.put("Green", "#008000");
    ourColorNames.put("Green Yellow", "#ADFF2F");
    ourColorNames.put("Honey Dew", "#F0FFF0");
    ourColorNames.put("Hot Pink", "#FF69B4");
    ourColorNames.put("Indian Red", "#CD5C5C");
    ourColorNames.put("Indigo", "#4B0082");
    ourColorNames.put("Ivory", "#FFFFF0");
    ourColorNames.put("Khaki", "#F0E68C");
    ourColorNames.put("Lavender", "#E6E6FA");
    ourColorNames.put("Lavender Blush", "#FFF0F5");
    ourColorNames.put("Lawn Green", "#7CFC00");
    ourColorNames.put("Lemon Chiffon", "#FFFACD");
    ourColorNames.put("Light Blue", "#ADD8E6");
    ourColorNames.put("Light Coral", "#F08080");
    ourColorNames.put("Light Cyan", "#E0FFFF");
    ourColorNames.put("Light Goldenrod Yellow", "#FAFAD2");
    ourColorNames.put("Light Green", "#90EE90");
    ourColorNames.put("Light Grey", "#D3D3D3");
    ourColorNames.put("Light Gray", "#D3D3D3");
    ourColorNames.put("Light Pink", "#FFB6C1");
    ourColorNames.put("Light Salmon", "#FFA07A");
    ourColorNames.put("Light Sea Green", "#20B2AA");
    ourColorNames.put("Light Sky Blue", "#87CEFA");
    ourColorNames.put("Light Slate Gray", "#778899");
    ourColorNames.put("Light Slate Grey", "#778899");
    ourColorNames.put("Light Steel Blue", "#B0C4DE");
    ourColorNames.put("Light Yellow", "#FFFFE0");
    ourColorNames.put("Lime", "#00FF00");
    ourColorNames.put("Lime Green", "#32CD32");
    ourColorNames.put("Linen", "#FAF0E6");
    ourColorNames.put("Magenta", "#FF00FF");
    ourColorNames.put("Maroon", "#800000");
    ourColorNames.put("Medium Aquamarine", "#66CDAA");
    ourColorNames.put("Medium Blue", "#0000CD");
    ourColorNames.put("Medium Orchid", "#BA55D3");
    ourColorNames.put("Medium Purple", "#9370DB");
    ourColorNames.put("Medium Sea Green", "#3CB371");
    ourColorNames.put("Medium Slate Blue", "#7B68EE");
    ourColorNames.put("Medium Spring Green", "#00FA9A");
    ourColorNames.put("Medium Turquoise", "#48D1CC");
    ourColorNames.put("Medium Violet Red", "#C71585");
    ourColorNames.put("Midnight Blue", "#191970");
    ourColorNames.put("Mint Cream", "#F5FFFA");
    ourColorNames.put("Misty Rose", "#FFE4E1");
    ourColorNames.put("Moccasin", "#FFE4B5");
    ourColorNames.put("Navajo White", "#FFDEAD");
    ourColorNames.put("Navy", "#000080");
    ourColorNames.put("Old Lace", "#FDF5E6");
    ourColorNames.put("Olive", "#808000");
    ourColorNames.put("Olive Drab", "#6B8E23");
    ourColorNames.put("Orange", "#FFA500");
    ourColorNames.put("Orange Red", "#FF4500");
    ourColorNames.put("Orchid", "#DA70D6");
    ourColorNames.put("Pale Goldenrod", "#EEE8AA");
    ourColorNames.put("Pale Green", "#98FB98");
    ourColorNames.put("Pale Turquoise", "#AFEEEE");
    ourColorNames.put("Pale Violet Red", "#DB7093");
    ourColorNames.put("Papaya Whip", "#FFEFD5");
    ourColorNames.put("Peachpuff", "#FFDAB9");
    ourColorNames.put("Peru", "#CD853F");
    ourColorNames.put("Pink", "#FFC0CB");
    ourColorNames.put("Plum", "#DDA0DD");
    ourColorNames.put("Powder Blue", "#B0E0E6");
    ourColorNames.put("Purple", "#800080");
    ourColorNames.put("Red", "#FF0000");
    ourColorNames.put("Rosy Brown", "#BC8F8F");
    ourColorNames.put("Royal Blue", "#4169E1");
    ourColorNames.put("Saddle Brown", "#8B4513");
    ourColorNames.put("Salmon", "#FA8072");
    ourColorNames.put("Sandy Brown", "#F4A460");
    ourColorNames.put("Sea Green", "#2E8B57");
    ourColorNames.put("Seashell", "#FFF5EE");
    ourColorNames.put("Sienna", "#A0522D");
    ourColorNames.put("Silver", "#C0C0C0");
    ourColorNames.put("Sky Blue", "#87CEEB");
    ourColorNames.put("Slate Blue", "#6A5ACD");
    ourColorNames.put("Slate Gray", "#708090");
    ourColorNames.put("Slate Grey", "#708090");
    ourColorNames.put("Snow", "#FFFAFA");
    ourColorNames.put("Spring Green", "#00FF7F");
    ourColorNames.put("Steel Blue", "#4682B4");
    ourColorNames.put("Tan", "#D2B48C");
    ourColorNames.put("Teal", "#008080");
    ourColorNames.put("Thistle", "#D8BFD8");
    ourColorNames.put("Tomato", "#FF6347");
    ourColorNames.put("Turquoise", "#40E0D0");
    ourColorNames.put("Violet", "#EE82EE");
    ourColorNames.put("Wheat", "#F5DEB3");
    ourColorNames.put("White", "#FFFFFF");
    ourColorNames.put("White Smoke", "#F5F5F5");
    ourColorNames.put("Yellow", "#FFFF00");
    ourColorNames.put("Yellow Green", "#9ACD32");
    ourColorNames.put("FANG Blue", "#6464FF");
    ourColorNames.put("SCG Red", "#A62126");

    colorMap = new TreeMap<String, Color>();
    canonicalToFullnameMap = new HashMap<String, String>();
    colorToFullnameMap = new HashMap<Color, String>();

    for (Map.Entry<String, String> entry : ourColorNames.entrySet()) {
      String fullName = entry.getKey();
      String hexRGB = entry.getValue();
      String canonicalName = getCanonicalName(fullName);
      Color matchingColor = getColorFromRGB(hexRGB);

      colorMap.put(fullName, matchingColor);
      colorToFullnameMap.put(matchingColor, fullName);
      canonicalToFullnameMap.put(canonicalName, fullName);
    }
    FANG_BLUE = colorMap.get("FANG Blue");
  }

  /**
   * Add a color to the list of colors this Palette knows. Lets you
   * define and use your own named colors. Color is specified as a Java
   * {@link java.awt.Color Color} object.
   *
   * @param  name   the name the color will have in the color map
   * @param  color  the color to be added
   */
  public static void addColor(String name, Color color) {
    Color cannonicalColor = getColor(color, OPAQUE);
    String cannonicalName = getCanonicalName(name);
    colorMap.put(name, cannonicalColor);
    colorToFullnameMap.put(cannonicalColor, name);
    canonicalToFullnameMap.put(cannonicalName, name);
  }

  /**
   * Add a color to the list of colors {@link Palette} knows. Lets you
   * define named colors which you can later use. Color is specified as
   * a Web string "#rrggbb" where rr, gg, and bb are hexidecimal numbers
   * specifying color channel values for red, green, and blue.
   *
   * @param  name  the new color's name
   * @param  rgb   the new color's rgb value ("#rrggbb")
   */
  public static void addColor(String name, String rgb) {
    addColor(name, getColorFromRGB(rgb, OPAQUE));
  }

  /**
   * Create a semi-transparent color from the given color. That is, the
   * R, G, and B channels remain unchanged while the transparency is set
   * to the given value.
   *
   * @param   shade    the color to match (in RGB)
   * @param   opacity  the transparency for the new color (0-255)
   *
   * @return  color matching given color and given opacity
   */
  public static Color getColor(Color shade, int opacity) {
    return getColor(shade.getRed(), shade.getGreen(), shade.getBlue(),
        opacity);
  }

  /**
   * Create a solid color with the given red, green, and blue values.
   *
   * @param   red    red channel value [0..255]
   * @param   green  green channel value [0..255]
   * @param   blue   blue channel value [0..255]
   *
   * @return  color created from the given values
   */
  public static Color getColor(int red, int green, int blue) {
    return new Color(red, green, blue, OPAQUE);
  }

  /**
   * Create a semi-transparent color with the given red, green, and blue
   * values. All parameter values are clamped to the required range.
   *
   * @param   red      red channel value [0..255]
   * @param   green    green channel value [0..255]
   * @param   blue     blue channel value [0..255]
   * @param   opacity  alpha channel value (higher = more opaque)
   *                   [0..255]
   *
   * @return  color created from the given values
   */
  public static Color getColor(int red, int green, int blue,
    int opacity) {
    red = Math.min(COLOR_CHANNEL_MAX, Math.max(COLOR_CHANNEL_MIN, red));
    green = Math.min(COLOR_CHANNEL_MAX,
        Math.max(COLOR_CHANNEL_MIN, green));
    blue = Math.min(COLOR_CHANNEL_MAX,
        Math.max(COLOR_CHANNEL_MIN, blue));
    opacity = Math.min(COLOR_CHANNEL_MAX,
        Math.max(COLOR_CHANNEL_MIN, opacity));
    return new Color(red, green, blue, opacity);
  }

  /**
   * Create an opaque color from the given name. Names are the entire
   * set of X Windows named colors (see
   * http://en.wikipedia.org/wiki/X11_color_names for more information
   * on color names) plus two additional colors, "FANG Blue", the
   * default color used for sprites in FANG and "SCG Red", the red color
   * used in the first edition of <i>Simple Computer Games</i>.
   *
   * @param   name  color name
   *
   * @return  matching color if one can be found; FANG Blue otherwise.
   */
  public static Color getColor(String name) {
    return getColor(name, OPAQUE);
  }

  /**
   * Create a semi-transparent color with from the given name. Names are
   * X Windows named colors (see
   * http://en.wikipedia.org/wiki/X11_color_names for more information
   * on color names) plus two additional colors, "FANG Blue", the
   * default color used for sprites in FANG and "SCG Red", the red color
   * used in the first edition of <i>Simple Computer Games</i>.
   *
   * @param   name     color name
   * @param   opacity  how opaque the color should be, (higer = more
   *                   opaque) [0..255]; clamped to the required range
   *
   * @return  the color returned by {@link #getColor(String)} (match or
   *          FANG Blue) with opacity set to the given value.
   */
  public static Color getColor(String name, int opacity) {
    Color color = null;
    String canonical = getCanonicalName(name);
    if (canonicalToFullnameMap.containsKey(canonical)) {
      String colorName = canonicalToFullnameMap.get(canonical);
      color = colorMap.get(colorName);
    }

    if (color == null) {
      color = FANG_BLUE;
    }
    return getColor(color, opacity);
  }

  /**
   * Convert a "#rrggbb" string into a Java color object corresponding
   * to the given color.
   *
   * @param   rgb  string of the form "#rrggbb" where rr, gg, and bb are
   *               the color channel values for red, green, and blue
   *               expressed in hex
   *
   * @return  an opaque Color corresponding to the given rgb values.
   */
  public static Color getColorFromRGB(String rgb) {
    return getColorFromRGB(rgb, OPAQUE);
  }

  /**
   * Convert a "#rrggbb" string into a Java color object corresponding
   * to the given color.
   *
   * @param   rgb      string of the form "#rrggbb" where rr, gg, and bb
   *                   are the color channel values for red, green, and
   *                   blue expressed in hex
   * @param   opacity  the opacity of the color [0..255] where 255 is
   *                   opaque and 0 is transparent
   *
   * @return  a Color corresponding to the given rgb and transparency
   *          values
   */
  public static Color getColorFromRGB(String rgb, int opacity) {
    int red = Integer.parseInt(rgb.substring(1, 3), 16);
    int green = Integer.parseInt(rgb.substring(3, 5), 16);
    int blue = Integer.parseInt(rgb.substring(5, 7), 16);

    return getColor(red, green, blue, opacity);
  }

  /**
   * Convert a "#rrggbbaa" string into a Java color object corresponding
   * to the given color.
   *
   * @param   rgba  string of the form "#rrggbbaa" where rr, gg, and bb
   *                are the color channel values for red, green, and
   *                blue expressed in hex and aa is the alpha channel
   *                (or opacity) expressed in hex
   *
   * @return  a Color corresponding to the given rgba values.
   */
  public static Color getColorFromRGBA(String rgba) {
    int transparency = Integer.parseInt(rgba.substring(7, 9), 16);
    return getColorFromRGB(rgba.substring(0, 7), transparency);
  }

  /**
   * Get the whole name/color map.
   *
   * @return  the name/color map used by this Palette.
   */
  public static Map<String, Color> getColorMap() {
    return new TreeMap<String, Color>(colorMap);
  }

  /**
   * Get the name of the color expressed as a Java Color object.
   *
   * @param   color  the color to name (if possible)
   *
   * @return  returns the name for the color (if there is one) and
   *          returns NO_SUCH_COLOR otherwise.
   */
  public static String getColorName(Color color) {
    String name = colorToFullnameMap.get(color);

    if (name == null) {
      name = "[" + color.getRed() + ", " + color.getGreen() + ", " +
        color.getBlue() + "]";
    }

    if (color.getAlpha() != OPAQUE) {
      name += ":" + color.getAlpha();
    }

    return name;
  }

  /**
   * Get the name of the given color (specified by red, green, and blue
   * channels.
   *
   * @param   r  red value [0..255]
   * @param   g  green value [0..255]
   * @param   b  blue value [0..255]
   *
   * @return  returns the name for the color (if there is one) and
   *          returns NO_SUCH_COLOR otherwise.
   */
  public static String getColorName(int r, int g, int b) {
    return getColorName(new Color(r, g, b));
  }

  /**
   * Get the name of given color (specified as a "#rrggbb" hex string
   * (note that it can safely be a "#rrggbbaa" string, too.
   *
   * @param   rgb  string of the form "#rrggbb" where rr, gg, and bb are
   *               the color channel values for red, green, and blue
   *               expressed in hex
   *
   * @return  returns the name for the color (if there is one) and
   *          returns NO_SUCH_COLOR otherwise.
   */
  public static String getColorName(String rgb) {
    return getColorName(getColorFromRGB(rgb));
  }

  /**
   * Create a random color. this method is not yet multiplayer tested in
   * shared jvm environments
   */
  /**
   * Generate a random color. Uses the parameter random number generator
   * if one is provided. If none is provided, returns null as the random
   * color.
   *
   * @param   random  - a Random object used to generate random numbers;
   *                  this class is ignorant of other FANG classes so it
   *                  does not poke around in the current game to find
   *                  the Random that is there. Callers provide the
   *                  Random (Game.getRandomColor() does this under the
   *                  hood).
   *
   * @return  a randomly selected color if the parameter is non-null;
   *          null otherwise
   */
  public static Color getRandomColor(Random random) {
    if (random != null) {
      return getColor(random.nextInt(COLOR_CHANNEL_MAX + 1),
          random.nextInt(COLOR_CHANNEL_MAX + 1),
          random.nextInt(COLOR_CHANNEL_MAX + 1));
    }

    return null;
  }

  /**
   * Return the Web "hex" string for the color of the form "#rrggbb".
   *
   * @param   color  Java color to get the string for
   *
   * @return  String of the form "#rrggbb" where rr, gg, and bb are the
   *          hex values for the red, green, and blue channels in color
   */
  public static String getRGB(Color color) {
    return "#" + Integer.toHexString(color.getRed()) +
      Integer.toHexString(color.getGreen()) +
      Integer.toHexString(color.getBlue());
  }

  /**
   * Return the Web "hex" string for the color of the form "#rrggbbaa".
   *
   * @param   color  Java color to get the string for
   *
   * @return  String of the form "#rrggbbaa" where rr, gg, and bb are
   *          the hex values for the red, green, and blue channels in
   *          color and aa is the alpha (opacity) channel
   */
  public static String getRGBA(Color color) {
    return "#" + Integer.toHexString(color.getRed()) +
      Integer.toHexString(color.getGreen()) +
      Integer.toHexString(color.getBlue()) +
      Integer.toHexString(color.getAlpha());
  }

  /**
   * Convert a name into canonical form: all lower case with all spaces
   * compressed out.
   *
   * @param   name  name to put in canonical form
   *
   * @return  canonical form of name
   */
  private static String getCanonicalName(String name) {
    return name.toLowerCase().replaceAll("\\s", "");
  }
}
