package fang2.attributes;

import fang2.core.Game;
import fang2.core.Sprite;

import java.awt.geom.Point2D;

/**
 * This class represents an (x, y) coordinate within some space. Both
 * coordinates are {@code double} values so they can hold fractional
 * parts. In FANG, {@code Location2D} is used to represent points on the
 * screen (where coordinates go from 0.0 to 1.0). They can also be used
 * to hold velocities and other two-dimensional vectors.
 *
 * @author  Robert C. Duvall
 * @author  Jam Jenkins
 * @author  Brian C. Ladd
 */
public class Location2D
  extends Point2D.Double {
  /**
   * Creates a new {@code Location2D} instance at (0.0, 0.0).
   */
  public Location2D() {
    this(0, 0);
  }

  /**
   * Creates a new {@code Location2D} instance at (x, y).
   *
   * @param  x  the x-coordinate of the location
   * @param  y  the y-coordinate of the location
   */
  public Location2D(double x, double y) {
    super(x, y);
  }

  /**
   * Creates a new {@code Location2D} instance that is a copy of the
   * given location.
   *
   * @param  original  the original location this is to copy
   */
  public Location2D(Location2D original) {
    this(original.getX(), original.getY());
  }

  /**
   * Creates a new {@code Location2D} instance with the same values as
   * the given point.
   *
   * @param  point  the point this location is to copy
   */
  public Location2D(Point2D point) {
    this(point.getX(), point.getY());
  }

  /**
   * Construct a {@link Location2D} which is equivalent to the given
   * vector.
   *
   * @param  v {@link Vector2D} to match
   */
  public Location2D(Vector2D v) {
    this(v.getX(), v.getY());
  }

  /**
   * Determine the magnitude of originalVector in the direction of
   * directionVector.
   *
   * @param   originalVector   the vector (in x, y basis) to be
   *                           projected into the directionVector
   * @param   directionVector  the vector onto which the original vector
   *                           should be projected
   *
   * @return  projected length of the originalVector in the
   *          directionVector direction
   */
  public static double componentMagnitude(Location2D originalVector,
    Location2D directionVector) {
    double alpha = directionVector.getRadians();
    double theta = originalVector.getRadians();
    double magnitude = originalVector.getMagintude();
    return magnitude * Math.cos(theta - alpha);
  }

  /**
   * Return the projection of originalVector onto directionVector; this
   * is a vector.
   *
   * @param   originalVector   the vector (in x, y basis) to be
   *                           projected onto the directionVector
   * @param   directionVector  the vector onto which the originalVector
   *                           should be projected
   *
   * @return  projected image of originalVector onto directionVector
   */
  public static Location2D componentVector(Location2D originalVector,
    Location2D directionVector) {
    Location2D v = directionVector.normalize();
    double magnitude = componentMagnitude(originalVector,
        directionVector);
    return new Location2D(v.getX() * magnitude, v.getY() * magnitude);
  }

  /**
   * Generate a random location somewhere on the screen (0,0) to (1, 1).
   *
   * @return  a random location on screen
   */
  public static final Location2D randomLocation() {
    return randomLocation(Game.getCurrentGame());
  }

  /**
   * Generate a random location inside the given box
   *
   * @param   box  box bounding random creation locations
   *
   * @return  a random location inside the box
   */
  public static final Location2D randomLocation(Box2D box) {
    return randomLocation(Game.getCurrentGame(), box.getMinX(),
        box.getMinY(), box.getMaxX(), box.getMaxY());
  }

  /**
   * Generate a random location inside the given rectangle. The
   * rectangle is specified by opposite corners
   *
   * @param   minX  smallest x-coordinate
   * @param   minY  smallest y-coordinate
   * @param   maxX  largest x-coordinate
   * @param   maxY  largest y-coordinate
   *
   * @return  a random location inside the rectangle
   */
  public static final Location2D randomLocation(double minX,
    double minY, double maxX, double maxY) {
    return randomLocation(Game.getCurrentGame(), minX, minY, maxX,
        maxY);
  }

  /**
   * Generate a random location somewhere on the screen using the given
   * game to get random numbers
   *
   * @param   game  game to use for random numbers
   *
   * @return  a random location on the screen
   */
  public static final Location2D randomLocation(Game game) {
    return randomLocation(game, 0.0, 0.0, 1.0, 1.0);
  }

  /**
   * Generate a random location inside the given box
   *
   * @param   box   box bounding random creation locations
   * @param   game  game to use for random numbers
   *
   * @return  a random location inside the box
   */
  public static final Location2D randomLocation(Game game, Box2D box) {
    return randomLocation(game, box.getMinX(), box.getMinY(),
        box.getMaxX(), box.getMaxY());
  }

  /**
   * Generate a random location inside the given rectangle. The
   * rectangle is specified by opposite corners
   *
   * @param   game  game to use for random numbers
   * @param   minX  smallest x-coordinate
   * @param   minY  smallest y-coordinate
   * @param   maxX  largest x-coordinate
   * @param   maxY  largest y-coordinate
   *
   * @return  a random location inside the rectangle
   */
  public static final Location2D randomLocation(Game game, double minX,
    double minY, double maxX, double maxY) {
    return new Location2D(game.randomDouble(minX, maxX),
        game.randomDouble(minY, maxY));
  }

  /**
   * Add two {@link Location2D} as vectors.
   *
   * @param   rhs  the vector to add to this one
   *
   * @return  a {@link Location2D} with the value of the sum
   */
  public Location2D add(Location2D rhs) {
    return new Location2D(x + rhs.x, y + rhs.y);
  }

  /**
   * Add a {@link Vector2D} to this point, returning resulting point.
   *
   * @param   v {@link Vector2D} to add
   *
   * @return  the point resulting from adding the vector to the point.
   */
  public Location2D add(Vector2D v) {
    return new Location2D(x + v.getX(), y + v.getY());
  }

  /**
   * Calculate the additive inverse of this vector
   *
   * @return  -this (by component)
   */
  public Location2D additiveInverse() {
    return new Location2D(-getX(), -getY());
  }

  /**
   * Return the Euclidian distance from this location to the location
   * (x, y).
   *
   * @param  x  x-coordinate of the location to measure distance to
   * @param  y  y-coordinate of the location to measure distance to
   *            return distance (in screens) between this location and
   *            the location {@code (x, y)}
   */
  @Override
  public double distance(double x, double y) {
    return super.distance(x, y);
  }

  /**
   * Return the Euclidian distance from this location to the given
   * location.
   *
   * @param   other  location to measure the distance to
   *
   * @return  distance (in screens) between this location and {@code
   *          other}
   */
  public double distance(Location2D other) {
    return super.distance(other);
  }

  /**
   * Return the Euclidian distance from this location to the location of
   * the sprite. This is the distance from this location to the center
   * of the sprite.
   *
   * @param   sprite  the {@link fang2.core.Sprite Sprite} from which the
   *                  distance of this location is to be measured
   *
   * @return  distance (in screens) between this location and the
   *          location of the sprite
   */
  public double distance(Sprite sprite) {
    return super.distance(sprite.getLocation());
  }

  /**
   * Calculate the dot-product of this vector with the given other
   * vector. The dot-product is the sum of the product of the components
   * (x1 * x2 + y1 * y2 for vector1 and vector2).
   *
   * @param   other  Location2D as vector representing value for which
   *                 we want the dot product
   *
   * @return  dot-product of this and other
   */
  public double dotProduct(Location2D other) {
    return (getX() * other.getX()) + (getY() * other.getY());
  }

  /**
   * Calculate the rotation of this vector around the origin.
   *
   * @return  the rotation, in radians, represented by the given vector
   */
  public double getRadians() {
    return Math.atan2(getY(), getX());
  }

  /**
   * Calculate the rotation of this vector around the origin.
   *
   * @return  the rotation, in revolutions, represented by the given
   *          vector
   */
  public double getRevolutions() {
    return getRadians() / (2 * Math.PI);
  }

  /**
   * Calculate the rotation of this vector around the origin.
   *
   * @return  the rotation, in degrees, represented by the given vector
   */
  public double getRotationDegrees() {
    return 360.0 * (getRadians() / (2 * Math.PI));
  }

  /**
   * Get the x-coordinate of this location.
   *
   * @return  location's x-coordinate
   */
  @Override
  public double getX() {
    return super.getX();
  }

  /**
   * Get the y-coordinate of this location
   *
   * @return  location's y-coordinate
   */
  @Override
  public double getY() {
    return super.getY();
  }

  /**
   * Determine whether this location and the given box coincide.
   *
   * @param   box  the {@link fang2.attributes.Box2D Box2D} to check this
   *               location against
   *
   * @return  {@code true} if the location is within the {@link
   *          fang2.attributes.Box2D Box2D}, {@code false} otherwise
   */
  public boolean intersects(Box2D box) {
    return box.contains(this);
  }

  /**
   * Determine whether this location and the sprite coincide.
   *
   * @param   sprite  the {@link fang2.core.Sprite Sprite} to check this
   *                  location against
   *
   * @return  {@code true} if the location is within the {@link
   *          fang2.core.Sprite Sprite}, {@code false} otherwise
   */
  public boolean intersects(Sprite sprite) {
    return sprite.intersects(this);
  }

  /**
   * Calculate the magnitude (or length) of this vector.
   *
   * @return  magnitude of this vector (the Pythagorean formula!)
   */
  public double getMagintude() {
    double dotProduct = dotProduct(this);
    // dot-product with self = x^2 + y^2
    return Math.sqrt(dotProduct);
  }

  /**
   * Multiply a location (as a vector) by a scalar.
   *
   * @param   scalar  scalar to multiply by
   *
   * @return  Location scaled by the given scalar value.
   */
  public Location2D multiply(double scalar) {
    return new Location2D(x * scalar, y * scalar);
  }

  /**
   * Return a normalized vector in the same direction as this vector.
   * That is, a vector with the same direction but a magnitude of 1.
   * (Divide each component by the magnitude of the vector).
   *
   * @return  normalized vector in direction of this vector
   */
  public Location2D normalize() {
    double len = getMagintude();
    return new Location2D(x / len, y / len);
  }

  /**
   * Set the x-coordinate of this location. The y-coordinate remains
   * unchanged.
   *
   * @param  x  the new x-coordinate for this location
   */
  public void setX(double x) {
    super.setLocation(x, getY());
  }

  /**
   * Set the y-coordinate of this location. The x-coordinate remains
   * unchanged.
   *
   * @param  y  the new y-coordinate for this location
   */
  public void setY(double y) {
    super.setLocation(getX(), y);
  }

  /**
   * Calculate the difference of this vector and the other vector.
   *
   * @param   other  the vector to subtract from this vector
   *
   * @return  this - other (by component)
   */
  public Location2D subtract(Location2D other) {
    return new Location2D(getX() - other.getX(), getY() - other.getY());
  }

  /**
   * Return a string representing the value of the {@code Location2D}.
   */
  @Override
  public String toString() {
    return getClass().getName() + "[x=" + getX() + ",y=" + getY() + "]";
  }
}
