package fang2.attributes;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import fang2.core.Sprite;
import fang2.sprites.RectangleSprite;

/**
 * Box2D is, effectively, a {@link java.awt.geom.Rectangle2D.Double
 * Rectangle2D} for use in fang2. It can convert itself to a Rectangle2D
 * when necessary.
 *
 * @author  The FANG Team
 */
public class Box2D {
  /** the height of the box. */
  private double height;

  /** the width of the box. */
  private double width;

  /** x-coordinate of the <b>center</b> of the box. */
  private double x;

  /** y-coordinate of the <b>center</b> of the box. */
  private double y;

  /**
   * Default constructor: Create a box at the origin with zero width and
   * height.
   */
  public Box2D() {
    this(0.0, 0.0, 0.0, 0.0);
  }

  /**
   * Construct a box with the same corners as the other box.
   *
   * @param  other  the box to match
   */
  public Box2D(Box2D other) {
    this(other.getMinX(), other.getMinY(), other.getWidth(),
      other.getHeight());
  }

  /**
   * The canonical constructor for a Box2D. Takes the coordinates of the
   * upper-left corner and width and height of the box and initializes
   * all of the fields. This is the <b>only</b> routine for {@code
   * Box2D} which works with the upper-left corner coordinate.
   *
   * @param  left    x-coordinate of the upper-left corner
   * @param  top     y-coordinate of the upper-left corner
   * @param  width   width of the box
   * @param  height  height of the box
   */
  public Box2D(double left, double top, double width, double height) {
    setX(left + (width / 2));
    setY(top + (height / 2));
    setWidth(width);
    setHeight(height);
  }

  /**
   * Construct a box with the given <b>center</b> and size.
   *
   * @param  center  location of the <b>center</b> of the box
   * @param  size    the size of the box
   */
  public Box2D(Location2D center, Size2D size) {
    setX(center.getX());
    setY(center.getY());
    setWidth(size.getWidth());
    setHeight(size.getHeight());
  }

  /**
   * Construct a {@code Box2D} with the same corners as the given {@link
   * java.awt.geom.Rectangle2D Rectangle2D}
   *
   * @param  rectangleToMatch  the {@link java.awt.geom.Rectangle2D
   *                           Rectangle2D} to match
   */
  public Box2D(Rectangle2D rectangleToMatch) {
    this(rectangleToMatch.getX(), rectangleToMatch.getY(),
      rectangleToMatch.getWidth(), rectangleToMatch.getHeight());
  }

  /**
   * Test whether this {@code Box2D} contains the given point or not.
   *
   * @param   point  the point to check
   *
   * @return  {@code true} if point is on the boundary or in the
   *          interior of the {@code Box2D}; {@code false} otherwise
   */
  public boolean contains(Point2D point) {
    return ((getMinX() <= point.getX()) &&
        (point.getX() <= getMaxX()) &&
        ((getMinY() <= point.getY()) && (point.getY() <= getMaxY())));
  }

  /**
   * Returns the largest y-coordinate of the box in {@code double}
   * precision
   *
   * @return  largest y-coordinate
   */
  public double getBottom() {
    return getMaxY();
  }

  /**
   * Get the {@link Location2D} of the center of the box.
   *
   * @return  the center of the box.
   */
  public Location2D getCenter() {
    return new Location2D(getX(), getY());
  }

  /**
   * Return the height of this box in {@code double} precision
   *
   * @return  the height of the box
   */
  public double getHeight() {
    return height;
  }

  /**
   * Returns the smallest x-coordinate of the box in {@code double}
   * precision
   *
   * @return  smallest x-coordinate
   */
  public double getLeft() {
    return getMinX();
  }

  /**
   * Returns the largest x-coordinate of the box in {@code double}
   * precision
   *
   * @return  largest x-coordinate
   */
  public double getMaxX() {
    return getX() + (getWidth() / 2);
  }

  /**
   * Returns the largest y-coordinate of the box in {@code double}
   * precision
   *
   * @return  largest y-coordinate
   */
  public double getMaxY() {
    return getY() + (getHeight() / 2);
  }

  /**
   * Returns the smallest x-coordinate of the box in {@code double}
   * precision
   *
   * @return  smallest x-coordinate
   */
  public double getMinX() {
    return getX() - (getWidth() / 2);
  }

  /**
   * Returns the smallest y-coordinate of the box in {@code double}
   * precision
   *
   * @return  smallest y-coordinate
   */
  public double getMinY() {
    return getY() - (getHeight() / 2);
  }

  /**
   * Returns the largest x-coordinate of the box in {@code double}
   * precision
   *
   * @return  largest x-coordinate
   */
  public double getRight() {
    return getMaxX();
  }

  /**
   * Returns the size of the box in a {@link Size2D}
   *
   * @return  the size of the box
   */
  public Size2D getSize() {
    return new Size2D(getWidth(), getHeight());
  }

  /**
   * Returns the smallest y-coordinate of the box in {@code double}
   * precision
   *
   * @return  smallest y-coordinate
   */
  public double getTop() {
    return getMinY();
  }

  /**
   * Return the width of this box in {@code double} precision
   *
   * @return  the width of the box
   */
  public double getWidth() {
    return width;
  }

  /**
   * Return the x-coordinate of the <b>center</b> of the box in {@code
   * double} precision.
   *
   * @return  the x-coordinate of the center of the box
   */
  public double getX() {
    return x;
  }

  /**
   * Return the y-coordinate of the <b>center</b> of the box in {@code
   * double} precision.
   *
   * @return  the y-coordinate of the center of the box
   */
  public double getY() {
    return y;
  }

  /**
   * Test whether this {@code Box2D} contains the given point or not.
   *
   * @param   location  the location to check
   *
   * @return  {@code true} if point is on the boundary or in the
   *          interior of the {@code Box2D}; {@code false} otherwise
   */
  public boolean intersects(Location2D location) {
    return contains(location);
  }

  /**
   * Test whether this {@code Box2D} intersects with the sprite.
   *
   * @param   sprite  the {@link fang2.core.Sprite Sprite} to check for
   *                  intersection
   *
   * @return  {@code true} if the sprite intersects the edges or
   *          interior of this {@code Box2D}; {@code false} otherwise
   */
  public boolean intersects(Sprite sprite) {
    Sprite box = new RectangleSprite(getWidth(), getHeight());
    box.setLocation(getX(), getY());
    return sprite.intersects(box);
  }

  /**
   * Determines where the specified point lies with respect to this
   * {@code Box2D}. This method computes a binary OR of the appropriate
   * mask values indicating, for each side of this {@code Box2D},
   * whether or not the specified point is on the same side of the edge
   * as the rest of this {@code Box2D}. [ {@link
   * java.awt.geom.Rectangle2D Rectangle2D} {@code outcode} comment]
   *
   * @param   x  x-coordinate of point to check
   * @param   y  y-coordinate of point to check
   *
   * @return  the logical OR of the appropriate values
   *
   * @see     java.awt.geom.Rectangle2D#OUT_BOTTOM OUT_BOTTOM
   * @see     java.awt.geom.Rectangle2D#OUT_LEFT OUT_LEFT
   * @see     java.awt.geom.Rectangle2D#OUT_RIGHT OUT_RIGHT
   * @see     java.awt.geom.Rectangle2D#OUT_TOP OUT_TOP
   */
  public int outcode(double x, double y) {
    return Rectangle2D().outcode(x, y);
  }

  /**
   * Determines where the specified {@link java.awt.geom.Point2D
   * Point2D} lies with respect to this {@code Box2D}. This method
   * computes a binary OR of the appropriate mask values indicating, for
   * each side of this {@code Box2D}, whether or not the specified
   * {@link java.awt.geom.Point2D Point2D} is on the same side of the
   * edge as the rest of this {@code Box2D}. [ {@link
   * java.awt.geom.Rectangle2D Rectangle2D} {@code outcode} comment]
   *
   * @param   p  the point to check
   *
   * @return  the logical OR of the appropriate values
   *
   * @see     java.awt.geom.Rectangle2D#OUT_BOTTOM OUT_BOTTOM
   * @see     java.awt.geom.Rectangle2D#OUT_LEFT OUT_LEFT
   * @see     java.awt.geom.Rectangle2D#OUT_RIGHT OUT_RIGHT
   * @see     java.awt.geom.Rectangle2D#OUT_TOP OUT_TOP
   */
  public int outcode(Point2D p) {
    return Rectangle2D().outcode(p);
  }

  /**
   * Return a {@link java.awt.geom.Rectangle2D Rectangle2D} with corners
   * matching this {@code Box2D}.
   *
   * @return  a {@link java.awt.geom.Rectangle2D Rectangle2D} matching
   *          this box
   */
  public Rectangle2D Rectangle2D() {
    return new Rectangle2D.Double(getMinX(), getMinY(), getWidth(),
        getHeight());
  }

  /**
   * Set the height of the box.
   *
   * @param  height  the new height of the box
   */
  public void setHeight(double height) {
    this.height = height;
  }

  /**
   * Set the width of the box.
   *
   * @param  width  the new width of the box
   */
  public void setWidth(double width) {
    this.width = width;
  }

  /**
   * Set the x-coordinate of the <b>center</b> of the box.
   *
   * @param  x  the new x-coordinate of the box's center.
   */
  public void setX(double x) {
    this.x = x;
  }

  /**
   * Set the y-coordinate of the <b>center</b> of the box.
   *
   * @param  y  the new y-coordinate of the box's center.
   */
  public void setY(double y) {
    this.y = y;
  }

  /**
   * Return a string representation of the {@code Box2D}. This describes
   * the x and y location and the width and height.
   */
  @Override
  public String toString() {
    return getClass().getName() + "[x = " + getX() + ", y = " + getY() +
      "], " + "[w = " + getWidth() + ", h = " + getHeight() + "]";
  }
}
